﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections;
using System.Linq;

using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Convert animation tables into an array of their key frame count.
    /// </summary>
    public class AnimationTableCountArrayConverter : ConverterBase
    {
        /// <summary>Constant the parameter name.</summary>
        private const string OutputIndicesParamName = "OutputIndices";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static AnimationTableCountArrayConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                OutputIndicesParamName,
                "The output indices of the input animation tables.",
                ValidateInputAnimTableOutputIndicesParameter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public AnimationTableCountArrayConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the minimum number of input values.
        /// </summary>
        public override int MinInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 4; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.AnimationTableCountArrayConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.AnimationTableCountArrayConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index >= 0 && index < 4)
            {
                return Properties.Resources.AnimationTableCountArrayConverterInputValueDesc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;

            // Get the output index of the input animation tables from parameter.
            int[] outputIndices = new int[] { 0, 1, 2, 3 };
            if (this.GetParameter(OutputIndicesParamName, out param) == true &&
                param is int[] &&
                ((int[])param).Length == 4)
            {
                outputIndices = (int[])param;
            }

            var animationTableCountArray = new int[4];
            for (int i = 0; i < 4; ++i)
            {
                if (this.IsInputValueAssigned(i) == false)
                {
                    animationTableCountArray[outputIndices[i]] = 0;
                }
                else
                {
                    var table = this.GetInputValue(i) as AnimationTableData;
                    if (table == null)
                    {
                        Logger.Log(LogLevels.Warning, "AnimationTableCountArrayConverter.Convert : The assigned input value {0}, is not an animation table.", i);
                        animationTableCountArray[i] = 0;
                        continue;
                    }

                    animationTableCountArray[outputIndices[i]] = table.Count;
                }
            }

            this.SetOutputValue(animationTableCountArray);

            return true;
        }

        /// <summary>
        /// Validate parameter that assigns the output index of input animation table.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateInputAnimTableOutputIndicesParameter(object value)
        {
            int[] indices = null;
            if (value is string)
            {
                // The format should be like "new int[] { 0, 1, 2, 3 }".
                string[] extractedTokens = RegexUtility.ExtractStrings(
                    "new int[ ]*\\[\\][ ]*\\{([ ]*[0-9]+[ ]*,[ ]*[0-9]+[ ]*,[ ]*[0-9]+[ ]*,[ ]*[0-9]+[ ]*)\\}",
                    (string)value);

                if (extractedTokens.Length < 1 ||
                    string.IsNullOrEmpty(extractedTokens[0]) == true)
                {
                    return false;
                }

                string[] indexTokens = extractedTokens[0].Split(',');
                if (indexTokens.Length != 4)
                {
                    return false;
                }

                indices = new int[4];
                for (int i = 0; i < indexTokens.Length; ++i)
                {
                    int index;
                    if (int.TryParse(indexTokens[i].Trim(), out index) == false)
                    {
                        return false;
                    }

                    indices[i] = index;
                }
            }
            else if (value is int[])
            {
                indices = (int[])value;
            }
            else
            {
                return false;
            }

            if (indices == null || indices.Length != 4)
            {
                return false;
            }

            // The index array can only contain 0, 1, 2, 3, without duplication.
            for (int i = 0; i < 4; ++i)
            {
                if (indices.Count(idx => idx == i) != 1)
                {
                    return false;
                }
            }

            return true;
        }
    }
}
