﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryData;

namespace EffectMaker.DataModelLogic.BinaryConversionInfo
{
    /// <summary>
    /// The base class for binary conversion info classes.
    /// </summary>
    public abstract class BinaryConversionInfoBase : IDisposable
    {
        /// <summary>
        /// Dispose.
        /// </summary>
        public abstract void Dispose();

        /// <summary>
        /// Helper method for adding binary field definition to a binary structure definition.
        /// </summary>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddBinaryFieldDefinition(
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new BinaryFieldDefinition(
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }

        /// <summary>
        /// Helper method for adding a fixed size field definition to a binary structure definition.
        /// </summary>
        /// <param name="size">The constant size for the field definition.</param>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag to set to the field definition.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddFixedSizeFieldDefinition(
            int size,
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new FixedSizeBinaryFieldDefinition(
                size,
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }

        /// <summary>
        /// Helper method for adding an alignment field definition to a binary structure definition.
        /// </summary>
        /// <param name="alignment">The alignment for the field definition.</param>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag to set to the field definition.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddAlignmentFieldDefinition(
            int alignment,
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new AlignmentFieldDefinition(
                alignment,
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }

        /// <summary>
        /// Helper method for adding a field definition that represents a data offset in the binary data.
        /// </summary>
        /// <param name="adjustValue">The value to add to the final offset value of the field.</param>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag to set to the field definition.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddDataOffsetFieldDefinition(
            int adjustValue,
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new BinaryStructOffsetFieldDefinition(
                adjustValue,
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }

        /// <summary>
        /// Helper method for adding a field definition that represents the size of a block in the binary data.
        /// </summary>
        /// <param name="adjustValue">The value to add to the final size value of the field.</param>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag to set to the field definition.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddDataSizeFieldDefinition(
            int adjustValue,
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new BinaryStructSizeFieldDefinition(
                adjustValue,
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }

        /// <summary>
        /// Helper method for adding a field definition that selects underlying binary structure
        /// according to the input data type.
        /// </summary>
        /// <param name="ownerStructDef">The owner binary structure definition.</param>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="tag">The tag to set to the field definition.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structFieldDef">The underlying binary structure definition for the binary field.</param>
        /// <param name="inputProperties">The definition of the input data model properties.</param>
        protected static void AddSelectBinaryDataFieldDefinition(
            BinaryStructDefinition ownerStructDef,
            SendModificationTypes sendModificationType,
            string tag,
            ConverterDefinition converterDef,
            BinaryStructDefinition structFieldDef,
            params InputPropertyDefinition[] inputProperties)
        {
            var field = new SelectBinaryDataFieldDefinition(
                sendModificationType,
                converterDef,
                structFieldDef,
                inputProperties);

            if (string.IsNullOrEmpty(tag) == false)
            {
                field.Tag = tag;
            }

            ownerStructDef.AddField(field);
        }
    }
}
