﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Text;

using EffectMaker.Foundation.Primitives;

namespace EffectMaker.BusinessLogic.ViewerMessages
{
    /// <summary>
    /// A class that writes message contents to a stream.
    /// </summary>
    public class MessageWriter : BinaryWriter
    {
        /// <summary>
        /// Default constructor that creates an output memory stream.
        /// </summary>
        public MessageWriter() :
            base()
        {
            this.OutStream = new MemoryStream();
        }

        /// <summary>
        /// Constructor that creates the memory stream with the specified initial capacity.
        /// </summary>
        /// <param name="capacity">The initial capacity of the memory stream.</param>
        public MessageWriter(int capacity) :
            base()
        {
            this.OutStream = new MemoryStream(capacity);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="stream">The output stream.</param>
        public MessageWriter(Stream stream) :
            base(stream)
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="stream">The output stream.</param>
        /// <param name="encoding">The encoding.</param>
        public MessageWriter(Stream stream, Encoding encoding) :
            base(stream, encoding)
        {
        }

        /// <summary>
        /// 現在のストリームに MessageType を書き込みます.
        /// </summary>
        /// <param name="type">書き込む値.</param>
        public virtual void Write(MessageTypes type)
        {
            base.Write((int)type);
        }

        /// <summary>
        /// 現在のストリームに AssetType を書き込みます.
        /// </summary>
        /// <param name="type">書き込む値.</param>
        public virtual void Write(AssetTypes type)
        {
            base.Write((int)type);
        }

        /// <summary>
        /// 現在のストリームに Guid を書き込みます.
        /// </summary>
        /// <param name="guid">書き込む値.</param>
        public virtual void Write(Guid guid)
        {
            base.Write(guid.ToByteArray());
        }

        /// <summary>
        /// Write string, assign -1 to the size to write the full length of the string,
        /// or any value greater than zero to write a fix-size string.
        /// </summary>
        /// <remarks>This method will always write the string in ASCII encoding.</remarks>
        /// <param name="str">The string to write.</param>
        /// <param name="size">The fix-size string length.</param>
        public virtual void Write(string str, int size)
        {
            if (size <= 0)
            {
                size = str.Length + 1;
            }

            byte[] output = Enumerable.Repeat((byte)0, size).ToArray();

            Encoding.ASCII.GetBytes(str, 0, Math.Min(str.Length, size - 1), output, 0);

            base.Write(output);
        }

        /// <summary>
        /// 現在のストリームに bool を書き込みます. 値を保持しない場合はデフォルト値を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(bool? value)
        {
            if (value.HasValue)
            {
                byte val = (value.Value == true) ? (byte)1 : (byte)0;
                base.Write(val);
            }
            else
            {
                byte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに byte を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(byte? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                byte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに char を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(char? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                char val = '\0';
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに decimal を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(decimal? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                decimal val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに double を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(double? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                double val = 0.0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに float を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(float? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                float val = 0.0f;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに int を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(int? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                int val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに long を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(long? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                long val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに sbyte を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(sbyte? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                sbyte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに short を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(short? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                short val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに uint を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(uint? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                uint val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに ulong を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(ulong? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                ulong val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに ushort を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(ushort? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                ushort val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに Vecctor1d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1d value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1f value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1i value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1u value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに ColorRgba を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(ColorRgba value)
        {
            if (value != null)
            {
                base.Write(value.R);
                base.Write(value.G);
                base.Write(value.B);
                base.Write(value.A);
            }
            else
            {
                base.Write(1.0f);
                base.Write(1.0f);
                base.Write(1.0f);
                base.Write(1.0f);
            }
        }
    }
}
