﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.ViewerMessages
{
    /// <summary>
    /// Viewer message class that sends modified data (instead of the whole binary) to the viewer.
    /// </summary>
    public class DataModifyMessage : MessageBase
    {
        /// <summary>The type of the asset the binary data is.</summary>
        private AssetTypes assetType;

        /// <summary>The Guid of the asset.</summary>
        private Guid assetGuid;

        /// <summary>The name of the asset.</summary>
        private string assetName;

        /// <summary>The byte array that contains the modified data.</summary>
        private byte[] binaryData;

        /// <summary>The actual size of the binary data in bytes.</summary>
        private int binaryDataSize;

        /// <summary>The offset of the data in the original binary.</summary>
        private int binaryDataOffset;

        /// <summary>リセットが必要かどうかのフラグ.</summary>
        private short isRequiredReset;

        /// <summary>リソースバイナリのCPUエンディアン.</summary>
        private short endian;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="assetType">The type of the asset.</param>
        /// <param name="assetGuid">The Guid of the asset.</param>
        /// <param name="assetName">The name of the asset.</param>
        /// <param name="binaryData">The byte array that contains the modified data.</param>
        /// <param name="binaryDataSize">The actual size of the binary data.</param>
        /// <param name="binaryDataOffset">The offset of the data in the original binary.</param>
        /// <param name="isRequiredReset">リセットが必要なら非0、不要なら0.</param>
        /// <remarks>
        /// The byte array can be larger than the binary data, as long as the binary data size
        /// is correctly set.
        /// </remarks>
        public DataModifyMessage(
            AssetTypes assetType,
            Guid assetGuid,
            string assetName,
            byte[] binaryData,
            int binaryDataSize,
            int binaryDataOffset,
            short isRequiredReset) : base()
        {
            this.assetType = assetType;
            this.assetGuid = assetGuid;
            this.assetName = assetName;
            this.binaryData = binaryData;
            this.binaryDataSize = binaryDataSize;
            this.binaryDataOffset = binaryDataOffset;
            this.isRequiredReset = isRequiredReset;
            this.endian = (short)(BinaryConversionUtility.ForResource.IsLittleEndian ? 0 : 1);
        }

        /// <summary>
        /// Get the type of the message.
        /// </summary>
        public override MessageTypes MessageType
        {
            get { return MessageTypes.Modify; }
        }

        /// <summary>
        /// Get the size of the message.
        /// </summary>
        public override int MessageSize
        {
            get
            {
                // (sizeof(assetType) = 4) +
                // (sizeof(assetGuid) = 16) +
                // (fix-size asset name = 64) +
                // (sizeof(binaryDataOffset) = 4) +
                // (sizeof(binaryDataSize) = 4) +
                // (sizeof(isRequiredReset) = 2) +
                // (sizeof(endian) = 2) = 96
                return 96 + this.binaryDataSize;
            }
        }

        /// <summary>
        /// Get the asset type of the binary data.
        /// </summary>
        public AssetTypes AssetType
        {
            get { return this.assetType; }
        }

        /// <summary>
        /// Get the Guid of the asset.
        /// </summary>
        public Guid AssetGuid
        {
            get { return this.assetGuid; }
        }

        /// <summary>
        /// Write message data to a stream with the given writer.
        /// </summary>
        /// <param name="writer">The message writer.</param>
        /// <returns>True on success.</returns>
        protected override bool WriteMessage(MessageWriter writer)
        {
            if (base.WriteMessage(writer) == false)
            {
                return false;
            }

            writer.Write(BinaryConversionUtility.ForProtocol.Convert((int)this.assetType));
            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.assetGuid));

            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.binaryDataOffset));
            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.binaryDataSize));

            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.isRequiredReset));
            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.endian));

            writer.Write(this.assetName, 64);

            writer.Write(this.binaryData, 0, this.binaryDataSize);

            return true;
        }
    }
}
