﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Manages user data for reserved shader.
    /// </summary>
    public static class ReservedShaderUserDataManager
    {
        /// <summary>The contant maximum reserved shader count.</summary>
        public const int ReservedShaderCount = 8;

        /// <summary>The dictionary that caches the reserved shader icons with their file path as the key.</summary>
        private static readonly Dictionary<string, Image> IconCacheMap =
            new Dictionary<string, Image>();

        /// <summary>The reserved shader definitions.</summary>
        private static ReservedShaderDefinitionList definition = null;

        /// <summary>
        /// Get or set the reserved shader definition list.
        /// </summary>
        public static ReservedShaderDefinitionList Definition
        {
            get { return definition; }
            set { SetupDefinition(value); }
        }

        /// <summary>
        /// Load the icon image for the given reserved shader definition.
        /// </summary>
        /// <param name="def">The reserved shader definition.</param>
        /// <returns>The icon image, or null if failed loading the image.</returns>
        public static Image LoadReservedShaderIcon(ReservedShaderDefinition def)
        {
            Image icon;
            if (IconCacheMap.TryGetValue(def.IconPath, out icon) == false)
            {
                try
                {
                    icon = Image.FromFile(def.IconPath);
                    if (icon != null)
                    {
                        IconCacheMap.Add(def.IconPath, icon);
                    }
                }
                catch
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "ReservedShaderUserDataManager.LoadReservedShaderIcon : Failed loading reserved shader {0}'s icon '{1}'",
                        def.Name,
                        def.IconPath);
                    return null;
                }
            }

            return icon;
        }

        /// <summary>
        /// Get the reserved shader definition at the specified index.
        /// </summary>
        /// <param name="index">The index to the reserved shader definition.</param>
        /// <returns>The reserved shader definition or null if doesn't exist.</returns>
        public static ReservedShaderDefinition GetReservedShaderDefinition(int index)
        {
            if (Definition == null)
            {
                return null;
            }

            if (index < 0 || index >= Definition.ReservedShaderDefinitions.Count)
            {
                return null;
            }

            return Definition.ReservedShaderDefinitions[index];
        }

        /// <summary>
        /// Find the reserved shader definition with the created reserved shader user data.
        /// </summary>
        /// <param name="reservedShaderUserData">The reserved shader user data model.</param>
        /// <returns>The reserved shader definition or null if doesn't exist.</returns>
        public static ReservedShaderDefinition FindReservedShaderDefinition(
            UserDataBase reservedShaderUserData)
        {
            if (Definition == null)
            {
                return null;
            }

            if (reservedShaderUserData == null)
            {
                return null;
            }

            foreach (ReservedShaderDefinition def in Definition.ReservedShaderDefinitions)
            {
                if (def != null &&
                    def.UserDataInfo != null &&
                    reservedShaderUserData.GetType() == def.UserDataInfo.DataModelType)
                {
                    return def;
                }
            }

            return null;
        }

        /// <summary>
        /// Find the index of the reserved shader definition with the generated user data model.
        /// </summary>
        /// <param name="reservedShaderUserData">The created reserved shader user data model.</param>
        /// <returns>The reserved shader definition index or -1 if not found.</returns>
        public static int FindReservedShaderIndexByUserData(
            UserDataBase reservedShaderUserData)
        {
            if (Definition == null)
            {
                return -1;
            }

            for (int i = 0; i < Definition.ReservedShaderDefinitions.Count; ++i)
            {
                var def = Definition.ReservedShaderDefinitions[i];
                if (def != null &&
                    def.UserDataInfo != null &&
                    reservedShaderUserData.GetType() == def.UserDataInfo.DataModelType)
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the index of the reserved shader definition with their callback ID.
        /// </summary>
        /// <param name="callbackId">The callback ID of the reserved shader definition.</param>
        /// <returns>The reserved shader definition index or -1 if not found.</returns>
        public static int FindReservedShaderIndexByCallbackID(int callbackId)
        {
            if (Definition == null)
            {
                return -1;
            }

            for (int i = 0; i < Definition.ReservedShaderDefinitions.Count; ++i)
            {
                var def = Definition.ReservedShaderDefinitions[i];
                if (def != null && def.Id == callbackId)
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the user data info for the reserved shader at the specified index.
        /// </summary>
        /// <param name="index">The reserved shader index.</param>
        /// <returns>The user data info or null if the reserved shader doesn't exist.</returns>
        public static UserDataInfo FindReservedShaderUserDataInfo(int index)
        {
            if (Definition == null || Definition.ReservedShaderDefinitions == null)
            {
                return null;
            }

            if (index < 0 || index >= Definition.ReservedShaderDefinitions.Count)
            {
                return null;
            }

            ReservedShaderDefinition def = Definition.ReservedShaderDefinitions[index];
            if (def == null)
            {
                return null;
            }

            return def.UserDataInfo;
        }

        /// <summary>
        /// Find the ID for the reserved shader data.
        /// </summary>
        /// <param name="reservedShaderData">The reserved shader data instance.</param>
        /// <returns>The callback ID of the reserved shader definition.</returns>
        public static int FindReservedShaderID(UserDataBase reservedShaderData)
        {
            ReservedShaderDefinition def = FindReservedShaderDefinition(reservedShaderData);
            if (def == null)
            {
                return 0;
            }

            return def.Id;
        }

        /// <summary>
        /// Find the reserved shader user data info which to convert the given source
        /// type to.
        /// This method is for updating stripe data to the newly implemented reserved
        /// shader.
        /// </summary>
        /// <param name="sourceTypeName">The source stripe data type name.</param>
        /// <returns>The reserved shader user data info to convert to, or null if not found.</returns>
        public static UserDataInfo FindUpdateTargetUserData(string sourceTypeName)
        {
            if (Definition == null || Definition.ReservedShaderDefinitions == null)
            {
                return null;
            }

            foreach (var def in Definition.ReservedShaderDefinitions)
            {
                if (def != null && string.Equals(def.UpdateFromType, sourceTypeName) == true)
                {
                    return def.UserDataInfo;
                }
            }

            return null;
        }

        /// <summary>
        /// Helper method for setting up the reserved shader definition.
        /// </summary>
        /// <param name="def">The reserved shader definitions.</param>
        private static void SetupDefinition(ReservedShaderDefinitionList def)
        {
            definition = def;

            if (def == null)
            {
                CustomShaderUserDataManager.RemoveShader(ShaderTypes.ReservedVertexShader);
                CustomShaderUserDataManager.RemoveShader(ShaderTypes.ReservedFragmentShader);
            }
            else
            {
                // Reserved vertex shaders.
                CustomShaderUserDataManager.SetShader(
                    ShaderTypes.ReservedVertexShader,
                    def.ReservedShaderDefinitions.Select(d => d.VertexShaderFullPaths));

                // Reserved fragment shaders.
                CustomShaderUserDataManager.SetShader(
                    ShaderTypes.ReservedFragmentShader,
                    def.ReservedShaderDefinitions.Select(d => d.FragmentShaderFullPaths));
            }
        }
    }
}
