﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Data class that holds reserved shader definitions.
    /// </summary>
    public class ReservedShaderDefinitionList : IXmlDocSerializable
    {
        /// <summary>The path where this definition is loaded from.</summary>
        private string filePath = string.Empty;

        /// <summary>The base folder path.</summary>
        private string baseFolderPath = string.Empty;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public ReservedShaderDefinitionList()
        {
            this.ReservedShaderDefinitions = new List<ReservedShaderDefinition>();
        }

        /// <summary>
        /// The base folder path of the definitions.
        /// </summary>
        public string BaseFolderPath
        {
            get
            {
                return this.baseFolderPath;
            }

            private set
            {
                this.baseFolderPath = PathUtility.LoadEnvironments(value);
                this.ReservedShaderDefinitions.ForEach(s => s.BaseFolderPath = this.baseFolderPath);
            }
        }

        /// <summary>
        /// Get or set the file path where this definition was loaded from.
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.filePath = value;
                this.BaseFolderPath = Path.GetDirectoryName(this.filePath);
            }
        }

        /// <summary>
        /// The reserved shader definitions.
        /// </summary>
        public List<ReservedShaderDefinition> ReservedShaderDefinitions { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.ReservedShaderDefinitions = this.ReadElementsByTagName<ReservedShaderDefinition>(
                context,
                "DefLinkData").ToList();

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Load the reserved shader definition file with the link path.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool LoadReservedShaderDefinitionFiles()
        {
            bool result = true;
            foreach (ReservedShaderDefinition link in this.ReservedShaderDefinitions)
            {
                result &= link.LoadReservedShaderDefinitionFile();
            }

            return result;
        }
    }

    /// <summary>
    /// Data class that holds informations of a reserved shader.
    /// </summary>
    public class ReservedShaderDefinition : IXmlDocSerializable
    {
        /// <summary>The relative path of the definition file.</summary>
        private string definitionRelativePath = string.Empty;

        /// <summary>The reserved shader definition.</summary>
        private InternalReservedShaderDefinition internalDefinition = null;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public ReservedShaderDefinition()
        {
        }

        /// <summary>
        /// The base folder path of the definitions.
        /// </summary>
        public string BaseFolderPath { get; set; }

        /// <summary>
        /// The ID of the reserved shader.
        /// </summary>
        public int Id { get; set; }

        /// <summary>
        /// Get or set the name of the type to update from.
        /// (e.g. RSStripeComplexUserData is converted from StripeComplexUserData)
        /// </summary>
        public string UpdateFromType { get; set; }

        /// <summary>
        /// The file path to the custom shader definition.
        /// </summary>
        public string DefinitionPath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.definitionRelativePath); }
            set { this.definitionRelativePath = value; }
        }

        /// <summary>
        /// Get the flag indicating whether the definition is correctly loaded.
        /// </summary>
        public bool IsDefinitionLoaded
        {
            get { return this.internalDefinition != null; }
        }

        /// <summary>
        /// The display name for the reserved shader.
        /// </summary>
        public string Name
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.Name : string.Empty; }
        }

        /// <summary>
        /// The English display name for the custom shader.
        /// </summary>
        public string NameEn
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.NameEn : string.Empty; }
        }

        /// <summary>
        /// The source file path of the data model.
        /// </summary>
        public string DataModelFilePath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.DataModelFullFilePath : string.Empty; }
        }

        /// <summary>
        /// The XAML file name for the custom shader UI.
        /// </summary>
        public string XamlFilePath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.XamlFullFilePath : string.Empty; }
        }

        /// <summary>
        /// The source file path of the binary conversion info.
        /// </summary>
        public string BinaryConversionInfoPath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.BinaryConversionInfoFullPath : string.Empty; }
        }

        /// <summary>
        /// The file path of the icon to represent the reserved shader.
        /// </summary>
        public string IconPath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.IconFullFilePath : string.Empty; }
        }

        /// <summary>
        /// このプラグインがGPUパーティクルを許容するか否かを取得します。
        /// </summary>
        public bool EnableGpu
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.EnableGpu : false; }
        }

        /// <summary>
        /// このプラグインがSOパーティクルを許容するか否かを取得します。
        /// </summary>
        public bool EnableSo
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.EnableSo : false; }
        }

        /// <summary>
        /// Enumerates the full path of all the vertex shader files.
        /// </summary>
        public IEnumerable<string> VertexShaderFullPaths
        {
            get
            {
                if (this.IsDefinitionLoaded == false)
                {
                    return Enumerable.Empty<string>();
                }

                return this.internalDefinition.VertexShaderFullPaths;
            }
        }

        /// <summary>
        /// Enumerates the full path of all the fragment shader files.
        /// </summary>
        public IEnumerable<string> FragmentShaderFullPaths
        {
            get
            {
                if (this.IsDefinitionLoaded == false)
                {
                    return Enumerable.Empty<string>();
                }

                return this.internalDefinition.FragmentShaderFullPaths;
            }
        }

        /// <summary>
        /// Get or set the loaded user data info for this reserved shader definition.
        /// </summary>
        public UserDataInfo UserDataInfo { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            this.Id = this.ReadAttribute(context, "SettingId", this.Id);
            this.UpdateFromType = this.ReadAttribute(context, "UpdateFrom", this.UpdateFromType);

            // Read the file path of the actual custom shader definition.
            this.definitionRelativePath = context.CurrentNode.InnerText;

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Load the reserved shader definition file with the link path.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool LoadReservedShaderDefinitionFile()
        {
            if (File.Exists(this.DefinitionPath) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "Reserved shader definition does not exist. ({0})",
                    this.DefinitionPath);
                return false;
            }

            try
            {
                InternalReservedShaderDefinition def =
                    SerializationHelper.LoadXmlDocSerializable<InternalReservedShaderDefinition>(this.DefinitionPath);

                def.BaseFolderPath = Path.GetDirectoryName(this.DefinitionPath);

                this.internalDefinition = def;
            }
            catch
            {
                Logger.Log(
                    LogLevels.Warning,
                    "Failed loading reserved shader definition {0}",
                    this.DefinitionPath);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Data class that holds informations of a reserved shader.
        /// </summary>
        private class InternalReservedShaderDefinition : GeneralShaderDefinition
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public InternalReservedShaderDefinition()
            {
            }

            /// <summary>
            /// The display name for the custom shader.
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The English display name for the custom shader.
            /// </summary>
            public string NameEn { get; set; }

            /// <summary>
            /// The source file path of the data model.
            /// </summary>
            public string DataModelFilePath { get; set; }

            /// <summary>
            /// The full path of the data model source file.
            /// </summary>
            public string DataModelFullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.DataModelFilePath); }
            }

            /// <summary>
            /// The XAML file name for the custom shader UI.
            /// </summary>
            public string XamlFilePath { get; set; }

            /// <summary>
            /// The full path of the XAML file for the custom action UI.
            /// </summary>
            public string XamlFullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.XamlFilePath); }
            }

            /// <summary>
            /// The source file path of the binary conversion info.
            /// </summary>
            public string BinaryConversionInfoPath { get; set; }

            /// <summary>
            /// The full path to the binary conversion info source file.
            /// </summary>
            public string BinaryConversionInfoFullPath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.BinaryConversionInfoPath); }
            }

            /// <summary>
            /// The file name for the icon to represent the reserved shader on the UI.
            /// </summary>
            public string IconPath { get; set; }

            /// <summary>
            /// このプラグインがGPUパーティクルを許容するか否かを取得または設定します。
            /// </summary>
            public bool EnableGpu { get; set; }

            /// <summary>
            /// このプラグインがSOパーティクルを許容するか否かを取得または設定します。
            /// </summary>
            public bool EnableSo { get; set; }

            /// <summary>
            /// The full path of the icon to represent the reserved shader on the UI.
            /// </summary>
            public string IconFullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.IconPath); }
            }

            /// <summary>
            /// Deserializes from the given XML node.
            /// </summary>
            /// <param name="context">The data context needed for the deserialization.</param>
            /// <returns>True on success.</returns>
            public override bool ReadXml(XmlDocSerializationContext context)
            {
                if (base.ReadXml(context) == false)
                {
                    return false;
                }

                this.Name = this.ReadAttribute(context, "Label", this.Name);
                this.NameEn = this.ReadAttribute(context, "Label_EN", this.NameEn);
                this.IconPath = this.ReadAttribute(context, "Icon", this.IconPath);
                this.EnableGpu = this.ReadAttribute(context, "EnableGpu", this.EnableGpu);
                this.EnableSo = this.ReadAttribute(context, "EnableSo", this.EnableSo);

                var node = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode,
                    "DataModelSetting").FirstOrDefault();

                if (node != null)
                {
                    context.PushCurrentNode(node);

                    this.DataModelFilePath =
                        this.ReadAttribute(context, "DataModel", this.DataModelFilePath);

                    this.XamlFilePath =
                        this.ReadAttribute(context, "UI", this.XamlFilePath);

                    this.BinaryConversionInfoPath =
                        this.ReadAttribute(context, "ConversionInfo", this.BinaryConversionInfoPath);

                    context.PopCurrentNode();
                }

                return true;
            }
        }
    }
}
