﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Holds all the user data conversion info.
    /// </summary>
    public class ObsoleteUserDataConversionInfoCollection : IXmlDocSerializable
    {
        /// <summary>The cached user data conversion info.</summary>
        private readonly Dictionary<string, ObsoleteUserDataConversionInfo> cachedConversionInfo =
            new Dictionary<string, ObsoleteUserDataConversionInfo>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public ObsoleteUserDataConversionInfoCollection()
        {
        }

        /// <summary>
        /// Get or set user data conversion info with the specified user data file path.
        /// </summary>
        /// <param name="filePath">The file path.</param>
        /// <returns>The user data conversion info.</returns>
        public ObsoleteUserDataConversionInfo this[string filePath]
        {
            get
            {
                ObsoleteUserDataConversionInfo info;
                if (this.cachedConversionInfo.TryGetValue(filePath, out info) == false)
                {
                    return null;
                }

                return info;
            }

            set
            {
                this.cachedConversionInfo[filePath] = value;
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            // Read the list of assembly data.
            var list = this.ReadListElement<ObsoleteUserDataConversionInfo>(context, "Items", null);

            this.cachedConversionInfo.Clear();
            if (list != null)
            {
                list.ForEach(item => this.cachedConversionInfo.Add(item.FilePath, item));
            }

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteEnumerableElement(
                context,
                "Items",
                this.cachedConversionInfo.Values);

            return true;
        }
    }

    /// <summary>
    /// Information for converting obsolete user data.
    /// </summary>
    public class ObsoleteUserDataConversionInfo : IXmlDocSerializable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public ObsoleteUserDataConversionInfo()
        {
            this.FilePath = string.Empty;
            this.ModifyTime = DateTime.MinValue;
            this.DataModelName = string.Empty;
            this.ConversionCommandName = string.Empty;
            this.OutputFolderPath = string.Empty;
            this.Language = string.Empty;
        }

        /// <summary>
        /// Get or set the path of the user data file.
        /// </summary>
        public string FilePath { get; set; }

        /// <summary>
        /// Get or set the last modified time of the user data file.
        /// </summary>
        public DateTime ModifyTime { get; set; }

        /// <summary>
        /// Get or set the data model class name.
        /// </summary>
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the command name for conversion.
        /// E.g. for user shader, 'usd'; for custom actions, 'udd';
        /// </summary>
        public string ConversionCommandName { get; set; }

        /// <summary>
        /// Get or set the directory path of the conversion output folder.
        /// </summary>
        public string OutputFolderPath { get; set; }

        /// <summary>
        /// Get or set the UI language being used.
        /// </summary>
        public string Language { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.FilePath = this.ReadAttribute(context, "FilePath", this.FilePath);

            this.ModifyTime = DateTime.FromFileTimeUtc(
                this.ReadAttribute<long>(context, "ModifyTime", (long)0));

            this.DataModelName = this.ReadAttribute(
                context,
                "DataModelName",
                this.DataModelName);

            this.ConversionCommandName = this.ReadAttribute(
                context,
                "ConversionCommandName",
                this.ConversionCommandName);

            this.OutputFolderPath = this.ReadAttribute(
                context,
                "OutputFolderPath",
                this.OutputFolderPath);

            this.Language = this.ReadAttribute(
                context,
                "Language",
                this.Language);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "FilePath", this.FilePath);
            this.WriteAttribute(context, "ModifyTime", this.ModifyTime.ToFileTimeUtc());
            this.WriteAttribute(context, "DataModelName", this.DataModelName);
            this.WriteAttribute(context, "ConversionCommandName", this.ConversionCommandName);
            this.WriteAttribute(context, "OutputFolderPath", this.OutputFolderPath);
            this.WriteAttribute(context, "Language", this.Language);

            return true;
        }
    }
}
