﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// エミッタ拡張パラメータは1つしか定義されない予定だが、
    /// 将来的に複数読み込んで切り替えたくなる予感がするのでリストを挟んでおく。
    /// </summary>
    public class EmitterExtParamsDefinitionList : IXmlDocSerializable
    {
        /// <summary>
        /// Get or set the Emitter Extension Parameters defiitions.
        /// </summary>
        public EmitterExtParamsDefinition EmitterExtParamsDefinition { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            var items = this.ReadElementsByTagNames<EmitterExtParamsDefinition>(
                context, new [] { "EmitterExtParamsDefinition" });

            this.EmitterExtParamsDefinition = items.FirstOrDefault();

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }
    }

    /// <summary>
    /// Stores a custom action definition.
    /// </summary>
    [Serializable]
    public class EmitterExtParamsDefinition : IXmlDocSerializable
    {
        /// <summary>
        /// The caption for the custom action.
        /// </summary>
        private string caption;

        /// <summary>The base folder path.</summary>
        private string baseFolderPath = string.Empty;

        /// <summary>
        /// UDD2.0を使用した定義か否かを取得します。
        /// </summary>
        [XmlIgnore]
        public bool UsingUdd2 { get; private set; }

        /// <summary>
        /// Get or set the base folder path.
        /// </summary>
        [XmlIgnore]
        public string BaseFolderPath
        {
            get
            {
                return this.baseFolderPath;
            }

            set
            {
                this.baseFolderPath = PathUtility.LoadEnvironments(value);
            }
        }

        /// <summary>
        /// The Udd2 file name. (can be a relative path)
        /// </summary>
        public string Udd2FileName { get; set; }

        /// <summary>
        /// The full path of the Udd2 file.
        /// </summary>
        public string Udd2FullFilePath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.Udd2FileName); }
        }

        /// <summary>
        /// The data model source file name. (can be a relative path)
        /// </summary>
        public string DataModelFileName { get; set; }

        /// <summary>
        /// The full path of the data model source file.
        /// </summary>
        public string DataModelFullFilePath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.DataModelFileName); }
        }

        /// <summary>
        /// The XAML file name for the custom action UI.
        /// </summary>
        public string XamlFileName { get; set; }

        /// <summary>
        /// The full path of the XAML file for the custom action UI.
        /// </summary>
        public string XamlFullFilePath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.XamlFileName); }
        }

        /// <summary>
        /// The binary conversion info source file name.
        /// </summary>
        public string BinaryConversionInfoPath { get; set; }

        /// <summary>
        /// The full path to the binary conversion info source file.
        /// </summary>
        public string BinaryConversionInfoFullPath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.BinaryConversionInfoPath); }
        }

        /// <summary>
        /// The text of the root control's text in the XAML file.
        /// </summary>
        public string Caption
        {
            get
            {
                if (string.IsNullOrEmpty(this.caption) != true)
                {
                    return this.caption;
                }

                if (this.UsingUdd2)
                {
                    this.caption = UserDataHelper.Instance.Udd2Converter.LoadDefinition(this.Udd2FullFilePath)
                        ? UserDataHelper.Instance.Udd2Converter.DefineName : string.Empty;
                }
                else
                {
                    this.caption = this.ParseXamlRootControlText(this.XamlFullFilePath);
                }

                return this.caption;
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            this.Udd2FileName =
                this.ReadAttribute(context, "DefinitionFile", this.Udd2FileName);
            if (string.IsNullOrEmpty(this.Udd2FileName))
            {
                this.Udd2FileName =
                    this.ReadAttribute(context, "Udd2Definition", this.Udd2FileName);
            }

            if (!string.IsNullOrEmpty(this.Udd2FileName))
            {
                this.UsingUdd2 = true;
                return true;
            }

            this.DataModelFileName =
                this.ReadAttribute(context, "DataModel", this.DataModelFileName);

            this.XamlFileName =
                this.ReadAttribute(context, "UI", this.XamlFileName);

            this.BinaryConversionInfoPath =
                this.ReadAttribute(context, "ConversionInfo", this.BinaryConversionInfoPath);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Parse the Text attribute of the root control in the XAML file.
        /// </summary>
        /// <param name="xamlFilePath">The XAML file path.</param>
        /// <returns>The text.</returns>
        private string ParseXamlRootControlText(string xamlFilePath)
        {
            if (string.IsNullOrEmpty(xamlFilePath) == true)
            {
                return string.Empty;
            }

            try
            {
                // Load the XAML file and parse it with XmlReader.
                using (var stream = new FileStream(this.XamlFullFilePath, FileMode.Open, FileAccess.Read))
                using (var reader = XmlReader.Create(stream))
                {
                    if (reader.Read() == true &&
                        reader.MoveToFirstAttribute() == true)
                    {
                        do
                        {
                            if (reader.Name == "Text")
                            {
                                return reader.Value;
                            }
                        }
                        while (reader.MoveToNextAttribute());
                    }
                }
            }
            catch
            {
            }

            return Path.GetFileNameWithoutExtension(xamlFilePath);
        }
    }
}
