﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.SpecDefinitions
{
    /// <summary>
    /// 動作ターゲット(Spec)定義を管理するマネージャです。
    /// </summary>
    public static class SpecManager
    {
        /// <summary>
        /// 読み込んだSpec定義のリストです。
        /// </summary>
        private static List<SpecDefinition> specDefinitions = new List<SpecDefinition>();

        /// <summary>
        /// 現在選択しているSpec定義のインデックスです。
        /// </summary>
        private static int selectedIndex = 0;

        /// <summary>
        /// 読み込んだSpec定義のリストを取得します。
        /// </summary>
        public static List<SpecDefinition> SpecDefinitions
        {
            get { return specDefinitions; }
        }

        /// <summary>
        /// 選択しているSpec定義のインデックスを取得または設定します。
        /// </summary>
        public static int SelectedIndex
        {
            get
            {
                return selectedIndex;
            }

            set
            {
                if (value < 0 || specDefinitions.Count <= value)
                {
                    return;
                }

                selectedIndex = value;

                // ここでスペック切り換え時の処理を行う
                BinaryConversionUtility.ForResource.IsLittleEndian = CurrentSpec.CpuEndian == 0;
                if (CurrentSpecChanged != null)
                {
                    CurrentSpecChanged(null, null);
                }
            }
        }

        /// <summary>
        /// 現在のSpec定義を取得します。
        /// </summary>
        public static SpecDefinition CurrentSpec
        {
            get
            {
                return specDefinitions[SelectedIndex];
            }
        }

        /// <summary>
        /// CurrentSpec変更イベント
        /// </summary>
        public static event EventHandler CurrentSpecChanged;

        /// <summary>
        /// Spec定義を読み込んで初期化します。
        /// </summary>
        /// <param name="executePath">実行ファイルのパス</param>
        public static void Initialize(string executePath)
        {
            var files = Directory.EnumerateFiles(Path.Combine(executePath, "Specs"), "*.def");
            foreach (var file in files)
            {
                try
                {
                    var result = SerializationHelper.LoadXmlDocSerializable<SpecDefinition>(file);
                    if (result != null)
                    {
                        if (result.BinaryHeader == "VFXB" && result.AddressSize == 4)
                        {
                            // VFXB は 32bit 版のみ受け付けるようにする
                            continue;
                        }

                        specDefinitions.Add(result);
                    }
                }
                catch(Exception e)
                {
                    Logger.Log(LogLevels.Error, "Invalid Spec Definiton!: " + e.Message);
                }
            }
        }
    }
}
