﻿using System;
using System.IO;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Search
{
    #region SearchItem

    /// <summary>
    /// 検索置換用データです。
    /// </summary>
    [Serializable]
    public class SearchItem : IXmlDocSerializable, ICloneable, ISettable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchItem()
        {
            this.OptionItems = new ArrayCollection<OptionItem>();
            this.EnumDefinitionItems = new ArrayCollection<EnumDefinitionItem>();
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchItem(SearchItem obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            SearchItem srcData = src as SearchItem;

            if (src == null)
            {
                return false;
            }

            this.UIPath = srcData.UIPath;
            this.Title = srcData.Title;
            this.XPath = srcData.XPath;
            this.Type = srcData.Type;

            this.OptionItems = (ArrayCollection<OptionItem>)srcData.OptionItems.Clone();
            this.EnumDefinitionItems = (ArrayCollection<EnumDefinitionItem>)srcData.EnumDefinitionItems.Clone();

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new SearchItem(this);
        }

        #endregion

        #region IXmlDocSerializable methods

        /// <summary>
        /// オブジェクトデータを XML ノードデータからデシリアライズします。
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.UIPath = this.ReadElement(context, nameof(this.UIPath), this.UIPath);
            this.Title  = this.ReadElement(context, nameof(this.Title ), this.Title);
            this.XPath  = this.ReadElement(context, nameof(this.XPath ), this.XPath);
            this.Type   = this.ReadElement(context, nameof(this.Type  ), this.Type);

            this.OptionItems = this.ReadElement(context, nameof(this.OptionItems), this.OptionItems);
            this.EnumDefinitionItems = this.ReadElement(context, nameof(this.EnumDefinitionItems), this.EnumDefinitionItems);

            return true;
        }

        /// <summary>
        /// オブジェクトデータを XML ノードデータにシリアライズします.
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.UIPath), this.UIPath);
            this.WriteElement(context, nameof(this.Title ), this.Title);
            this.WriteElement(context, nameof(this.XPath ), this.XPath);
            this.WriteElement(context, nameof(this.Type  ), this.Type);

            this.WriteElement(context, nameof(this.OptionItems), this.OptionItems);
            this.WriteElement(context, nameof(this.EnumDefinitionItems), this.EnumDefinitionItems);

            return true;
        }

        #endregion

        /// <summary>
        /// Get or set the Title data.
        /// </summary>
        public string Title { get; set; }

        /// <summary>
        /// Get or set the XPath data.
        /// </summary>
        public string XPath { get; set; }

        /// <summary>
        /// Get or set the UIPath data.
        /// </summary>
        public string UIPath { get; set; }

        /// <summary>
        /// Get or set the Type data.
        /// </summary>
        public string Type { get; set; }

        /// <summary>
        /// オプションデータコレクションを取得または設定します。
        /// </summary>
        public ArrayCollection<OptionItem> OptionItems { get; set; }

        /// <summary>
        /// 定義データコレクションを取得または設定します。
        /// </summary>
        public ArrayCollection<EnumDefinitionItem> EnumDefinitionItems { get; set; }
    }

    #endregion

    #region EnumDefinitionItem

    /// <summary>
    /// 列挙値定義データアイテムです。
    /// </summary>
    [Serializable]
    public class EnumDefinitionItem : IXmlDocSerializable, ICloneable, ISettable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EnumDefinitionItem()
        {
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public EnumDefinitionItem(EnumDefinitionItem obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            EnumDefinitionItem srcData = src as EnumDefinitionItem;

            if (src == null)
            {
                return false;
            }

            this.Key = srcData.Key;
            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new EnumDefinitionItem(this);
        }

        #endregion

        #region IXmlDocSerializable methods

        /// <summary>
        /// オブジェクトデータを XML ノードデータからデシリアライズします。
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Key = this.ReadElement(context, nameof(this.Key), this.Key);
            this.Value = this.ReadElement(context, nameof(this.Value), this.Value);

            return true;
        }

        /// <summary>
        /// オブジェクトデータを XML ノードデータにシリアライズします.
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.Key), this.Key);
            this.WriteElement(context, nameof(this.Value), this.Value);

            return true;
        }

        #endregion

        /// <summary>
        /// キーを取得または設定します。
        /// </summary>
        public string Key { get; set; }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public string Value { get; set; }
    }

    #endregion

    #region OptionItem

    /// <summary>
    /// 検索置換オプションアイテムです。
    /// </summary>
    [Serializable]
    public class OptionItem : IXmlDocSerializable, ICloneable, ISettable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OptionItem()
        {
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public OptionItem(OptionItem obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            OptionItem srcData = src as OptionItem;

            if (src == null)
            {
                return false;
            }

            this.Key = srcData.Key;
            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new OptionItem(this);
        }

        #endregion

        #region IXmlDocSerializable methods

        /// <summary>
        /// オブジェクトデータを XML ノードデータからデシリアライズします。
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Key = this.ReadElement(context, nameof(this.Key), this.Key);
            this.Value = this.ReadElement(context, nameof(this.Value), this.Value);

            return true;
        }

        /// <summary>
        /// オブジェクトデータを XML ノードデータにシリアライズします.
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.Key), this.Key);
            this.WriteElement(context, nameof(this.Value), this.Value);

            return true;
        }

        #endregion

        /// <summary>
        /// キーを取得または設定します。
        /// </summary>
        public string Key { get; set; }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public string Value { get; set; }
    }

    #endregion

    #region SearchTableByDataModelMaker

    /// <summary>
    /// 検索置換用データテーブルです。
    /// </summary>
    public class SearchTableByDataModelMaker : IXmlDocSerializable, ICloneable, ISettable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchTableByDataModelMaker()
        {
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchTableByDataModelMaker(SearchTableByDataModelMaker obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new SearchTableByDataModelMaker(this);
        }

        #endregion

        #region IXmlDocSerializable methods

        /// <summary>
        /// オブジェクトデータを XML ノードデータからデシリアライズします。
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.SearchItems = this.ReadElement(context, nameof(this.SearchItems), this.SearchItems);

            return true;
        }

        /// <summary>
        /// オブジェクトデータを XML ノードデータにシリアライズします.
        /// </summary>
        /// <param name="context">データコンテキスト</param>
        /// <returns>処理が成功したとき True を返します。</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.SearchItems), this.SearchItems);

            return true;
        }

        #endregion

        /// <summary>
        /// 検索置換用データコレクションを取得または設定します。
        /// </summary>
        public ArrayCollection<SearchItem> SearchItems { get; set; }
    }

    #endregion

    /// <summary>
    /// シリアライズクラス
    /// </summary>
    public static class SearchTableByDataModelMakerSerializer
    {
        /// <summary>
        /// 保存します。
        /// </summary>
        /// <param name="src">保存対象です。</param>
        /// <param name="path">保存パスです。</param>
        public static void Save(SearchTableByDataModelMaker src, string path)
        {
            using (var stream = new FileStream(path, FileMode.Create, FileAccess.Write))
            {
                SerializationHelper.SaveXmlDocSerializable(src, stream);
            }
        }

        /// <summary>
        /// ロードします。
        /// </summary>
        /// <param name="path">ロードパスです。</param>
        /// <returns>ロード対象です。</returns>
        public static SearchTableByDataModelMaker Load(string path)
        {
            using (var stream = new FileStream(path, FileMode.Open, FileAccess.Read))
            {
                return SerializationHelper.LoadXmlDocSerializable<SearchTableByDataModelMaker>(stream);
            }
        }

        /// <summary>
        /// ロードします。
        /// </summary>
        /// <param name="path">ロードパスです。</param>
        /// <returns>ロード対象です。</returns>
        public static SearchTableByDataModelMaker Load(Stream stream)
        {
            return SerializationHelper.LoadXmlDocSerializable<SearchTableByDataModelMaker>(stream);
        }
    }
}
