﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Protocol
{
    /// <summary>
    /// ビューアからの要求メッセージです.
    /// </summary>
    public class RequestMessage
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RequestMessage()
        {
            this.Type = MessageType.Unkonwn;
            this.PacketSize = 0;
            this.Operation = OperationType.None;
            this.ResourceSize = 0;
            this.EmitterSetName = string.Empty;
            this.DuplicateEmitterSetName = string.Empty;
        }

        /// <summary>
        /// メッセージタイプです.
        /// </summary>
        public MessageType Type { get; set; }

        /// <summary>
        /// バッファサイズです.
        /// </summary>
        public int PacketSize { get; set; }

        /// <summary>
        /// リソースサイズです.
        /// </summary>
        public int ResourceSize { get; set; }

        /// <summary>
        /// 操作タイプです.
        /// </summary>
        public OperationType Operation { get; set; }

        /// <summary>
        /// エミッタセット名です.
        /// </summary>
        public string EmitterSetName { get; set; }

        /// <summary>
        /// 複製するエミッタセット名です.
        /// </summary>
        public string DuplicateEmitterSetName { get; set; }

        /// <summary>
        /// デシリアライズします.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        /// <returns>成功したときtrueを返します.</returns>
        public bool Deserialize(EffectMaker.Communicator.Message msg)
        {
            if (msg.Size > 0)
            {
                int offset = 0;     // 先頭からのオフセット.

                // メッセージのカテゴリを取得.
                MessageType type = (MessageType)BinaryConversionUtility.ForProtocol.ConvertBack<int>(msg.Buffer, offset);

                // エディタ制御メッセージではない場合は終了.
                if (type != MessageType.Operation)
                {
                    return false;
                }

                int nameSize = 260 * 2; // ランタイム側で設定している最大文字数.

                offset += 4;
                this.Type = type;

                // メッセージ全体のサイズを取得.
                this.PacketSize = BinaryConversionUtility.ForProtocol.ConvertBack<int>(msg.Buffer, offset);
                offset += 4;

                // エディタ操作タイプを取得.
                this.Operation = (OperationType)BinaryConversionUtility.ForProtocol.ConvertBack<int>(msg.Buffer, offset);
                offset += 4;

                // リソースサイズを取得.
                this.ResourceSize = BinaryConversionUtility.ForProtocol.ConvertBack<int>(msg.Buffer, offset);
                offset += 4;

                // エミッタセット名を取得.
                char[] esetNames = BinaryConversionUtility.ForProtocol.ToCharArray(nameSize, msg.Buffer, offset);

                // null文字を削除して，string型に変換.
                this.EmitterSetName = new string(esetNames).Trim('\0');
                offset += nameSize;

                // 複製操作の場合.
                if (this.Operation == OperationType.Duplicate)
                {
                    // 複製後の名前を取得.
                    char[] duplicateName = BinaryConversionUtility.ForProtocol.ToCharArray(nameSize, msg.Buffer, offset);

                    // null文字を削除して，string型に変換.
                    this.DuplicateEmitterSetName = new string(duplicateName).Trim('\0');
                    offset += nameSize;
                }

                return true;
            }

            return false;
        }
    }
}
