﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.BusinessLogic.Protocol
{
    /// <summary>
    /// パケットライターです.
    /// </summary>
    public class PacketWriter : BinaryWriter
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="output">出力ストリーム</param>
        public PacketWriter(Stream output)
            : base(output)
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="output">供給するストリーム</param>
        /// <param name="encoding">文字エンコーディング</param>
        public PacketWriter(Stream output, Encoding encoding)
            : base(output, encoding)
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// 現在のストリームに MessageType を書き込みます.
        /// </summary>
        /// <param name="type">書き込む値.</param>
        public virtual void Write(MessageType type)
        {
            base.Write((int)type);
        }

        /// <summary>
        /// 現在のストリームに AssetType を書き込みます.
        /// </summary>
        /// <param name="type">書き込む値.</param>
        public virtual void Write(AssetTypes type)
        {
            base.Write((int)type);
        }

        /// <summary>
        /// 現在のストリームに固定長文字列を書き込みます.
        /// </summary>
        /// <param name="str">可変長文字列.</param>
        /// <param name="size">固定長のサイズ.</param>
        /// <param name="encoding">エンコーディング.</param>
        public virtual void Write(string str, int size, Encoding encoding)
        {
            if (size <= 0)
            {
                size = str.Length + 1;
            }

            byte[] output = Enumerable.Repeat((byte)0, size).ToArray();

            encoding.GetBytes(str, 0, Math.Min(str.Length, size - 1), output, 0);

            base.Write(output);
        }

        /// <summary>
        /// 現在のストリームに bool を書き込みます. 値を保持しない場合はデフォルト値を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(bool? value)
        {
            if (value.HasValue)
            {
                byte val = (value.Value == true) ? (byte)1 : (byte)0;
                base.Write(val);
            }
            else
            {
                byte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに byte を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(byte? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                byte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに char を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(char? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                char val = '\0';
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに decimal を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(decimal? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                decimal val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに double を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(double? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                double val = 0.0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに float を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(float? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                float val = 0.0f;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに int を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(int? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                int val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに long を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(long? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                long val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに sbyte を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(sbyte? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                sbyte val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに short を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(short? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                short val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに uint を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(uint? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                uint val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに ulong を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(ulong? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                ulong val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに ushort を書き込みます. 値を保持しない場合はデフォルト値が書き込まれます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(ushort? value)
        {
            if (value.HasValue)
            {
                base.Write(value.Value);
            }
            else
            {
                ushort val = 0;
                base.Write(val);
            }
        }

        /// <summary>
        /// 現在のストリームに Vecctor1d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1d value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1f value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1i value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector1u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector1u value)
        {
            if (value != null)
            {
                base.Write(value.X);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector2u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector2u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector3u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector3u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4d を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4d value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                double zero = 0.0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4f を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4f value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                float zero = 0.0f;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4i を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4i value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                int zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }

        /// <summary>
        /// 現在のストリームに Vector4u を書き込みます.
        /// </summary>
        /// <param name="value">書き込む値</param>
        public virtual void Write(Vector4u value)
        {
            if (value != null)
            {
                base.Write(value.X);
                base.Write(value.Y);
                base.Write(value.Z);
                base.Write(value.W);
            }
            else
            {
                uint zero = 0;
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
                base.Write(zero);
            }
        }
    }
}
