﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.ProjectConfig
{
    /// <summary>
    /// プロジェクト設定のファイルイベントデータです。
    /// </summary>
    public class FileEventProjectConfig : OptionsBase
    {
        /// <summary>
        /// プリセットデータフォルダパスです。
        /// </summary>
        private ExpandablePath presetFolderPath;

        /// <summary>
        /// 保存前コマンドパスです。
        /// </summary>
        private ExpandablePath preSaveCommandFilePath;

        /// <summary>
        /// 保存後コマンドパスです。
        /// </summary>
        private ExpandablePath postSaveCommandFilePath;

        /// <summary>
        /// 削除前コマンドパスです。
        /// </summary>
        private ExpandablePath preDeleteCommandFilePath;

        /// <summary>
        /// 削除後コマンドパスです。
        /// </summary>
        private ExpandablePath postDeleteCommandFilePath;

        /// <summary>
        /// オープン前コマンドパスを取得または設定します。
        /// </summary>
        private ExpandablePath preOpenCommandFilePath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileEventProjectConfig()
        {
            this.presetFolderPath          = new ExpandablePath();
            this.preSaveCommandFilePath    = new ExpandablePath();
            this.postSaveCommandFilePath   = new ExpandablePath();
            this.preDeleteCommandFilePath  = new ExpandablePath();
            this.postDeleteCommandFilePath = new ExpandablePath();
            this.preOpenCommandFilePath    = new ExpandablePath();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="src">初期値</param>
        public FileEventProjectConfig(FileEventProjectConfig src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// ファイルイベントのベースフォルダパスを設定します。
        /// </summary>
        public string BasePath
        {
            set
            {
                this.presetFolderPath.BasePath          = value;
                this.preSaveCommandFilePath.BasePath    = value;
                this.postSaveCommandFilePath.BasePath   = value;
                this.preDeleteCommandFilePath.BasePath  = value;
                this.postDeleteCommandFilePath.BasePath = value;
                this.preOpenCommandFilePath.BasePath    = value;
            }
        }

        /// <summary>
        /// プリセットデータフォルダパスを取得または設定します。
        /// </summary>
        public string PresetFolderPath
        {
            get { return this.presetFolderPath.ExpandedPath; }
            set { this.presetFolderPath.Path = value; }
        }

        /// <summary>
        /// 保存前コマンドパスを取得または設定します。
        /// </summary>
        public string PreSaveCommandFilePath
        {
            get { return this.preSaveCommandFilePath.ExpandedPath; }
            set { this.preSaveCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 保存後コマンドパスを取得または設定します。
        /// </summary>
        public string PostSaveCommandFilePath
        {
            get { return this.postSaveCommandFilePath.ExpandedPath; }
            set { this.postSaveCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 削除前コマンドパスを取得または設定します。
        /// </summary>
        public string PreDeleteCommandFilePath
        {
            get { return this.preDeleteCommandFilePath.ExpandedPath; }
            set { this.preDeleteCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 削除後コマンドパスを取得または設定します。
        /// </summary>
        public string PostDeleteCommandFilePath
        {
            get { return this.postDeleteCommandFilePath.ExpandedPath; }
            set { this.postDeleteCommandFilePath.Path = value; }
        }

        /// <summary>
        /// オープン前コマンドパスを取得または設定します。
        /// </summary>
        public string PreOpenCommandFilePath
        {
            get { return this.preOpenCommandFilePath.ExpandedPath; }
            set { this.preOpenCommandFilePath.SetPath(value); }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.presetFolderPath.Path          = this.ReadElement(context, "PresetFolderPath", this.PresetFolderPath).Trim();
            this.preSaveCommandFilePath.Path    = this.ReadElement(context, "PreSaveCommandFilePath", this.PreSaveCommandFilePath).Trim();
            this.postSaveCommandFilePath.Path   = this.ReadElement(context, "PostSaveCommandFilePath", this.PostSaveCommandFilePath).Trim();
            this.preDeleteCommandFilePath.Path  = this.ReadElement(context, "PreDeleteCommandFilePath", this.PreDeleteCommandFilePath).Trim();
            this.postDeleteCommandFilePath.Path = this.ReadElement(context, "PostDeleteCommandFilePath", this.PostDeleteCommandFilePath).Trim();
            this.preOpenCommandFilePath.Path    = this.ReadElement(context, "PreOpenCommandFilePath", this.PreOpenCommandFilePath).Trim();

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "PresetFolderPath", this.presetFolderPath.Path);
            this.WriteElement(context, "PreSaveCommandFilePath", this.preSaveCommandFilePath.Path);
            this.WriteElement(context, "PostSaveCommandFilePath", this.postSaveCommandFilePath.Path);
            this.WriteElement(context, "PreDeleteCommandFilePath", this.preDeleteCommandFilePath.Path);
            this.WriteElement(context, "PostDeleteCommandFilePath", this.postDeleteCommandFilePath.Path);
            this.WriteElement(context, "PreOpenCommandFilePath", this.preOpenCommandFilePath.Path);

            return true;
        }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            FileEventProjectConfig options = src as FileEventProjectConfig;

            if (options == null)
            {
                return false;
            }

            this.presetFolderPath.Set(options.presetFolderPath);
            this.preSaveCommandFilePath.Set(options.preSaveCommandFilePath);
            this.postSaveCommandFilePath.Set(options.postSaveCommandFilePath);
            this.preDeleteCommandFilePath.Set(options.preDeleteCommandFilePath);
            this.postDeleteCommandFilePath.Set(options.postDeleteCommandFilePath);
            this.preOpenCommandFilePath.Set(options.preOpenCommandFilePath);

            return true;
        }
    }
}
