﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// The base options class.
    /// </summary>
    public abstract class OptionsBase : ICloneable, ISettable, IXmlDocSerializable
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public OptionsBase()
        {
        }

        /// <summary>
        /// Copy the OptionsBase instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public OptionsBase(OptionsBase src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public virtual bool Set(object src)
        {
            PropertyInfo[] properties = this.GetType().GetProperties();

            foreach (PropertyInfo property in properties)
            {
                Type propertyType = property.PropertyType;

                if (propertyType.IsValueType || propertyType == typeof(string))
                {
                    // 値型またはstringをセット
                    property.SetValue(this, property.GetValue(src, null), null);
                }
                else
                {
                    // オブジェクト型をセット
                    ISettable settable = property.GetValue(this, null) as ISettable;

                    if (settable != null)
                    {
                        settable.Set(property.GetValue(src, null));
                    }
                    else
                    {
                        Logger.Log(
                            LogLevels.Warning,
                            "OptionBase.Set : Failed to set the property.");
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// Clone this options and return a new instance.
        /// </summary>
        /// <returns>The created instance.</returns>
        public virtual object Clone()
        {
            var newInstance = Activator.CreateInstance(
                this.GetType(), new object[] { this }) as OptionsBase;

            return newInstance;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public abstract bool ReadXml(XmlDocSerializationContext context);

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public abstract bool WriteXml(XmlDocSerializationContext context);
    }
}
