﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.RuntimeOptions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Store all the different option root instances.
    /// </summary>
    public static class OptionStore
    {
        /// <summary>
        /// The relative filename of the root options.
        /// </summary>
        public const string RootOptionsRelativeFilename = "options.xml";

        /// <summary>
        /// The relative filename of the project configure.
        /// </summary>
        public const string ProjectConfigRelativeFilename = "prjconfig.xml";

        /// <summary>
        /// Static initialization of the RootOptions class.
        /// </summary>
        static OptionStore()
        {
            RootOptions = new RootOptions();
            RuntimeOptions = new RuntimeOptions.RuntimeOptions();
            ProjectConfig = new EffectMakerProjectConfig();
        }

        /// <summary>
        /// Raised when the option changed.
        /// </summary>
        public static event EventHandler OptionChanged;

        /// <summary>
        /// Gets the main RootOptions instance.
        /// </summary>
        public static RootOptions RootOptions { get; set; }

        /// <summary>
        /// 実行時オプションを取得または設定します。
        /// </summary>
        public static RuntimeOptions.RuntimeOptions RuntimeOptions { get; set; }

        /// <summary>
        /// Gets the EffectMakerProjectConfig instance.
        /// </summary>
        public static EffectMakerProjectConfig ProjectConfig { get; set; }

        /// <summary>
        /// Trigger the option changed event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public static void TriggerOptionChangedEvent(EventArgs e)
        {
            if (OptionChanged != null)
            {
                OptionChanged(RootOptions, e);
            }
        }

        /// <summary>
        /// Loads the root options.
        /// </summary>
        public static void LoadRootOptions()
        {
            string path;
            if (Path.IsPathRooted(RootOptionsRelativeFilename))
            {
                path = RootOptionsRelativeFilename;
            }
            else
            {
                path = Path.GetFullPath(Path.Combine(
                    EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath,
                    RootOptionsRelativeFilename));
            }

            if (File.Exists(path) == true)
            {
                using (var stream = new FileStream(path, FileMode.Open, FileAccess.Read))
                {
                    RootOptions = SerializationHelper.LoadXmlDocSerializable<RootOptions>(stream);
                }
            }

            TriggerOptionChangedEvent(null);
        }

        /// <summary>
        /// Saves the root options.
        /// </summary>
        public static void SaveRootOptions()
        {
            string path;
            if (Path.IsPathRooted(RootOptionsRelativeFilename))
            {
                path = RootOptionsRelativeFilename;
            }
            else
            {
                path = Path.GetFullPath(Path.Combine(
                    EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath,
                    RootOptionsRelativeFilename));
            }

            try
            {
                if (File.Exists(path) == true)
                {
                    File.Copy(path, string.Format("{0}.backup", path), true);
                }
            }
            catch
            {
                // Failed accessing the option file, just skip saving the options.
                // (another EffectMaker instance is accessing it)
            }

            try
            {
                using (var stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    SerializationHelper.SaveXmlDocSerializable(RootOptions, stream);
                }
            }
            catch
            {
                // Failed accessing the option file, just skip saving the options.
                // (another EffectMaker instance is accessing it)
            }
        }

        /// <summary>
        /// Initialize the project configure.
        /// </summary>
        public static void InitializeProjectConfig()
        {
            ProjectConfig.Initialize();

            // Load custom shader and custom action.
            UserDataHelper.Instance.LoadAndSetupCustomShaderAndAction(
                ProjectConfig.CustomShaderPath,
                ProjectConfig.CustomActionPath);

            UserDataHelper.Instance.LoadEmitterExtParamsUserData(
                ProjectConfig.EmitterExtParamsPath, true);
        }

        /// <summary>
        /// Loads the project configure.
        /// </summary>
        /// <param name="path">
        /// File path
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool LoadProjectConfig(string path)
        {
            bool result = false;

            // パスをフルパスに変換
            string fullPath;
            if (Path.IsPathRooted(path))
            {
                fullPath = path;
            }
            else
            {
                fullPath = Path.GetFullPath(Path.Combine(
                    EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath,
                    path));
            }

            // プロジェクト設定を読み込む
            EffectMakerProjectConfig prjConfig = EffectMakerProjectConfig.Load(fullPath);
            if (prjConfig != null)
            {
                ProjectConfig = prjConfig;
                OptionStore.RootOptions.FileEvent.UserSettingsFilePath = fullPath;
                result = true;
            }
            else
            {
                ProjectConfig.Initialize();
            }

            // Load custom shader and custom action.
            UserDataHelper.Instance.LoadAndSetupCustomShaderAndAction(
                ProjectConfig.CustomShaderPath,
                ProjectConfig.CustomActionPath);

            UserDataHelper.Instance.LoadEmitterExtParamsUserData(
                ProjectConfig.EmitterExtParamsPath, true);

            return result;
        }

        /// <summary>
        /// Saves the project configure.
        /// </summary>
        /// <param name="path">File path</param>
        public static void SaveProjectConfig(string path)
        {
            string fullPath;
            if (Path.IsPathRooted(path))
            {
                fullPath = path;
            }
            else
            {
                fullPath = Path.GetFullPath(Path.Combine(
                    EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath,
                    path));
            }

            try
            {
                if (File.Exists(fullPath) == true)
                {
                    File.Copy(fullPath, string.Format("{0}.backup", fullPath), true);
                }
            }
            catch
            {
                // Failed accessing the file, just skip saving it.
                // (another EffectMaker instance is accessing it)
            }

            try
            {
                using (var stream = new FileStream(fullPath, FileMode.Create, FileAccess.Write))
                {
                    SerializationHelper.SaveXmlDocSerializable(ProjectConfig, stream);
                }
            }
            catch
            {
                // Failed accessing the file, just skip saving it.
                // (another EffectMaker instance is accessing it)
            }
        }

        /// <summary>
        /// Load an options class based on the relative filename.
        /// The filename is relative to the application executable path.
        /// </summary>
        /// <typeparam name="T">The type of data class to load.</typeparam>
        /// <param name="relativeFilename">The relative filename.
        /// The filename is relative to the application executable path.</param>
        /// <returns>Returns the loaded class, or a newly created
        /// class if the file is not available.</returns>
        public static T Load<T>(string relativeFilename) where T : class, new()
        {
            string filename;

            if (Path.IsPathRooted(relativeFilename))
            {
                filename = relativeFilename;
            }
            else
            {
                string applicationPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                filename = string.Format("{0}\\{1}", applicationPath, relativeFilename);
            }

            if (File.Exists(filename))
            {
                using (var stream = File.OpenRead(filename))
                {
                    var options = SerializationHelper.Load<T>(stream);
                    if (options != null)
                    {
                        return options;
                    }
                    else
                    {
                        Logger.Log(LogLevels.Warning, "Serialize error: {0}", filename);
                    }
                }
            }

            return new T();
        }

        /// <summary>
        /// Save a given set of options to a given file.
        /// </summary>
        /// <typeparam name="T">Type of option instance to save.</typeparam>
        /// <param name="relativeFilename">The file to save to,
        /// relative to the application root folder.</param>
        /// <param name="instance">The instance to save.</param>
        public static void Save<T>(string relativeFilename, T instance)
        {
            string filename;

            if (Path.IsPathRooted(relativeFilename))
            {
                filename = relativeFilename;
            }
            else
            {
                string applicationPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                filename = string.Format("{0}\\{1}", applicationPath, relativeFilename);
            }

            try
            {
                if (File.Exists(filename) == true)
                {
                    File.Copy(filename, string.Format("{0}.backup", filename), true);
                }
            }
            catch
            {
                // Failed accessing the file, just skip saving it.
                // (another EffectMaker instance is accessing it)
            }

            try
            {
                using (var stream = new FileStream(
                    filename,
                    FileMode.Create,
                    FileAccess.Write,
                    FileShare.ReadWrite))
                {
                    SerializationHelper.Save(instance, stream);
                }
            }
            catch
            {
                // Failed accessing the file, just skip saving it.
                // (another EffectMaker instance is accessing it)
            }
        }
    }
}
