﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// ベースパスを持ち環境変数にも対応した展開可能なファイルパスです。
    /// </summary>
    public class ExpandablePath
    {
        /// <summary>
        /// 展開前のパスです。
        /// </summary>
        private string path;

        /// <summary>
        /// ベースパスです。
        /// </summary>
        private string basePath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ExpandablePath()
        {
            this.ForceSetPath(string.Empty, null);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="path">パス</param>
        /// <param name="basePath">ベースパス</param>
        public ExpandablePath(string path, string basePath = null)
        {
            this.ForceSetPath(path, basePath);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public ExpandablePath(ExpandablePath obj)
        {
            this.Set(obj);
        }

        /// <summary>
        /// 展開前のパスを取得します。
        /// </summary>
        public string Path
        {
            get { return this.path; }
            set { this.SetPath(value, this.BasePath); }
        }

        /// <summary>
        /// ベースパスを取得または設定します。
        /// </summary>
        public string BasePath
        {
            get { return this.basePath; }
            set { this.ForceSetPath(this.Path, value); }
        }

        /// <summary>
        /// 展開後のパスを取得します。
        /// </summary>
        public string ExpandedPath { get; private set; }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="obj">値</param>
        public void Set(ExpandablePath obj)
        {
            this.path = obj.path;
            this.basePath = obj.basePath;
            this.ExpandedPath = obj.ExpandedPath;
        }

        /// <summary>
        /// パスを設定します。
        /// 展開後のパスが一致するとき、展開前のパスが異なっても内容を更新しません。
        /// </summary>
        /// <param name="path">パス</param>
        /// <param name="basePath">ベースパス</param>
        public void SetPath(string path, string basePath = null)
        {
            string expandedPath = PathUtility.ToAbsolutePath(path, basePath);

            if (string.IsNullOrEmpty(basePath))
            {
                expandedPath = path;
            }

            if (expandedPath != this.ExpandedPath)
            {
                this.path = path;
                this.basePath = basePath;
                this.ExpandedPath = expandedPath;
            }
        }

        /// <summary>
        /// パスを設定します。
        /// 展開後のパスが一致するときも強制的に内容を更新します。
        /// </summary>
        /// <param name="path">パス</param>
        /// <param name="basePath">ベースパス</param>
        public void ForceSetPath(string path, string basePath = null)
        {
            string expandedPath = PathUtility.ToAbsolutePath(path, basePath);

            if (string.IsNullOrEmpty(basePath))
            {
                expandedPath = path;
            }

            this.path = path;
            this.basePath = basePath;
            this.ExpandedPath = expandedPath;
        }
    }
}
