﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.BusinessLogic.Manager
{
    /// <summary>
    /// テクスチャマネージャ
    /// </summary>
    public class TextureManager
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static TextureManager singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static TextureManager Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new TextureManager();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// テクスチャマネージャの実体です。
        /// </summary>
        private readonly EffectMaker.TextureManager.TextureManager textureManager = new EffectMaker.TextureManager.TextureManager();

        /// <summary>
        /// サブバイナリコンバート時にメイン側が持つテクスチャのGUIDのテーブル
        /// </summary>
        private readonly Dictionary<string, ulong?> sharedGuidTable = new Dictionary<string, ulong?>();

        /// <summary>
        /// 発行したハッシュが衝突した際のインクリメントテーブル
        /// </summary>
        private readonly Dictionary<int, uint> hashCollisionCounter = new Dictionary<int, uint>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private TextureManager()
        {
        }

        /// <summary>
        /// 無効なテクスチャを表す定数値
        /// </summary>
        public const ulong InvalidTextureId = 0xFFFFFFFFFFFFFFFF;

        /// <summary>
        /// ファイルリロード後に発生するイベントです。
        /// </summary>
        public event EventHandler<FileReloadedEventArgs> FileReloaded
        {
            add { this.textureManager.FileReloaded += value; }
            remove { this.textureManager.FileReloaded -= value; }
        }


        /// <summary>
        /// バイナリコンバート時にテクスチャに割り当てたGUIDのテーブルを取得します。
        /// </summary>
        public Dictionary<string, ulong?> GuidTable { get; } = new Dictionary<string, ulong?>();

        /// <summary>
        /// サブバイナリコンバート時にメイン側が持つテクスチャのGUIDのテーブルを取得または設定します。
        /// </summary>
        public Dictionary<string, ulong?> SharedGuidTable
        {
            get
            {
                return this.sharedGuidTable;
            }

            set
            {
                this.sharedGuidTable.Clear();
                if (value != null)
                {
                    foreach (var pair in value)
                    {
                        this.sharedGuidTable.Add(pair.Key, pair.Value);
                    }
                }
            }
        }

        /// <summary>
        /// ファイルが読み込めるか調査する
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>読み込めるかどうか？</returns>
        public bool CanLoad(string filePath)
        {
            return this.textureManager.CanLoad(filePath);
        }

        /// <summary>
        /// キャッシュされているか
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>キャッシュされているか.</returns>
        public bool IsCached(string filePath)
        {
            return this.textureManager.IsCached(filePath);
        }

        /// <summary>
        /// キャッシュサイズを設定する
        /// </summary>
        /// <param name="maxCapacity">キャッシュサイズの最大値(Byte単位)</param>
        public void SetMaxCapacity(int maxCapacity)
        {
            this.textureManager.SetMaxCapacity(maxCapacity);
        }

        /// <summary>
        /// テクスチャファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="enableCache">キャッシュからロードするかどうか</param>
        /// <returns>読み込み結果を返します。</returns>
        public LoadTextureResult LoadTexture(string filePath, bool enableCache)
        {
            LoadTextureResult result = this.textureManager.LoadTexture(filePath, enableCache);

            return result;
        }

        /// <summary>
        /// キャッシュをフラッシュする
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void FlushCache(string filePath)
        {
            this.textureManager.FlushCache(filePath);
        }

        /// <summary>
        /// リロードする
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void Reload(string filePath)
        {
            this.textureManager.Reload(filePath);
        }

        /// <summary>
        /// 事前JITする
        /// </summary>
        /// <param name="textureData">テクスチャデータ</param>
        /// <param name="extName">テクスチャファイル拡張子</param>
        public void PreJit(byte[] textureData, string extName)
        {
            this.textureManager.PreJit(textureData, extName);
        }

        /// <summary>
        /// GUIDテーブルをクリアします。
        /// </summary>
        public void ClearGuidTable()
        {
            this.GuidTable.Clear();
            this.hashCollisionCounter.Clear();
        }

        /// <summary>
        /// sharedGuidTableのguidを追加する
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void AddSharedGuidTable(string filePath)
        {
            string keyName = Path.GetFileNameWithoutExtension(filePath).ToLower();
            ulong? guid;

            this.sharedGuidTable.TryGetValue(keyName, out guid);

            if (guid == null)
            {
                int hash = keyName.GetHashCode();
                if (this.hashCollisionCounter.ContainsKey(hash))
                {
                    this.hashCollisionCounter[hash]++;
                }
                else
                {
                    this.hashCollisionCounter.Add(hash, 0);
                }

                // 下位32ビットをファイル名から生成したハッシュコード
                // 上位32ビットを衝突カウンタの値として完全にユニークなIDを生成
                guid = BitConverter.ToUInt32(BitConverter.GetBytes(hash), 0) | (this.hashCollisionCounter[hash] << 32);

                this.sharedGuidTable.Add(keyName, guid);
            }
        }

        /// <summary>
        /// テクスチャのGUIDを取得します。
        /// </summary>
        /// <param name="filePath">テクスチャのファイルパス</param>
        /// <returns>テクスチャのGUIDを返します。</returns>
        public ulong GetGuid(string filePath)
        {
            // テクスチャをロードしてみて失敗したらエラー値を返す
            LoadTextureResult result = LoadTexture(filePath, true);

            if (string.IsNullOrEmpty(filePath) || result.ResultCode != LoadTextureResultCode.Success)
            {
                return InvalidTextureId;
            }

            // テーブルからGUIDを取得
            string keyName = Path.GetFileNameWithoutExtension(filePath).ToLower();
            ulong? guid;
            this.GuidTable.TryGetValue(keyName, out guid);

            // サブバイナリコンバート中なら、メイン側テーブルも調べる
            if (OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                if (guid == null)
                {
                    this.sharedGuidTable.TryGetValue(keyName, out guid);
                }
            }

            // GUIDが登録されていなければ新しく追加
            if (guid == null)
            {
                int hash = keyName.GetHashCode();
                if (this.hashCollisionCounter.ContainsKey(hash))
                {
                    this.hashCollisionCounter[hash]++;
                }
                else
                {
                    this.hashCollisionCounter.Add(hash, 0);
                }

                // 下位32ビットをファイル名から生成したハッシュコード
                // 上位32ビットを衝突カウンタの値として完全にユニークなIDを生成
                guid = BitConverter.ToUInt32(BitConverter.GetBytes(hash), 0) | (this.hashCollisionCounter[hash] << 32);

                this.GuidTable.Add(keyName, guid);
            }

            return (ulong)guid;
        }

        /// <summary>
        /// 指定したGUIDのテクスチャが他のバイナリにあるかどうかを調べます。
        /// </summary>
        /// <param name="guid">テクスチャGUID</param>
        /// <returns>メイン側のテーブルに見つかればtrue, そうでなければfalse.</returns>
        public bool IsLocatedOnAnotherBinary(ulong guid)
        {
            if (!OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                return false;
            }

            return this.sharedGuidTable.ContainsValue(guid);
        }

        /// <summary>
        /// アプリケーションの起動時に、初期化を行うメソッド
        /// </summary>
        public static void InitializeOnStartUp()
        {
            EffectMaker.TextureManager.TextureManager.InitializeOnStartUp(
                new[]
                {
                    Path.Combine(IOConstants.ToolsDirectoryPath, @"Graphics\3dTools\"),
                    Path.Combine(IOConstants.ToolsDirectoryPath, @"Graphics\GraphicsTools\"),
                    Path.Combine(IOConstants.CoreModulesPath, @".\"),
                });
        }

        /// <summary>
        /// 読み込んだテクスチャファイルの変更日時を取得します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <returns>
        /// 読み込んだテクスチャファイルの変更日時を返します。
        /// テクスチャファイルが読み込まれていないときはデフォルト値を返します。
        /// </returns>
        public DateTime GetLoadefFileModifyTime(string filePath)
        {
            return this.textureManager.GetLoadedFileModifyTime(filePath);
        }
    }
}
