﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.BusinessLogic.Manager
{
    /// <summary>
    /// プリミティブマネージャクラス.
    /// </summary>
    public class PrimitiveManager
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static PrimitiveManager singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static PrimitiveManager Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new PrimitiveManager();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// マネージャの実体.
        /// </summary>
        private readonly EffectMaker.PrimitiveManager.PrimitiveManager primitiveManager =
            new EffectMaker.PrimitiveManager.PrimitiveManager();

        /// <summary>
        /// サブバイナリコンバート時にメイン側が持つエミッタ形状primitiveのGUIDのテーブル
        /// </summary>
        private readonly Dictionary<string, ulong?> sharedOriginalPrimitiveGuidTable = new Dictionary<string, ulong?>();

        /// <summary>
        /// サブバイナリコンバート時にメイン側が持つパーティクル形状プリミティブのGUIDのテーブル
        /// </summary>
        private readonly Dictionary<string, ulong?> sharedG3dPrimitiveGuidTable = new Dictionary<string, ulong?>();

        /// <summary>
        /// バイナリコンバート時にテクスチャに割り当てるGUIDのテーブル
        /// </summary>
        private readonly Dictionary<string, ulong?> guidTable = new Dictionary<string, ulong?>();

        /// <summary>
        /// 発行したハッシュが衝突した際のインクリメントテーブル
        /// </summary>
        private readonly Dictionary<int, uint> hashCollisionCounter = new Dictionary<int, uint>();

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        private PrimitiveManager()
        {
            this.primitiveManager.FileReloaded += (s, e) =>
            {
                if (this.FileReloaded != null)
                {
                    this.FileReloaded(this, e);
                }
            };

            SpecManager.CurrentSpecChanged += (sender, args) =>
            {
                this.primitiveManager.FlushTrimmingPrimitiveCache();
            };
        }

        /// <summary>
        /// 無効なプリミティブを表す定数値
        /// </summary>
        public const ulong InvalidPrimitiveId = 0xFFFFFFFFFFFFFFFF;

        /// <summary>
        /// ファイルリロード後イベント.
        /// </summary>
        public event EventHandler<FileReloadedEventArgs> FileReloaded;

        /// <summary>
        /// アプリケーションのフォルダパスを取得または設定します.
        /// </summary>
        public string AppFolderPath
        {
            get
            {
                return this.primitiveManager.AppFolderPath;
            }

            set
            {
                this.primitiveManager.AppFolderPath = value;
            }
        }

        /// <summary>
        /// バイナリコンバート時にプリミティブに割り当てるGUIDのテーブルを取得します。
        /// </summary>
        public Dictionary<string, ulong?> GuidTable
        {
            get { return this.guidTable; }
        }

        /// <summary>
        /// サブバイナリコンバート時に、メイン側が持つエミッタ形状プリミティブのGUIDのテーブルを取得または設定します。
        /// </summary>
        public Dictionary<string, ulong?> SharedOriginalPrimitiveGuidTable
        {
            get
            {
                return this.sharedOriginalPrimitiveGuidTable;
            }

            set
            {
                this.sharedOriginalPrimitiveGuidTable.Clear();
                if ( value != null )
                {
                    foreach (var pair in value)
                    {
                        this.sharedOriginalPrimitiveGuidTable.Add(pair.Key, pair.Value);
                    }
                }
            }
        }

        /// <summary>
        /// サブバイナリコンバート時に、メイン側が持つパーティクル形状プリミティブのGUIDのテーブルを取得または設定します。
        /// </summary>
        public Dictionary<string, ulong?> SharedG3dPrimitiveGuidTable
        {
            get
            {
                return this.sharedG3dPrimitiveGuidTable;
            }

            set
            {
                this.sharedG3dPrimitiveGuidTable.Clear();
                if ( value != null )
                {
                    foreach (var pair in value)
                    {
                        this.sharedG3dPrimitiveGuidTable.Add(pair.Key, pair.Value);
                    }
                }
            }
        }

        /// <summary>
        /// ファイルが読み込めるかどうか調査する.
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>ファイルが読み込める場合には true を返却.</returns>
        public bool CanLoad(string filePath)
        {
            return this.primitiveManager.CanLoad(filePath);
        }

        /// <summary>
        /// 同期でモデルを読み込みます.
        /// </summary>
        /// <param name="filePath">ファイルパス.</param>
        /// <param name="enableCache">キャッシュからロードするか?</param>
        /// <param name="modelData">ロードされたモデルデータ.</param>
        /// <returns>読み込み結果.</returns>
        public LoadModelResults LoadModelWithData(
            string filePath,
            bool enableCache,
            out ModelData modelData)
        {
#if TEST_TRIMMING_PRIMITIVE
            var result = this.primitiveManager.CreateTrimmingPrimitive(
                "Test.ftxb",
                128,
                128,
                21,
                1,
                86,
                126);
            modelData = result.ModelData;
            return result.Result;
#else
            var data = this.primitiveManager.LoadModel(filePath, enableCache);
            modelData = data.ModelData;

            return data.Result;
#endif
        }

        /// <summary>
        /// 同期でモデルを読み込みます.
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>読み込み結果と読み込んだモデルデータ.</returns>
        public LoadModelResultWithData LoadModel(string filePath)
        {
#if TEST_TRIMMING_PRIMITIVE
            return this.primitiveManager.CreateTrimmingPrimitive(
                "Test.ftxb",
                128,
                128,
                21,
                1,
                86,
                126);
#else
            return this.primitiveManager.LoadModel(filePath);
#endif
        }

        /// <summary>
        /// fmdbファイルを読み込んでbfresバイナリにコンバートし、ストリームに出力します。
        /// </summary>
        /// <param name="filePaths">fmdbのファイルパスのリスト</param>
        /// <param name="outStream">出力するストリーム</param>
        /// <returns>コンバート結果を返します。</returns>
        public LoadModelResultWithFilePath LoadFmdbModels(IEnumerable<string> filePaths, Stream outStream)
        {
            return this.primitiveManager.LoadFmdbModels(filePaths, outStream);
        }

        /// <summary>
        /// コメントを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>読み込み結果</returns>
        public LoadComment LoadComment(string filePath)
        {
            return this.primitiveManager.LoadComment(filePath);
        }

        /// <summary>
        /// コメントを上書き保存します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="comment">コメント</param>
        /// <param name="color">カラー</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        public bool SaveComment(string filePath, string comment, string color)
        {
            return this.primitiveManager.SaveComment(filePath, comment, color);
        }

        /// <summary>
        /// キャッシュをフラッシュする.
        /// </summary>
        /// <param name="filePath">ファイルパス.</param>
        public void FlushCache(string filePath)
        {
            this.primitiveManager.FlushCache(filePath);
        }

        /// <summary>
        /// リロードする.
        /// </summary>
        /// <param name="filePath">ファイルパス.</param>
        public void Reload(string filePath)
        {
            this.primitiveManager.Reload(filePath);
        }

        /// <summary>
        /// トリミングプリミティブを作成します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <returns>読み込み結果.</returns>
        public LoadModelResultWithData CreateTrimmingPrimitive(string filePath)
        {
            // テクスチャを取得
            LoadTextureResult texResult = TextureManager.Instance.LoadTexture(filePath, true);

            TextureData textureData = texResult.TextureData;

            // テクスチャが見つからないとき、エラー
            if (texResult.ResultCode != LoadTextureResultCode.Success)
            {
                var result = new LoadModelResultWithData()
                {
                    Result = LoadModelResults.FileNotFound
                };

                return result;
            }

            bool inverse = false;
            if (SpecManager.CurrentSpec.FrontFace != "CounterClockwise")
            {
                inverse = true;
            }

            // プリミティブを作成
            return this.primitiveManager.CreateTrimmingPrimitive(
                filePath,
                textureData.Width,
                textureData.Height,
                textureData.TrimRect.X,
                textureData.TrimRect.Y,
                textureData.TrimRect.Width,
                textureData.TrimRect.Height,
                inverse);
        }

        /// <summary>
        /// トリミングプリミティブを作成します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <param name="enableCache">キャッシュからロードするか</param>
        /// <param name="modelData">ロードされたモデルデータ</param>
        /// <returns>読み込み結果.</returns>
        public LoadModelResults CreateTrimmingPrimitiveWithData(
            string filePath,
            bool enableCache,
            out ModelData modelData)
        {
            // テクスチャを取得
            LoadTextureResult texResult = TextureManager.Instance.LoadTexture(filePath, true);

            TextureData textureData = texResult.TextureData;

            // テクスチャが見つからないとき、エラー
            if (texResult.ResultCode != LoadTextureResultCode.Success)
            {
                modelData = null;
                return LoadModelResults.FileNotFound;
            }

            bool inverse = false;
            if (SpecManager.CurrentSpec.FrontFace != "CounterClockwise")
            {
                inverse = true;
            }

            // プリミティブを作成
            LoadModelResultWithData result = this.primitiveManager.CreateTrimmingPrimitive(
                filePath,
                enableCache,
                textureData.Width,
                textureData.Height,
                textureData.TrimRect.X,
                textureData.TrimRect.Y,
                textureData.TrimRect.Width,
                textureData.TrimRect.Height,
                inverse);

            modelData = result.ModelData;
            return result.Result;
        }

        /// <summary>
        /// 中間ファイルをバイナリファイルに変換します.
        /// </summary>
        /// <param name="modelPath">モデルファイルパス</param>
        /// <param name="animationPaths">アニメーションファイルパス</param>
        /// <param name="outStream">出力ストリーム</param>
        /// <returns>成功したときtrueを返します.</returns>
        public LoadModelResultWithData ConvertBinary(
            string modelPath,
            string[] animationPaths,
            Stream outStream)
        {
            this.primitiveManager.BinaryMode = SpecManager.CurrentSpec.TexturePackingOption != null
                ? EffectMaker.PrimitiveManager.PrimitiveManager.G3DBinaryMode.NnBinary
                : EffectMaker.PrimitiveManager.PrimitiveManager.G3DBinaryMode.NwBinary;
            string markerPath = Path.Combine(IOConstants.ExecutableFolderPath, "MarkerCreateOldBinary");

            if (this.primitiveManager.BinaryMode ==
                EffectMaker.PrimitiveManager.PrimitiveManager.G3DBinaryMode.NnBinary)
            {
                this.primitiveManager.ConvertBntx = (texPaths) =>
                {
                    string outputPath = Path.Combine(
                        TextureConverterExecuter.Instance.IntermediateDirectoryPath,
                        "model_output.bntx");

                    var param = new TextureConverterExecuter.ConvertParameter()
                    {
                        OutputPath = outputPath,
                        TileMode   = SpecManager.CurrentSpec.TexturePackingOption.TileMode,
                        FilePaths  = texPaths
                    };

                    var resConvert = TextureConverterExecuter.Instance.Execute(param);

                    if (resConvert.ResultCode != 0)
                    {
                        return null;
                    }

                    return outputPath;
                };

                if (File.Exists(markerPath))
                {
                    try
                    {
                        File.Delete(markerPath);
                    }
                    catch
                    {
                        //
                    }
                }
            }
            else
            {
                this.primitiveManager.ConvertBntx = null;
                if (!File.Exists(markerPath))
                {
                    try
                    {
                        File.WriteAllText(markerPath, string.Empty);
                    }
                    catch
                    {
                        //
                    }
                }

            }

            return this.primitiveManager.ConvertBinary(modelPath, animationPaths, outStream);
        }

        /// <summary>
        /// 事前JITする
        /// </summary>
        /// <param name="modelData">モデルデータ</param>
        /// <param name="extName">モデルファイル拡張子</param>
        public void PreJit(byte[] modelData, string extName)
        {
            this.primitiveManager.PreJit(modelData, extName);
        }

        /// <summary>
        /// GUIDテーブルをクリアします。
        /// </summary>
        public void ClearGuidTable()
        {
            this.guidTable.Clear();
            this.hashCollisionCounter.Clear();
        }

        /// <summary>
        /// プリミティブのGUIDを取得します。
        /// </summary>
        /// <param name="filePath">プリミティブのファイルパス</param>
        /// <returns>プリミティブのGUIDを返します。</returns>
        public ulong GetGuid(string filePath)
        {
            // プリミティブをロードしてみて失敗したらエラー値を返す
            ModelData data;
            var result = LoadModelWithData(filePath, true, out data);

            if (string.IsNullOrEmpty(filePath) || result != LoadModelResults.Success)
            {
                result = CreateTrimmingPrimitiveWithData(filePath, true, out data);
                if (string.IsNullOrEmpty(filePath) || result != LoadModelResults.Success)
                {
                    return InvalidPrimitiveId;
                }
            }

            string keyName = filePath.ToLower();

            // トリミングプリミティブのときは拡張子が.ftxbになるので
            // プリミティブファイルと被らないようにキーの拡張子を残しておく
            if (Path.GetExtension(keyName) == ".ftxb")
            {
                keyName = Path.GetFileName(keyName);
            }
            else {
                keyName = Path.GetFileNameWithoutExtension(keyName);
            }

            // テーブルからGUIDを取得
            ulong? guid;
            this.guidTable.TryGetValue(keyName, out guid);

            // サブバイナリコンバート中なら、メイン側テーブルも調べる
            if (OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                if (guid == null)
                {
                    this.sharedG3dPrimitiveGuidTable.TryGetValue(keyName, out guid);
                    if ( guid == null )
                    {
                        this.sharedOriginalPrimitiveGuidTable.TryGetValue(keyName, out guid);
                    }
                }
            }

            // GUIDが登録されていなければ新しく追加
            if (guid == null)
            {
                int hash = keyName.GetHashCode();
                if (this.hashCollisionCounter.ContainsKey(hash))
                {
                    this.hashCollisionCounter[hash]++;
                }
                else
                {
                    this.hashCollisionCounter.Add(hash, 0);
                }

                // 下位32ビットをファイル名から生成したハッシュコード
                // 上位32ビットを衝突カウンタの値として完全にユニークなIDを生成
                guid = BitConverter.ToUInt32(BitConverter.GetBytes(hash), 0) | (this.hashCollisionCounter[hash] << 32);

                this.guidTable.Add(keyName, guid);
            }

            return (ulong)guid;
        }

        /// <summary>
        /// 指定したGUIDのプリミティブが他のバイナリに、パーティクル形状向けとしてあるかどうかを調べます。
        /// </summary>
        /// <param name="guid"></param>
        /// <returns></returns>
        public bool IsLocatedOnAnotherBinaryAsParticlePrimitive(ulong guid)
        {
            if (!OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                return false;
            }

            if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
            {
                return this.sharedG3dPrimitiveGuidTable.ContainsValue(guid);
            }
            else
            {
                return this.sharedOriginalPrimitiveGuidTable.ContainsValue(guid);
            }
        }

        /// <summary>
        /// 指定したGUIDのプリミティブが他のバイナリに、オリジナルプリミティブとしてあるかどうかを調べます。
        /// </summary>
        /// <param name="guid">プリミティブGUID</param>
        /// <returns>メイン側のテーブルに見つかればtrue, そうでなければfalse.</returns>
        public bool IsLocatedOnAnotherBinaryAsOriginalPrimitive(ulong guid)
        {
            if (!OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                return false;
            }

            return this.sharedOriginalPrimitiveGuidTable.ContainsValue(guid);
        }

        /// <summary>
        /// 指定したGUIDのプリミティブが他のバイナリに、g3dプリミティブとしてあるかどうかを調べます。
        /// </summary>
        /// <param name="guid">プリミティブGUID</param>
        /// <returns>メイン側のテーブルに見つかればtrue, そうでなければfalse.</returns>
        public bool IsLocatedOnAnotherBinaryAsG3dPrimitive(ulong guid)
        {
            if (!OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                return false;
            }

            return this.sharedG3dPrimitiveGuidTable.ContainsValue(guid);
        }

        /// <summary>
        /// 読み込んだプリミティブファイルの変更日時を取得します。
        /// </summary>
        /// <param name="filePath">プリミティブファイルパス</param>
        /// <returns>
        /// 読み込んだプリミティブファイルの変更日時を返します。
        /// プリミティブファイルが読み込まれていないときはデフォルト値を返します。
        /// </returns>
        public DateTime GetLoadedFileModifyTime(string filePath)
        {
            return this.primitiveManager.GetLoadedFileModifyTime(filePath);
        }
    }
}
