﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml;
using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.IO
{
    /// <summary>
    /// IOConstants.
    /// </summary>
    public static class IOConstants
    {
        /// <summary>
        /// テクスチャアセットのフォルダ名です。
        /// </summary>
        public const string TextureFolderName = "Textures";

        /// <summary>
        /// プリミティブアセットのフォルダ名です。
        /// </summary>
        public const string PrimitiveFolderName = "Primitives";

        /// <summary>
        /// コンバイナシェーダアセットのフォルダ名です。
        /// </summary>
        public const string CombinerShaderFolderName = "Combiners";

        /// <summary>
        /// エミッタセットファイルの拡張子です。
        /// </summary>
        public const string EmitterSetFileExtension = ".eset";

        /// <summary>
        /// プレビューファイルの拡張子です。
        /// </summary>
        public const string PreviewFileExtension = ".prev";

        /// <summary>
        /// ワークスペースファイルの拡張子です。
        /// </summary>
        public const string WorkspaceFileExtension = ".work";

        /// <summary>
        /// PTCLバイナリファイルの拡張子です。
        /// </summary>
        public const string PtclBinaryFileExtension = ".ptcl";

        /// <summary>
        /// エミッタセットの最大数です。
        /// </summary>
        public const int MaxEmitterSetCount = 16;

        /// <summary>
        /// 1つのエミッタセット内での、エミッタの最大数です。
        /// </summary>
        public const int MaxEmitterCount = 16;

        /// <summary>
        /// ファイル名の最大文字数です。
        /// </summary>
        public const int MaxFileNameLength = 128;

        /// <summary>The application executable file path.</summary>
        private static string executableFilePath = null;

        /// <summary>The application executable folder path.</summary>
        private static string executableFolderPath = null;

        /// <summary>The application temporary folder path.</summary>
        private static string temporaryFolderPath = null;

        /// <summary>The directory path for outputting the converted obsolete user data.</summary>
        private static string obsoleteUserDataConvertFolderPath = null;

        /// <summary>The directory path of the default shader files.</summary>
        private static string defaultShaderFolderPath = null;

        /// <summary>The directory path for the shader compile error log files.</summary>
        private static string shaderErrorLogFolderPath = null;

        /// <summary>The directory path for the texture compile error log files.</summary>
        private static string textureErrorLogFolderPath = null;

        /// <summary>DLLを集約したフォルダのパスです。</summary>
        private static string coreModulesFolderPath = null;

        /// <summary>G3D XSDのベースパスです。</summary>
        private static string g3dXsdBasePath = null;

        /// <summary>
        /// ツールフォルダパスです。
        /// Visual Studioからの実行時とToolsフォルダ以下からの実行時両方に対応します。
        /// </summary>
        private static string toolsFolderPath = null;

        /// <summary>
        /// The flag indicating whether to enable detailed output mode
        /// for command line application.
        /// </summary>
        private static bool detailCommandLineOutputMode = false;

        /// <summary>
        /// アプリケーション起動単位のインスタンスを識別するGUIDです。
        /// </summary>
        private static Guid guidByAppInstance = Guid.NewGuid();

        /// <summary>
        /// Get or set the flag indicating whether to enable detailed output mode
        /// for command line application.
        /// </summary>
        public static bool DetailCommandLineOutputMode
        {
            get { return detailCommandLineOutputMode; }
            set { detailCommandLineOutputMode = value; }
        }

        /// <summary>
        /// Get the executable file path.
        /// </summary>
        public static string ExecutablePath
        {
            get
            {
                if (string.IsNullOrEmpty(executableFilePath) == true)
                {
                    var asm = Assembly.GetEntryAssembly();
                    executableFilePath = asm != null ?
                        Path.GetFullPath(asm.Location) :
                        Path.GetFullPath(Assembly.GetExecutingAssembly().Location);
                }

                return executableFilePath;
            }
        }

        /// <summary>
        /// Get the executable folder path.
        /// </summary>
        public static string ExecutableFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(executableFolderPath) == true)
                {
                    executableFolderPath = Path.GetDirectoryName(ExecutablePath);
                }

                return executableFolderPath;
            }
        }

        /// <summary>
        /// Get the temporary folder path.
        /// </summary>
        public static string TemporaryFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(temporaryFolderPath) == true)
                {
                    temporaryFolderPath = Path.Combine(ExecutableFolderPath, "temp");
                }

                return temporaryFolderPath;
            }
        }

        /// <summary>
        /// ProcessId のバッキングフィールドです。
        /// </summary>
        private static string processId;

        /// <summary>
        /// プロセス ID を取得します。
        /// </summary>
        public static string ProcessId
        {
            get
            {
                if (string.IsNullOrEmpty(processId))
                {
                    processId = System.Diagnostics.Process.GetCurrentProcess().Id.ToString();
                }

                return processId;
            }
        }

        /// <summary>
        /// AppDataPath のバッキングフィールドです。
        /// </summary>
        private static string appDataPath;

        /// <summary>
        /// EffectMaker のアプリケーションデータ配置フォルダパスを取得します。
        /// 例: C:\Users\UserName\AppData\Local\Nintendo\EffectMaker
        /// </summary>
        public static string AppDataPath
        {
            get
            {
                if (string.IsNullOrEmpty(appDataPath))
                {
                    string appDataRoot = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
                    appDataPath = Path.Combine(appDataRoot, "Nintendo", "EffectMaker");
                }

                return appDataPath;
            }
        }

        /// <summary>
        /// AppDataTempPath のバッキングフィールドです。
        /// </summary>
        private static string appDataTempPath;

        /// <summary>
        /// EffectMaker のテンポラリデータ配置フォルダパスを取得します。
        /// 例: C:\Users\UserName\AppData\Local\Temp\Nintendo_EffectMaker
        /// </summary>
        public static string AppDataTempPath
        {
            get
            {
                if (string.IsNullOrEmpty(appDataTempPath))
                {
                    string tempRoot = Path.GetTempPath();
                    appDataTempPath = Path.Combine(tempRoot, "Nintendo_EffectMaker");
                }

                return appDataTempPath;
            }
        }

        /// <summary>
        /// AppDataWorkPath のバッキングフィールドです。
        /// </summary>
        private static string appDataWorkPath;

        /// <summary>
        /// EffectMaker のテンポラリデータ配置フォルダパスを取得します。
        /// AppDataTempPath と違い、EffectMaker 内のフォルダを指します。
        /// 例: C:\Tools\NintendoSDK\Tools\Graphics\EffectMaker\Work
        /// </summary>
        public static string AppDataWorkPath
        {
            get
            {
                if (string.IsNullOrEmpty(appDataWorkPath))
                {
                    appDataWorkPath = Path.Combine(ExecutableFolderPath, "Work");
                }

                return appDataWorkPath;
            }
        }

        /// <summary>
        /// Get the folder path for outputting the converted obsolete user data.
        /// (.udd and .usd files)
        /// </summary>
        public static string ObsoleteUserDataConvertFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(obsoleteUserDataConvertFolderPath) == true)
                {
                    obsoleteUserDataConvertFolderPath = Path.Combine(
                        ExecutableFolderPath,
                        "Caches",
                        "ConvertedOldUserData");
                }

                return obsoleteUserDataConvertFolderPath;
            }
        }

        /// <summary>
        /// Get the directory path of the default shader files.
        /// </summary>
        public static string DefaultShaderFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(defaultShaderFolderPath) == true)
                {
                    defaultShaderFolderPath = Path.Combine(
                        ExecutableFolderPath,
                        "Converter",
                        "shader");
                }

                var subFolder = SpecDefinitions.SpecManager.CurrentSpec.ShaderConversionOption.CodeFolderName;
                return !string.IsNullOrEmpty(subFolder) ?
                    Path.Combine(defaultShaderFolderPath, subFolder) : defaultShaderFolderPath;
            }
        }

        /// <summary>
        /// Get the directory path for the shader compile error log files.
        /// </summary>
        public static string ShaderErrorLogFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(shaderErrorLogFolderPath) == true)
                {
                    shaderErrorLogFolderPath = Path.Combine(
                        ExecutableFolderPath,
                        "Converter",
                        "shader_error");
                }

                return shaderErrorLogFolderPath;
            }
        }

        /// <summary>
        /// Get the directory path for the texture compile error log files.
        /// </summary>
        public static string TextureErrorLogFolderPath
        {
            get
            {
                if (string.IsNullOrEmpty(textureErrorLogFolderPath) == true)
                {
                    textureErrorLogFolderPath = Path.Combine(
                        ExecutableFolderPath,
                        "Converter",
                        "texture_error");
                }

                return textureErrorLogFolderPath;
            }
        }

        /// <summary>
        /// DLLを集約したCoreModulesフォルダのパスを取得します。
        /// Visual Studio上から実行した場合はexeと同じパスになります。
        /// </summary>
        public static string CoreModulesPath
        {
            get
            {
                if (string.IsNullOrEmpty(coreModulesFolderPath))
                {
                    coreModulesFolderPath = Directory.Exists(Path.Combine(ExecutableFolderPath, "CoreModules")) ?
                        Path.Combine(ExecutableFolderPath, "CoreModules") :
                        ExecutableFolderPath;
                }

                return coreModulesFolderPath;
            }
        }

        /// <summary>
        /// G3D XSDのベースパスを取得します。
        /// </summary>
        public static string G3dXsdBasePath
        {
            get
            {
                // Tools/Graphics/3dTools/3dIntermediateFileXsd か、なければ
                // Tools/Graphics/EffectMaker/CoreModules/3dIntermediateFileXsd (Cafe用)
                if (string.IsNullOrEmpty(g3dXsdBasePath))
                {
                    g3dXsdBasePath = Path.Combine(ToolsDirectoryPath, @"Graphics\3dTools\3dIntermediateFileXsd\");

                    if (Directory.Exists(g3dXsdBasePath) == false)
                    {
                        g3dXsdBasePath = Path.Combine(CoreModulesPath, @"3dIntermediateFileXsd\");
                    }
                }

                return g3dXsdBasePath;
            }
        }

        /// <summary>
        /// ObsoleteUserDataConverterのDLLのパスを取得します。
        /// </summary>
        public static string ObsoleteUserDataConverterDllPath
        {
            get
            {
                return Path.Combine(CoreModulesPath, "ObsoleteUserDataConverter.dll");
            }
        }

        /// <summary>
        /// Udd2ConverterのDLLのパスを取得します。
        /// </summary>
        public static string Udd2ConverterDllPath
        {
            get
            {
                return Path.Combine(CoreModulesPath, "Udd2Converter.dll");
            }
        }

        /// <summary>
        /// Get the error log file path.
        /// </summary>
        public static string ApplicationErrorLogPath
        {
            get { return Path.Combine(ExecutableFolderPath, "error.log"); }
        }

        /// <summary>
        /// エミッタセットのダイアログフィルタを取得します。
        /// </summary>
        public static string EmitterSetDialogFilter
        {
            get { return Properties.Resources.IOFileFilterEmitterSet; }
        }

        /// <summary>
        /// ワークスペースのダイアログフィルタを取得します。
        /// </summary>
        public static string WorkspaceDialogFilter
        {
            get { return Properties.Resources.IOFileFilterWorkspace; }
        }

        /// <summary>
        /// プレビューのダイアログフィルタを取得します。
        /// </summary>
        public static string PreviewDialogFilter
        {
            get { return Properties.Resources.IOFileFilterPreview; }
        }

        /// <summary>
        /// テクスチャのダイアログフィルタを取得します。
        /// </summary>
        public static string TextureDialogFilter
        {
            get { return Properties.Resources.IOFileFilterTexture; }
        }

        /// <summary>
        /// モデルのダイアログフィルタを取得します。
        /// </summary>
        public static string ModelDialogFilter
        {
            get { return Properties.Resources.IOFileFilterModel; }
        }

        /// <summary>
        /// PTCLバイナリのダイアログフィルタを取得します。
        /// </summary>
        public static string PtclBinaryDialogFilter
        {
            get { return Properties.Resources.IOFileFilterPtclBinary; }
        }

        /// <summary>
        /// コンバイナプロジェクトのダイアログフィルタを取得します。
        /// </summary>
        public static string CombinerProjectDialogFilter
        {
            get { return Properties.Resources.IOFileFilterCombinerProject; }
        }

        /// <summary>
        /// ファイルを開くダイアログのフィルタを取得します。
        /// </summary>
        public static string OpenFileDialogFilter
        {
            get
            {
                StringBuilder sb = new StringBuilder(Properties.Resources.IOFileFilterAll);
                sb.Append("|").Append(Properties.Resources.IOFileFilterEmitterSet);
                sb.Append("|").Append(Properties.Resources.IOFileFilterWorkspace);
                sb.Append("|").Append(Properties.Resources.IOFileFilterPreview);

                return sb.ToString();
            }
        }

        /// <summary>
        /// Toolsフォルダパスを取得します。
        /// </summary>
        public static string ToolsDirectoryPath
        {
            get
            {
                if (!string.IsNullOrEmpty(toolsFolderPath))
                {
                    return toolsFolderPath;
                }

                const string toolsExeEndPath = @"\Graphics\EffectMaker";
                const string repositoryPath = @"\Programs\NintendoWare";

                // パッケージ or Toolsフォルダからの実行時
                toolsFolderPath = ExecutableFolderPath;
                if (toolsFolderPath.EndsWith(toolsExeEndPath))
                {
                    // "\Graphics\EffectMaker"をカットしたパスがToolsフォルダ
                    toolsFolderPath = toolsFolderPath.Substring(0, toolsFolderPath.Length - toolsExeEndPath.Length);
                    return toolsFolderPath;
                }

                // Visual Studioからの実行時
                while (!toolsFolderPath.EndsWith(repositoryPath))
                {
                    toolsFolderPath = Path.GetDirectoryName(toolsFolderPath);
                    if (string.IsNullOrEmpty(toolsFolderPath))
                    {
                        return IOConstants.ExecutableFolderPath;
                    }
                }

                // "\Programs\NintendoWare"をカットして、Toolsを付加したパスを返す
                toolsFolderPath = Path.Combine(
                    toolsFolderPath.Substring(0, toolsFolderPath.Length - repositoryPath.Length),
                    "Tools");
                return toolsFolderPath;
            }
        }

        /// <summary>
        /// アプリケーションインスタンスを識別するためのユニークなハッシュコードを取得します。
        /// </summary>
        public static string ApplicationInstanceHash
        {
            get { return guidByAppInstance.ToString().GetHashCode().ToString("X"); }
        }

        /// <summary>
        /// 指定されたXMLファイルが想定している内容か、ルート要素をチェックする
        /// </summary>
        /// <param name="filepath">XMLファイルのパス</param>
        /// <param name="rootName">想定したルート要素の名前</param>
        /// <returns>想定通りならtrue,そうじゃなかったらfalse</returns>
        public static bool CheckRootElement(string filepath, string rootName)
        {
            bool retVal = false;
            if (!File.Exists(filepath))
            {
                return false;
            }

            using (var fs = new FileStream(filepath, FileMode.Open, FileAccess.Read))
            {
                XmlReader reader = null;
                try
                {
                    reader = XmlReader.Create(
                        fs,
                        new XmlReaderSettings
                        {
                            IgnoreComments = true,
                            IgnoreWhitespace = true
                        });
                }
                catch
                {
                    reader = null;
                }

                if (reader == null)
                {
                    return false;
                }

                try
                {
                    while (reader.Read())
                    {
                        if (reader.IsEmptyElement)
                        {
                            continue;
                        }

                        if (reader.Name == rootName && reader.NodeType == XmlNodeType.Element)
                        {
                            retVal = true;
                            break;
                        }
                    }
                }
                catch
                {
                    return false;
                }
            }

            return retVal;
        }

        /// <summary>
        /// アセットのフォルダ名を取得します。
        /// </summary>
        /// <param name="type">アセットタイプ</param>
        /// <returns>アセットのフォルダ名を返します。</returns>
        public static string GetAssetFolderName(AssetResourceTypes type)
        {
            switch (type)
            {
                case AssetResourceTypes.Texture:
                    return TextureFolderName;

                case AssetResourceTypes.Primitive:
                    return PrimitiveFolderName;

                case AssetResourceTypes.CombinerShader:
                    return CombinerShaderFolderName;
            }

            return string.Empty;
        }

        /// <summary>
        /// ファイルタイプに対応したフィルタを取得します。
        /// </summary>
        /// <param name="fileType">ファイルタイプ</param>
        /// <returns>ファイルタイプに対応したフィルタを返します。</returns>
        public static string GetFilter(LastAccessDirectoryTypes fileType)
        {
            switch (fileType)
            {
                case LastAccessDirectoryTypes.Any:
                    return OpenFileDialogFilter;

                case LastAccessDirectoryTypes.EmitterSet:
                    return EmitterSetDialogFilter;

                case LastAccessDirectoryTypes.Preview:
                    return PreviewDialogFilter;

                case LastAccessDirectoryTypes.Texture:
                    return TextureDialogFilter;

                case LastAccessDirectoryTypes.Model:
                    return ModelDialogFilter;

                case LastAccessDirectoryTypes.Workspace:
                    return WorkspaceDialogFilter;

                case LastAccessDirectoryTypes.PtclBinary:
                    return PtclBinaryDialogFilter;

                case LastAccessDirectoryTypes.CombinerProject:
                    return CombinerProjectDialogFilter;
            }

            Logger.Log(LogLevels.Warning, "IOConstants.GetFilter : Invalid file type.");

            return null;
        }

        /// <summary>
        /// コピー先のパスが存在しないディレクトリを指していたら作成する。
        /// （大文字小文字違いの同名で存在する場合には作成しないようにする）
        /// </summary>
        public static void AdjustPathCase(FileCopyInfo copyInfo)
        {
            // コピー先のディレクトリパスを取得
            string directoryPath = Path.GetDirectoryName(copyInfo.DstPath);
            if (string.IsNullOrEmpty(directoryPath))
            {
                return;
            }

            // コピー先ディレクトリの一つ上のパスを取得
            string onceUpPath = Path.GetDirectoryName(directoryPath);
            if (!string.IsNullOrEmpty(onceUpPath))
            {
                // 末端のディレクトリ名を取得
                string lastDirectoryName = directoryPath.Substring(onceUpPath.Length).Replace("\\", "");

                // アセット用フォルダ名ごとに調べる
                var nameList = new[] { TextureFolderName, PrimitiveFolderName, CombinerShaderFolderName };
                foreach (var name in nameList)
                {
                    // 末端ディレクトリがアセット用のフォルダになっているか？
                    if (string.Equals(lastDirectoryName, name, StringComparison.OrdinalIgnoreCase))
                    {
                        string fileName = Path.GetFileName(copyInfo.DstPath);

                        // 先頭が大文字の場合
                        if (Directory.Exists(Path.Combine(onceUpPath, name)))
                        {
                            copyInfo.DstPath = Path.Combine(onceUpPath, name, fileName);
                            return;
                        }

                        // 全部大文字の場合
                        if (Directory.Exists(Path.Combine(onceUpPath, name.ToUpper())))
                        {
                            copyInfo.DstPath = Path.Combine(onceUpPath, name.ToUpper(), fileName);
                            return;
                        }

                        // 全部小文字の場合
                        if (Directory.Exists(Path.Combine(onceUpPath, name.ToLower())))
                        {
                            copyInfo.DstPath = Path.Combine(onceUpPath, name.ToLower(), fileName);
                            return;
                        }

                        // 上記3パターンに合致した場合は、コピーパスを見つかったパターンに合わせて修正し、作成はしない
                        // 上記3パターンに合致しない場合はディレクトリがないものと見なし、作成を行う
                        break;
                    }
                }
            }

            if (!Directory.Exists(directoryPath))
            {
                Directory.CreateDirectory(directoryPath);
            }
        }

        /// <summary>
        /// デフォルトのドキュメント起動パスを返します。
        /// </summary>
        public static string DefaultDocumentPath
        {
            get
            {
                return @"..\..\..\Documents\index.html";
            }
        }
    }
}
