﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Synchronization;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.GfxToolsUtility
{
    /// <summary>
    /// テクスチャコンバータの実行管理クラスです。
    /// </summary>
    public class TextureConverterExecuter
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static TextureConverterExecuter singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static TextureConverterExecuter Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new TextureConverterExecuter();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// テクスチャコンバータの実行ファイルパスを取得します。
        /// </summary>
        public string ExecutableFilePath { get; private set; }

        /// <summary>
        /// テクスチャコンバータの中間生成物配置フォルダパスを取得します。
        /// </summary>
        public string IntermediateDirectoryPath { get; private set; }

        /// <summary>
        /// テクスチャコンバータのエラーログ配置フォルダパスを取得します。
        /// </summary>
        public string ErrorLogDirecotoryPath { get; private set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private TextureConverterExecuter()
        {
            string processId = Process.GetCurrentProcess().Id.ToString();

            string globalIntermediateDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\texture_intermediate");
            string localIntermediateDirectoryPath = Path.Combine(globalIntermediateDirectoryPath, processId);

            string globalErrorLogDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\texture_error");
            string localErrorLogDirectoryPath = Path.Combine(globalErrorLogDirectoryPath, processId);

            // 実行に必要なフォルダを作成
            using (new GlobalSyncSection())
            {
                // 古いデータを削除
                IOUtility.DeleteExpiredFileSystemEntries(globalIntermediateDirectoryPath, TimeSpan.FromDays(3.0));
                IOUtility.DeleteExpiredFileSystemEntries(globalErrorLogDirectoryPath, TimeSpan.FromDays(3.0));

                // 中間生成物配置フォルダを作成
                IOUtility.SafeCreateTemporaryDirectory(globalIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForMyProcess);

                // エラーログ配置フォルダを作成
                // エラーログファイルは確認用に残しておくためForGlobalで作成する
                IOUtility.SafeCreateTemporaryDirectory(globalErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
            }

            this.ExecutableFilePath = Path.Combine(IOConstants.ToolsDirectoryPath, @"Graphics\3dTools\3dTextureConverter.exe");

            this.IntermediateDirectoryPath = localIntermediateDirectoryPath;
            this.ErrorLogDirecotoryPath = localErrorLogDirectoryPath;
        }

        /// <summary>
        /// テクスチャコンバータを実行します。
        /// </summary>
        /// <param name="param">コンバートパラメータ</param>
        /// <param name="outputOptions">コマンドラインオプションをログに出力するか</param>
        /// <param name="standardOutput">ツールの標準出力</param>
        /// <returns>コンバート結果を返します。</returns>
        public ConvertResult Execute(ConvertParameter param)
        {
            // ツールの作業フォルダがあるかチェック
            {
                string[] workingDirectories = new string[]
                {
                    this.IntermediateDirectoryPath,
                    this.ErrorLogDirecotoryPath
                };

                foreach (string path in workingDirectories)
                {
                    if (Directory.Exists(path) == false)
                    {
                        ConvertResult res = new ConvertResult()
                        {
                            ResultCode    = -1,
                            StandardError = string.Format("Temporary folder not found at {0}.", path)
                        };

                        return res;
                    }
                }
            }

            // 実行ファイルがあるかチェック
            if (File.Exists(this.ExecutableFilePath) == false)
            {
                ConvertResult res = new ConvertResult()
                {
                    ResultCode    = -1,
                    StandardError = string.Format("Executable file not found at {0}.", this.ExecutableFilePath)
                };

                return res;
            }

            StringBuilder args = new StringBuilder();

            // コマンドラインオプションを作成
            {
                args.Append("-o " + param.OutputPath);
                args.Append(" --tile-mode=" + param.TileMode);

                if (string.IsNullOrEmpty(param.TileOptimize) == false)
                {
                    args.Append(" --tile-optimize=" + param.TileOptimize.ToLower());
                }

                if (param.TileSizeThreshold != null)
                {
                    args.Append(" --tile-size-threshold=" + param.TileSizeThreshold);
                }

                foreach (string path in param.FilePaths)
                {
                    args.Append(string.Format(" \"{0}\"", path));
                }
            }

            GfxToolExecutionUtility.ExecutionResult resExecute;

            string workingDirectoryPath = Path.GetDirectoryName(this.ExecutableFilePath);

            // コマンドライン引数は最長で32767文字という制限があるため、それを超えるときは
            // コマンドラインオプションをファイルに出力してコンバータに渡す
            // (32767文字きっかりで分岐するのも不安なので1000文字分余裕を持たせています)
            if (args.Length >= 32767 - 1000)
            {
                string argsFilePath = Path.Combine(this.IntermediateDirectoryPath, "args.txt");

                // コマンドラインオプションをファイルに書き出す
                bool resWrite = IOUtility.WriteAllTextEx(argsFilePath, args.ToString(), Encoding.Default, 10, 30);

                // コマンドラインオプションを作り直す
                {
                    args = new StringBuilder();

                    args.Append(string.Format("--args-file=\"{0}\"", argsFilePath));
                }

                // コンバートを実行
                resExecute = GfxToolExecutionUtility.ExecuteExeFile(this.ExecutableFilePath, workingDirectoryPath, args.ToString());
            }
            // ファイルの入出力が絡むと処理速度や安定性が低下するため、32767文字以下のときは
            // コマンドライン引数にテクスチャファイルパスを渡すようにする
            else
            {
                // コンバートを実行
                resExecute = GfxToolExecutionUtility.ExecuteExeFile(this.ExecutableFilePath, workingDirectoryPath, args.ToString());
            }

            ConvertResult result = new ConvertResult()
            {
                ResultCode     = resExecute.ResultCode,
                StandardOutput = resExecute.StandardOutputLog,
                StandardError  = resExecute.StandardErrorLog,
            };

            return result;
        }

        /// <summary>
        /// テクスチャコンバートパラメータです。
        /// </summary>
        public class ConvertParameter
        {
            /// <summary>
            /// バイナリファイルの出力先を取得または設定します。
            /// </summary>
            public string OutputPath { get; set; }

            /// <summary>
            /// タイリングモードを取得または設定します。
            /// </summary>
            public string TileMode { get; set; }

            /// <summary>
            /// タイリング最適化モードを取得または設定します。
            /// </summary>
            public string TileOptimize { get; set; }

            /// <summary>
            /// タイリングサイズ最適化のしきい値を取得または設定します。
            /// </summary>
            public int? TileSizeThreshold { get; set; }

            /// <summary>
            /// テクスチャファイルパスを取得または設定します。
            /// </summary>
            public IEnumerable<string> FilePaths { get; set; }
        }

        /// <summary>
        /// コンバート結果です。
        /// </summary>
        public class ConvertResult
        {
            /// <summary>
            /// リザルトコードを取得または設定します。
            /// </summary>
            public int ResultCode { get; set; }

            /// <summary>
            /// 標準出力のテキストを取得または設定します。
            /// </summary>
            public string StandardOutput { get; set; }

            /// <summary>
            /// エラー出力のテキストを取得または設定します。
            /// </summary>
            public string StandardError { get; set; }

            /// <summary>
            /// 全ての出力をまとめたテキストを取得します。
            /// </summary>
            /// <returns>全ての出力をまとめたテキスト</returns>
            public string GetAllOutputs()
            {
                StringBuilder log = new StringBuilder();

                log.AppendLine(" -- Standard Output --");
                log.AppendLine(this.StandardOutput.ToString());
                log.AppendLine();

                log.AppendLine(" -- Standard Error --");
                log.AppendLine(this.StandardError.ToString());
                log.AppendLine();

                return log.ToString();
            }
        }
    }
}
