﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.Foundation.Log;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.SpecDefinitions;

using TextureManagerClass = EffectMaker.BusinessLogic.Manager.TextureManager;

namespace EffectMaker.BusinessLogic.GfxToolsUtility
{
    /// <summary>
    /// テクスチャのbntxバイナリのキャッシュを管理します。
    /// </summary>
    public class GfxTextureBinaryConverter
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static GfxTextureBinaryConverter singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static GfxTextureBinaryConverter Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new GfxTextureBinaryConverter();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// バイナリキャッシュです。
        /// キャッシュ情報はエミッタセットごとに作成します。
        /// </summary>
        private Dictionary<Guid, BinaryCacheInfo> binaryCache = new Dictionary<Guid, BinaryCacheInfo>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private GfxTextureBinaryConverter()
        {
            // スペック切り替え時にキャッシュをクリアする
            SpecManager.CurrentSpecChanged += (s, e) =>
            {
                this.binaryCache.Clear();
            };
        }

        /// <summary>
        /// G3Dテクスチャをコンバートしてbntxバイナリを生成します。
        /// コンバート元データに変更がないときはキャッシュを利用してコンバート処理を省略します。
        /// </summary>
        /// <param name="emitterSetGuid">bntxバイナリを利用するエミッタセットのGUID</param>
        /// <param name="texturePaths">バイナリコンバートするテクスチャのファイルパス一覧</param>
        /// <param name="outputPath">バイナリコンバートするときの、bntxバイナリの出力ファイルパス</param>
        /// <param name="forceToBinarize">キャッシュを利用せずに、強制的にバイナリコンバートを実行するときtrue</param>
        /// <returns>生成したバイナリデータを返します。コンバートに失敗したときはnullを返します。</returns>
        public byte[] Convert(Guid                emitterSetGuid,
                              IEnumerable<string> texturePaths,
                              string              outputPath,
                              bool                forceToBinarize = false)
        {
            string[] texturePathArray = texturePaths.ToArray();

            // バイナリキャッシュが利用可能ならキャッシュを返す
            if (forceToBinarize == false)
            {
                BinaryCacheInfo cacheInfo;
                bool findCache = binaryCache.TryGetValue(emitterSetGuid, out cacheInfo);

                if (findCache)
                {
                    if (IsBinaryCacheValid(cacheInfo, texturePathArray))
                    {
                        return cacheInfo.BinaryData;
                    }
                    else
                    {
                        binaryCache.Remove(emitterSetGuid);
                    }
                }
            }

            // コンバートを実行
            byte[] binaryData = this.ConvertInternal(outputPath, texturePathArray);

            if (binaryData == null)
            {
                return null;
            }

            // バイナリデータをキャッシュに追加する
            {
                TextureManagerClass textureManager = TextureManagerClass.Instance;

                BinarySourceFileInfo[] sourceFileInfo = texturePathArray.Select(x => new BinarySourceFileInfo()
                {
                    FilePath = x,
                    ModifyTime = textureManager.GetLoadefFileModifyTime(x)
                }).ToArray();

                BinaryCacheInfo cacheInfo = new BinaryCacheInfo()
                {
                    EmitterSetGuid = emitterSetGuid,
                    SourceFiles = sourceFileInfo,
                    BinaryData = binaryData
                };

                binaryCache[emitterSetGuid] = cacheInfo;
            }

            return binaryData;
        }

        /// <summary>
        /// コンバート処理を実行します。
        /// </summary>
        /// <param name ="outputPath">バイナリファイルの出力先</param>
        /// <param name="textureArray">バイナリにコンバートするテクスチャ</param>
        /// <returns>生成したバイナリデータを返します。コンバートに失敗したときはnullを返します。</returns>
        private byte[] ConvertInternal(string outputPath, IEnumerable<string> filePathList)
        {
            TexturePackingOption texturePackingOption = SpecManager.CurrentSpec.TexturePackingOption;

            // コンバートパラメータを設定
            var param = new TextureConverterExecuter.ConvertParameter()
            {
                OutputPath        = outputPath,
                TileMode          = texturePackingOption.TileMode,
                TileOptimize      = texturePackingOption.TileOptimize,
                TileSizeThreshold = texturePackingOption.TileSizeThreshold,
                FilePaths         = filePathList
            };

            TextureConverterExecuter executer = TextureConverterExecuter.Instance;

            // コンバート処理を実行
            var resConvert = executer.Execute(param);

            // エラー処理を行う
            if (resConvert.ResultCode != 0)
            {
                string allLog = resConvert.GetAllOutputs();
                string logFilePath = Path.Combine(executer.ErrorLogDirecotoryPath, "output_bg.bntx");

                GfxToolExecutionUtility.SaveAndShowErrorMessage(logFilePath, allLog);

                if (OptionStore.RuntimeOptions.IsCommandLineMode)
                {
                    Logger.Log("Console", LogLevels.Error, "Texture Convert Error:\r\n{0}", allLog);
                }

                return null;
            }

            // テクスチャバイナリをファイルから読み込む
            byte[] bntxData = File.ReadAllBytes(outputPath);

            return bntxData;
        }

        /// <summary>
        /// バイナリキャッシュが利用可能かどうかチェックします。
        /// </summary>
        /// <param name="cacheInfo">バイナリキャッシュ情報</param>
        /// <param name="sourceFiles">生成元となるファイル</param>
        /// <returns>バイナリキャッシュが利用可能であればtrue、それ以外はfalseを返します。</returns>
        private bool IsBinaryCacheValid(BinaryCacheInfo cacheInfo, string[] sourceFiles)
        {
            // 生成元となるファイルの増減をチェック
            if (cacheInfo.SourceFiles.Length != sourceFiles.Length)
            {
                return false;
            }

            // 生成元となるファイルに差異がないかチェック
            // ファイルの順番が入れ替わった場合もキャッシュは無効となる
            for (int i = 0; i < cacheInfo.SourceFiles.Length; ++i)
            {
                // ファイルパスを比較
                if (cacheInfo.SourceFiles[i].FilePath != sourceFiles[i])
                {
                    return false;
                }

                DateTime modifyTime = File.GetLastWriteTime(cacheInfo.SourceFiles[i].FilePath);

                // ファイルの変更日時を比較
                if (modifyTime != cacheInfo.SourceFiles[i].ModifyTime)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// エミッタセットに関連付けられたキャッシュを削除します。
        /// </summary>
        /// <param name="emitterSetGuid">エミッタセットのGUID</param>
        public void RemoveCache(Guid emitterSetGuid)
        {
            this.binaryCache.Remove(emitterSetGuid);
        }

        /// <summary>
        /// バイナリキャッシュ情報です。
        /// </summary>
        private class BinaryCacheInfo
        {
            /// <summary>
            /// エミッタセットのGUIDを取得または設定します。
            /// </summary>
            public Guid EmitterSetGuid { get; set; }

            /// <summary>
            /// バイナリの生成元となるファイルの情報を取得または設定します。
            /// </summary>
            public BinarySourceFileInfo[] SourceFiles { get; set; }

            /// <summary>
            /// バイナリデータを取得または設定します。
            /// </summary>
            public byte[] BinaryData { get; set; }
        }

        /// <summary>
        /// バイナリの生成元となるファイルの情報です。
        /// </summary>
        private class BinarySourceFileInfo
        {
            /// <summary>
            /// ファイルパスを取得または設定します。
            /// </summary>
            public string FilePath { get; set; }

            /// <summary>
            /// ファイルの変更日時を取得または設定します。
            /// </summary>
            public DateTime ModifyTime { get; set; }
        }
    }
}
