﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.DataModelOperation
{
    /// <summary>
    /// DefaultNameGenerator.
    /// </summary>
    public class DefaultNameGenerator
    {
        #region Generate default document name

        /// <summary>
        /// Generate default emitter set name.
        /// </summary>
        /// <param name="existingNames">existng emitter name.</param>
        /// <returns>The document name.</returns>
        public static string GenerateDefaultEmitterSetName(IEnumerable<string> existingNames)
        {
            string docName = "EmitterSet";

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex("^" + docName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach (string emitterSetName in existingNames)
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match(emitterSetName);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int currSerial;
                if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                {
                    existingSerialList.Add(currSerial);
                }
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber(existingSerialList).ToString();

            return docName;
        }

        /// <summary>
        /// Generate default emitter name.
        /// </summary>
        /// <param name="existingNames">existng emitter name.</param>
        /// <returns>The emitter name.</returns>
        public static string GenerateDefaultEmitterName(IEnumerable<string> existingNames)
        {
            string docName = "Emitter";

            if (existingNames == null)
            {
                return docName + "1";
            }

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex("^" + docName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach (string emitterName in existingNames)
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match(emitterName);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int currSerial;
                if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                {
                    existingSerialList.Add(currSerial);
                }
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber(existingSerialList).ToString();

            return docName;
        }

        /// <summary>
        /// Generate default preview name.
        /// </summary>
        /// <param name="ownerName">The name of parent data model.</param>
        /// <param name="existingNames">existng emitter name.</param>
        /// <returns>The preview name.</returns>
        public static string GenerateDefaultPreviewName(string ownerName, IEnumerable<string> existingNames)
        {
            string docName = ownerName;

            if (existingNames == null)
            {
                return docName;
            }

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex("^" + docName + "_([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach (string previewName in existingNames)
            {
                if (previewName == docName)
                {
                    existingSerialList.Add(1);
                }
                else
                {
                    System.Text.RegularExpressions.Match match =
                        regexCaptureSerial.Match(previewName);
                    if (match.Groups.Count < 2)
                    {
                        continue;
                    }

                    int currSerial;
                    if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                    {
                        existingSerialList.Add(currSerial);
                    }
                }
            }

            // Add the serial number to the document name
            int serial = FindUnusedSerialNumber(existingSerialList);

            if (serial > 1)
            {
                docName += "_" + serial.ToString();
            }

            return docName;
        }

        /// <summary>
        /// Generate default preview name.
        /// </summary>
        /// <param name="ownerName">The name of parent data model.</param>
        /// <returns>preview name.</returns>
        public static string GenerateDefaultPreviewName(string ownerName)
        {
            return ownerName;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        private static int FindUnusedSerialNumber(List<int> usedSerialNumList)
        {
            int serial = 1;
            bool found  = true;

            while (found == true)
            {
                found = false;

                foreach (int num in usedSerialNumList)
                {
                    if (num == serial)
                    {
                        ++serial;
                        found = true;
                        break;
                    }
                }
            }

            return serial;
        }

        #endregion
    }
}
