﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#region File Description
#endregion

using System.Threading;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using Nintendo.InGameEditing.Communication;

namespace EffectMaker.BusinessLogic.Connecter
{
    #region Using Statements
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Net;
    using System.Net.Sockets;
    using System.Text;
    using EffectMaker.Communicator;
    #endregion

    /// <summary>
    /// TCP/IPコネクターです.
    /// </summary>
    public class TcpConnecter : IConnecter
    {
        #region Private Fields
        /// <summary>
        /// 受信バッファサイズです(バイト単位).
        /// </summary>
        protected static readonly int ReceivedBufferSize = 262144; // 256 byte * 1024 = 262144.

        /// <summary>
        /// 送信バッファサイズです(バイト単位).
        /// </summary>
        protected static readonly int SendBufferSize = 5242880; // 1024 * 1024 * 5 byte.

        /// <summary>
        /// ポートマッピング解決に使用するマネージャーです。
        /// </summary>
        private static ConnectionManager portMapper = null;

        /// <summary>
        /// 全スペックで使用しているプロトコル名とポート番号のマップです。
        /// </summary>
        private static Dictionary<string, int> peerMap = null;

        /// <summary>
        /// コネクションコンテキストです.
        /// </summary>
        private readonly ConnectionContext context = null;

        /// <summary>
        /// TCPクライアントです.
        /// </summary>
        private TcpClient socket = null;

        /// <summary>
        /// 排他制御用オブジェクトです.
        /// </summary>
        private volatile object lockobj = new object();

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        private static string ConnectionInfoHeader
        {
            get { return "NW_FE2"; }
        }

        #endregion

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="context">接続コンテキストです.</param>
        public TcpConnecter(ConnectionContext context)
        {
            this.context = new ConnectionContext(context);
            if (portMapper == null)
            {
                portMapper = new ConnectionManager();

                // ConnectionManager起動時にスペックリストからプロトコル名とポート番号のペアを引っ張っておく
                peerMap = new Dictionary<string, int>();
                foreach (var connection in SpecManager.SpecDefinitions
                    .SelectMany(specDefinition => specDefinition.Connections)
                    .Where(connection => !string.IsNullOrEmpty(connection.Protocol)
                        && !peerMap.ContainsKey(connection.Protocol)))
                {
                    peerMap.Add(connection.Protocol, connection.Port);
                }
            }
        }

        #region Public Properties

        /// <summary>
        /// 接続されているかどうかを取得します.
        /// </summary>
        public virtual bool IsConnected
        {
            get
            {
                // コンテキストがない場合.
                if (this.context == null)
                {
                    return false;
                }

                // コンテキストがあって，準備済みの場合.
                if ((this.context != null) && (this.context.IsReady != true))
                {
                    return false;
                }

                // TCPクライアントがある場合.
                if (this.socket != null)
                {
                    // 接続フラグを返却.
                    return this.socket.Connected;
                }

                // 接続できていない.
                return false;
            }
        }

        /// <summary>
        /// コネクター名を取得します.
        /// </summary>
        public string Name
        {
            get { return "TcpConnecter"; }
        }

        /// <summary>
        /// 受信バッファサイズをこえるかどうかチェックします.
        /// </summary>
        public bool IsReceivedBufferFull
        {
            get
            {
                // 接続されていない, またはソケットが無い場合はfalse.
                if ((this.IsConnected == false) || (this.socket == null))
                {
                    return false;
                }

                // 受信サイズがバッファサイズ以上であれば，あふれるのでtrueを返却.
                return this.socket.Available >= ReceivedBufferSize;
            }
        }

        /// <summary>
        /// コネクションコンテキストを取得します.
        /// </summary>
        public ConnectionContext ConnectionContext
        {
            get { return this.context; }
        }

        #endregion

        /// <summary>
        /// エディタ終了時にポートマッパーを停止します。
        /// </summary>
        public static void ShutdownPortMapper()
        {
            // 一応nullチェック追加
            if (portMapper != null)
            {
                portMapper.Stop();
                portMapper = null;
            }
        }

        /// <summary>
        /// 破棄処理.
        /// </summary>
        public void Dispose()
        {
            // 切断処理とフィールドを解放.
            this.Disconnect();
        }

        /// <summary>
        /// 接続ターゲットが存在するかチェックします.
        /// </summary>
        /// <param name="context">コネクションコンテキストです.</param>
        /// <returns>接続ターゲットが存在する場合はtrue, 存在しない場合はfalseを返却します.</returns>
        public virtual bool CheckTargetExistence(ConnectionContext context)
        {
            // 存在フラグ.
            bool isExist = false;

            if (context.Platform == (int) Protocol.PlatformType.Cafe)
            {
                portMapper.Stop("HIO");
                isExist = IsExistTarget("HIO");
            }
            else if (context.Platform == (int)Protocol.PlatformType.HTC)
            {
                isExist = IsExistTarget(context.Protocol);
            }
            else if (context.Platform == (int)Protocol.PlatformType.Windows)
            {
                return true;
            }

            // 結果を返却.
            return isExist;
        }

        /// <summary>
        /// ターゲットと接続します.
        /// </summary>
        /// <param name="context">コネクションコンテキストです.</param>
        /// <returns>接続に成功したらtrue, 失敗したらfalseを返却します.</returns>
        public virtual bool Connect(ConnectionContext context)
        {
            // コンテキストがあること，準備できていることを確認.
            if (context == null || context.IsReady == false)
            {
                // 接続できないのでfalseを返却.
                return false;
            }

            // 既に接続済みの場合.
            if (this.IsConnected == true)
            {
                // ターゲットが同じであるかチェック.
                if (context.IsSameTarget(this.context) == true)
                {
                    // 接続済みターゲットのなので, trueを返却.
                    return true;
                }
                else
                {
                    // ターゲットが異なるので, 今繋がっているターゲットと切断.
                    this.Disconnect();
                }
            }

            // 現在の状態を一時退避.
            var prevContext = new ConnectionContext(this.context);

            lock (this.lockobj)
            {
                try
                {
                    // コンテキストをコピる.
                    this.context.Copy(context);

                    if (this.context.Platform == (int) Protocol.PlatformType.Cafe)
                    {
                        portMapper.Stop("HIO");
                        var mappedPoint = ResolvePortMapping("HIO", "Cafe", "Cafe");
                        if (mappedPoint == null)
                        {
                            return false;
                        }

                        // 差し替え.
                        context.IPAddress = mappedPoint.Address;
                        context.Port = mappedPoint.Port;
                    }
                    else if (this.context.Platform == (int) Protocol.PlatformType.HTC)
                    {
                        var mappedPoint = ResolvePortMapping(this.context.Protocol, this.context.PeerType);
                        if (mappedPoint == null)
                        {
                            return false;
                        }

                        // 差し替え.
                        context.IPAddress = mappedPoint.Address;
                        context.Port = mappedPoint.Port;
                    }

                    // Create the socket.
                    this.socket = new TcpClient();

                    // 受信バッファサイズを設定.
                    this.socket.ReceiveBufferSize = ReceivedBufferSize;

                    // 送信バッファサイズを設定.
                    this.socket.SendBufferSize = SendBufferSize;

                    // Connect to the target.
                    // If the target does not exist, WinSocket tries two to three times,
                    // with an interval of 500 milliseconds.
                    // This is why it takes more than 1 seconds to connect while
                    // the viewer is not launched.
                    // Maybe we should consider using TcpClient.ConnectAync().
                    this.socket.Connect(context.IPAddress, context.Port);
                }
                catch (SocketException exception)
                {
                    // 型 'System.Net.Sockets.SocketException' の初回例外が System.dll で発生しました
                    if (exception.SocketErrorCode != SocketError.ConnectionRefused)
                    {
                        /* DO_NOTHING */
                    }
                }
                catch (Exception exception)
                {
                    string msg = "Error : " + exception.ToString();
                    /* DO_NOTHING */
                }
            }

            // TCPクライアントがない，または接続されていない場合は失敗.
            if (this.socket == null || this.socket.Connected == false)
            {
                // TCPクライアントをnullに.
                this.socket = null;

                // コンテキストをリセット.
                this.context.Reset();

                // 以前のコンテキストを再設定.
                this.context.Copy(prevContext);

                // 異常終了.
                return false;
            }

            // コンテキストをリセット.
            this.context.Reset();

            // 適用したコンテキストを再設定(これをしないと再接続時に切断されてしまう).
            this.context.Copy(context);

            // 正常終了.
            return true;
        }

        /// <summary>
        /// ターゲットと切断します.
        /// </summary>
        public void Disconnect()
        {
            // TCPクライアントがない, また接続されて無い場合.
            if ((this.socket == null) || (this.socket.Connected == false))
            {
                // TCPクライアントを破棄.
                this.socket = null;

                // 終了.
                return;
            }

            // 排他制御.
            lock (this.lockobj)
            {
                // 切断する.
                this.socket.Close();

                // TCPクライアントを破棄.
                this.socket = null;
            }
        }

        /// <summary>
        /// ターゲットにパケットを送信します.
        /// </summary>
        /// <param name="buffer">送信パケットです</param>
        /// <param name="bufferSize">送信バッファサイズです</param>
        /// <returns>送信に成功したらtrue, 失敗したらfalseを返却します.</returns>
        public bool Send(byte[] buffer, int bufferSize)
        {
            // 接続されていない場合.
            if (this.IsConnected == false)
            {
                // 異常終了.
                return false;
            }

            // TCPクライアントがない, または接続済みでない場合.
            if ((this.socket == null) || (this.socket.Connected == false))
            {
                // 異常終了.
                return false;
            }

            try
            {
                // 排他制御.
                lock (this.lockobj)
                {
                    // ストリームを取得.
                    NetworkStream stream = this.socket.GetStream();

                    // 送信パケットを書き込み.
                    stream.Write(buffer, 0, bufferSize);

                    // 送信パケットデータをフラッシュする.
                    stream.Flush();
                }
            }
            catch (Exception exception)
            {
                string msg = "Error : " + exception.ToString();

                // 切断する.
                this.Disconnect();

                // 異常終了.
                return false;
            }

            // 正常終了.
            return true;
        }

        /// <summary>
        /// ターゲットからパケットを受信します.
        /// </summary>
        /// <param name="buffer">受信パケット</param>
        /// <returns>受信したパケットサイズを返却します.</returns>
        public int Receive(out byte[] buffer)
        {
            buffer = null;

            // 接続済みでない.
            if (this.IsConnected == false)
            {
                // 受信できないのでゼロを返却.
                return 0;
            }

            // TCPクライアントがない, また接続済みでない場合
            if (this.socket == null || this.socket.Connected == false)
            {
                // 受信できないのでゼロを返却.
                return 0;
            }

            // 受信サイズ.
            int size = 0;

            try
            {
                // 利用可能なサイズがある場合.
                if (this.socket.Available > 0)
                {
                    // 排他制御.
                    lock (this.lockobj)
                    {
                        // ストリームを取得.
                        NetworkStream stream = this.socket.GetStream();

                        // バッファのメモリを確保.
                        buffer = new byte[this.socket.Available];

                        // 受信データを読み込み.
                        size = stream.Read(buffer, 0, this.socket.Available);
                    }
                }
            }
            catch
            {
                /* DO_NOTHING */
            }

            // バッファがnullかチェック.
            if (buffer == null)
            {
                // バッファがnullなのでゼロを返却.
                return 0;
            }

            // サイズが小さい方を返却.
            return System.Math.Min(size, buffer.Length);
        }

        /// <summary>
        /// ターゲットの存在確認を行い、結果を得ます。
        /// HTCならデーモンが起動しているか、HIOなら本体が起動しているかを返します。
        /// </summary>
        /// <param name="protocolName">確認したい通信プロトコル名.</param>
        /// <returns>ターゲットが起動していたらtrue,見つからなければfalse.</returns>
        private static bool IsExistTarget(string protocolName)
        {
            int port;
            if (!peerMap.TryGetValue(protocolName, out port))
            {
                return false;
            }

            portMapper.Start(protocolName, port);

            // 100msずつ10回試行(Cafeは30回)
            int retryCount = (protocolName == "HIO") ? 30 : 10;
            while (retryCount > 0)
            {
                if (portMapper.IsAvailable(protocolName))
                {
                    portMapper.Stop(protocolName);
                    return true;
                }

                Thread.Sleep(100);
                --retryCount;
            }

            // portMapper.Stop(protocolName);

            return false;
        }

        /// <summary>
        /// InGameEditing.Communication を利用してポートマッピングを解決します。
        /// </summary>
        /// <param name="protocolName">対象とする通信プロトコル名.</param>
        /// <param name="peerTypeName">対象とするピア名(HTCSの時のみ使用).</param>
        /// <param name="hardwareName">引っ掛けるハードウェア識別名(省略時は全ハードが対象).</param>
        /// <returns>解決済みのエンドポイント</returns>
        private static IPEndPoint ResolvePortMapping(string protocolName, string peerTypeName, string hardwareName = "")
        {
            int port;
            if (!peerMap.TryGetValue(protocolName, out port))
            {
                return null;
            }

            portMapper.Start(protocolName, port);

            // 100msずつ10回試行(Cafeは30回)
            IPEndPoint mappedPoint = null;
            int retryCount = (protocolName == "HIO") ? 30 : 10;
            while (retryCount > 0)
            {
                if (portMapper.IsAvailable(protocolName))
                {
                    var info = portMapper.TargetInfos
                        // HTCSならピア名の一致をチェック、それ以外ならプロトコル名のみをチェック
                        .Where(t => (protocolName == "HTCS" && t.PeerType == peerTypeName)
                            || t.PeerType == protocolName)
                        // ハード名が空白なら全部受け入れ、指定がある場合はそれのみ
                        .Where(t => string.IsNullOrEmpty(hardwareName) || t.HtcsPeerName == hardwareName)
                        // 以上に該当するポート情報のリストを抽出し……
                        .SelectMany(t => t.PortInfos)
                        // エフェクトライブラリのポート名で待ち構えている奴を選ぶ
                        .FirstOrDefault(t => t.PortName == ConnectionInfoHeader);

                    if (info != null)
                    {
                        mappedPoint = info.EndPoint;
                    }

                    // プロトコルには繋がっているのに上記のLINQで見つからない場合は
                    // ターゲットが起動していないことが確定なのでnullのままbreak
                    break;
                }

                Thread.Sleep(100);
                --retryCount;
            }

            // portMapper.Stop(protocolName);

            return mappedPoint;
        }

        /// <summary>
        /// 全プロトコルを対象としてポートマッピングを解決します。
        /// </summary>
        /// <param name="hardwareName">引っ掛けるハードウェア識別名(省略時は全ハードが対象).</param>
        /// <returns>解決済みのエンドポイント</returns>
        private static IPEndPoint ResolvePortMappingFromAll(string hardwareName = "")
        {
            foreach (var peer in peerMap)
            {
                portMapper.Start(peer.Key, peer.Value);
            }

            // 100msずつ30回試行
            IPEndPoint mappedPoint = null;
            int retryCount = 30;
            while (retryCount > 0)
            {
                var info = portMapper.TargetInfos
                    // ハード名が空白なら全部受け入れ、指定がある場合はそれのみ
                    .Where(t => string.IsNullOrEmpty(hardwareName) || t.HtcsPeerName == hardwareName)
                    // 以上に該当するポート情報のリストを抽出し……
                    .SelectMany(t => t.PortInfos)
                    // エフェクトライブラリのポート名で待ち構えている奴を選ぶ
                    .FirstOrDefault(t => t.PortName == ConnectionInfoHeader);

                if (info != null)
                {
                    // 全プロトコル対象の場合は見つかり次第抜ける
                    mappedPoint = info.EndPoint;
                    break;
                }

                Thread.Sleep(100);
                --retryCount;
            }

            // portMapper.Stop();

            return mappedPoint;
        }
    }
}
