﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using MyTextureManager = EffectMaker.BusinessLogic.Manager.TextureManager;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Texture
{
    /// <summary>
    /// Write texture resource to binary stream.
    /// </summary>
    public class TextureResourceWriter : IBinaryResourceWriter
    {
        /// <summary>Texture pixel format conversion table.</summary>
        private static readonly Dictionary<PixelFormats, byte> PixelFormatConversionTable = new Dictionary<PixelFormats, byte>()
        {
            { PixelFormats.Unorm_8_8_8_8,     2 },  // EFT_TEXTURE_FORMAT_32BIT_COLOR.
            { PixelFormats.Unorm_bc1,         3 },  // EFT_TEXTURE_FORMAT_UNORM_BC1.
            { PixelFormats.Srgb_bc1,          4 },  // EFT_TEXTURE_FORMAT_SRGB_BC1.
            { PixelFormats.Unorm_bc2,         5 },  // EFT_TEXTURE_FORMAT_UNORM_BC2
            { PixelFormats.Srgb_bc2,          6 },  // EFT_TEXTURE_FORMAT_SRGB_BC2
            { PixelFormats.Unorm_bc3,         7 },  // EFT_TEXTURE_FORMAT_UNORM_BC3
            { PixelFormats.Srgb_bc3,          8 },  // EFT_TEXTURE_FORMAT_SRGB_BC3
            { PixelFormats.Unorm_bc4,         9 },  // EFT_TEXTURE_FORMAT_UNORM_BC4
            { PixelFormats.Snorm_bc4,         10 }, // EFT_TEXTURE_FORMAT_SNORM_BC4
            { PixelFormats.Unorm_bc5,         11 }, // EFT_TEXTURE_FORMAT_UNORM_BC5
            { PixelFormats.Snorm_bc5,         12 }, // EFT_TEXTURE_FORMAT_SNORM_BC5
            { PixelFormats.Unorm_8,           13 }, // EFT_TEXTURE_FORMAT_UNORM_8
            { PixelFormats.Unorm_8_8,         14 }, // EFT_TEXTURE_FORMAT_UNORM_8_8
            { PixelFormats.Srgb_8_8_8_8,      15 }, // EFT_TEXTURE_FORMAT_SRGB_8_8_8_8
            { PixelFormats.Snorm_8,           16 }, // EFT_TEXTURE_FORMAT_SNORM_8
            { PixelFormats.Unorm_4_4,         17 }, // EFT_TEXTURE_FORMAT_UNORM_4_4
            { PixelFormats.Float_11_11_10,    18 }, // EFT_TEXTURE_FORMAT_11_11_10
            { PixelFormats.Float_16,          19 }, // EFT_TEXTURE_FORMAT_16
            { PixelFormats.Float_16_16,       20 }, // EFT_TEXTURE_FORMAT_16_16
            { PixelFormats.Float_16_16_16_16, 21 }, // EFT_TEXTURE_FORMAT_16_16_16_16
            { PixelFormats.Float_32,          22 }, // EFT_TEXTURE_FORMAT_32
            { PixelFormats.Float_32_32,       23 }, // EFT_TEXTURE_FORMAT_32_32
            { PixelFormats.Float_32_32_32_32, 24 }, // EFT_TEXTURE_FORMAT_32_32_32_32
            { PixelFormats.Unorm_pvrtc1_2bpp,       41 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC1_2BPP
            { PixelFormats.Unorm_pvrtc1_4bpp,       42 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC1_4BPP
            { PixelFormats.Unorm_pvrtc1_alpha_2bpp, 43 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_2BPP
            { PixelFormats.Unorm_pvrtc1_alpha_4bpp, 44 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_4BPP
            { PixelFormats.Unorm_pvrtc2_alpha_2bpp, 45 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_2BPP
            { PixelFormats.Unorm_pvrtc2_alpha_4bpp, 46 }, // EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_4BPP
            { PixelFormats.Srgb_pvrtc1_2bpp,        47 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC1_2BPP
            { PixelFormats.Srgb_pvrtc1_4bpp,        48 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC1_4BPP
            { PixelFormats.Srgb_pvrtc1_alpha_2bpp,  49 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_2BPP
            { PixelFormats.Srgb_pvrtc1_alpha_4bpp,  50 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_4BPP
            { PixelFormats.Srgb_pvrtc2_alpha_2bpp,  51 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_2BPP
            { PixelFormats.Srgb_pvrtc2_alpha_4bpp,  52 }, // EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_4BPP
        };

        /// <summary>Texture native pixel format conervsion table.</summary>
        private static readonly Dictionary<PixelFormats, uint> NativeFormatTable = new Dictionary<PixelFormats, uint>()
        {
            { PixelFormats.Unorm_8,             0x00000001 },
            { PixelFormats.Uint_8,              0x00000101 },
            { PixelFormats.Snorm_8,             0x00000201 },
            { PixelFormats.Sint_8,              0x00000301 },
            { PixelFormats.Unorm_4_4,           0x00000002 },
            { PixelFormats.Unorm_16,            0x00000005 },
            { PixelFormats.Uint_16,             0x00000105 },
            { PixelFormats.Snorm_16,            0x00000205 },
            { PixelFormats.Sint_16,             0x00000305 },
            { PixelFormats.Float_16,            0x00000806 },
            { PixelFormats.Unorm_8_8,           0x00000007 },
            { PixelFormats.Uint_8_8,            0x00000107 },
            { PixelFormats.Snorm_8_8,           0x00000207 },
            { PixelFormats.Sint_8_8,            0x00000307 },
            { PixelFormats.Unorm_5_6_5,         0x00000008 },
            { PixelFormats.Unorm_5_5_5_1,       0x0000000a },
            { PixelFormats.Unorm_4_4_4_4,       0x0000000b },
            { PixelFormats.Unorm_1_5_5_5,       0x0000000c },
            { PixelFormats.Uint_32,             0x0000010d },
            { PixelFormats.Sint_32,             0x0000030d },
            { PixelFormats.Float_32,            0x0000080e },
            { PixelFormats.Unorm_16_16,         0x0000000f },
            { PixelFormats.Uint_16_16,          0x0000010f },
            { PixelFormats.Snorm_16_16,         0x0000020f },
            { PixelFormats.Sint_16_16,          0x0000030f },
            { PixelFormats.Float_16_16,         0x00000810 },
            { PixelFormats.Float_11_11_10,      0x00000816 },
            { PixelFormats.Unorm_10_10_10_2,    0x00000019 },
            { PixelFormats.Uint_10_10_10_2,     0x00000119 },
            { PixelFormats.Unorm_8_8_8_8,       0x0000001a },
            { PixelFormats.Uint_8_8_8_8,        0x0000011a },
            { PixelFormats.Snorm_8_8_8_8,       0x0000021a },
            { PixelFormats.Sint_8_8_8_8,        0x0000031a },
            { PixelFormats.Srgb_8_8_8_8,        0x0000041a },
            { PixelFormats.Unorm_2_10_10_10,    0x0000001b },
            { PixelFormats.Uint_2_10_10_10,     0x0000011b },
            { PixelFormats.Uint_32_32,          0x0000011d },
            { PixelFormats.Sint_32_32,          0x0000031d },
            { PixelFormats.Float_32_32,         0x0000081e },
            { PixelFormats.Unorm_16_16_16_16,   0x0000001f },
            { PixelFormats.Uint_16_16_16_16,    0x0000011f },
            { PixelFormats.Snorm_16_16_16_16,   0x0000021f },
            { PixelFormats.Sint_16_16_16_16,    0x0000031f },
            { PixelFormats.Float_16_16_16_16,   0x00000820 },
            { PixelFormats.Uint_32_32_32_32,    0x00000122 },
            { PixelFormats.Sint_32_32_32_32,    0x00000322 },
            { PixelFormats.Float_32_32_32_32,   0x00000823 },
            { PixelFormats.Unorm_bc1,           0x00000031 },
            { PixelFormats.Srgb_bc1,            0x00000431 },
            { PixelFormats.Unorm_bc2,           0x00000032 },
            { PixelFormats.Srgb_bc2,            0x00000432 },
            { PixelFormats.Unorm_bc3,           0x00000033 },
            { PixelFormats.Srgb_bc3,            0x00000433 },
            { PixelFormats.Unorm_bc4,           0x00000034 },
            { PixelFormats.Snorm_bc4,           0x00000234 },
            { PixelFormats.Unorm_bc5,           0x00000035 },
            { PixelFormats.Snorm_bc5,           0x00000235 },
        };

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="data">The data of texture resource.</param>
        public TextureResourceWriter(TextureResourceData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The texture resource data must not be null.");
            }

            this.DataModel = data;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var textureRes = this.DataModel as TextureResourceData;
            if (textureRes == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Try to get the texture.
            LoadTextureResult result = MyTextureManager.Instance.LoadTexture(textureRes.TexturePath, true);

            // テクスチャのロードに失敗したときコンバートを中止する
            // tgaテクスチャを指定したFE1データを開いたときなどに来る
            if (result.ResultCode != LoadTextureResultCode.Success)
            {
                Logger.Log("Console,LogView", LogLevels.Error, Properties.Resources.TextureResourceWriterFailedToLoadTexture, textureRes.TexturePath);
                return false;
            }

            // Only FTX textures are supported now.
            var ftxTexData = result.TextureData as FtxTextureData;
            if (ftxTexData == null || string.IsNullOrEmpty(textureRes.TexturePath))
            {
                return false;
            }

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            //// The ResTexture structure is defined as below :
            //// struct ResTexture
            //// {
            ////     u16             width;             // 横幅.
            ////     u16             height;            // 縦幅.
            ////     u32             depth;             // 配列数.
            ////     u32             compSel;           // コンポーネントスワップ.
            ////     u32             mipLevel;          // ミップレベル数.
            ////     u32             nativeFormat;      // 機種固有フォーマット(GX2のフォーマット)
            ////     u32             nativeTileMode;    // 機種固有タイルモード.
            ////     u32             nativeSwizzle;     // 機種固有スウィズル.
            ////     u32             swizzle;           // Win用スウィズル.
            ////     u64             hashCode;          // ハッシュコード.
            ////     u8              format;            // フォーマット.
            ////     u8              dummy[7];          // ダミー.
            //// };

            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)ftxTexData.Width);
            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)ftxTexData.Height);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)ftxTexData.Depth);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)this.ConvertComponentSelector(ftxTexData.CompSel));
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)ftxTexData.MipmapCount);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)this.ConvertNativeFormat(ftxTexData.PixelFormat));
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)ftxTexData.TileMode);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)ftxTexData.InitialSwizzle);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)ftxTexData.Swizzle);

            var guid = MyTextureManager.Instance.GetGuid(textureRes.TexturePath);
            BinaryConversionUtility.ForResource.WriteStream(stream, guid);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.ConvertPixelFormat(ftxTexData.PixelFormat));
            BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, 7));

            // Set the size.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }

        /// <summary>
        /// Convert pixel format to runtime format.
        /// </summary>
        /// <param name="pixelFormat">The pixel format.</param>
        /// <returns>The pixel format the runtime uses.</returns>
        private byte ConvertPixelFormat(PixelFormats pixelFormat)
        {
            if (PixelFormatConversionTable.ContainsKey(pixelFormat) == false)
            {
                return 0;
            }

            return PixelFormatConversionTable[pixelFormat];
        }

        /// <summary>
        /// Convert component selector.
        /// </summary>
        /// <param name="compSel">The component selector.</param>
        /// <returns>The converted component selector.</returns>
        private uint ConvertComponentSelector(ColorComponents[] compSel)
        {
            uint result = 0x00000000;
            if (compSel != null)
            {
                foreach (ColorComponents component in compSel)
                {
                    result = (result << 8) | (uint)component;
                }
            }

            return result;
        }

        /// <summary>
        /// Convert native pixel format
        /// </summary>
        /// <param name="pixelFormat">The pixel format.</param>
        /// <returns>The native pixel format the runtime uses.</returns>
        private uint ConvertNativeFormat(PixelFormats pixelFormat)
        {
            if (NativeFormatTable.ContainsKey(pixelFormat) == false)
            {
                return 0;
            }

            return NativeFormatTable[pixelFormat];
        }
    }
}
