﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Text;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Shader
{
    /// <summary>
    /// Write shader resource to binary stream.
    /// </summary>
    public class ShaderResourceWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="data">The primitive resource data.</param>
        public ShaderResourceWriter(ShaderResourceData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The shader resource data must not be null.");
            }

            this.DataModel = data;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var shaderRes = this.DataModel as ShaderResourceData;
            if (shaderRes == null)
            {
                return false;
            }

            this.Position = stream.Position;

            // Write empty binary structure header,
            // will be filled up later in the binary conversion process.
            BinaryStructHeader.Empty.Write(stream);

            // Write shader source code to the binary stream.
            byte[] data = Encoding.UTF8.GetBytes(shaderRes.ShaderCode);
            BinaryConversionUtility.ForResource.WriteStream(stream, data);

            this.Size = stream.Position - this.Position;

            // Add binary writer to the context.
            context.AddBinaryWriter(this);

            // 正常終了.
            return true;
        }
    }
}
