﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Utility;
using MyPrimitiveManager = EffectMaker.BusinessLogic.Manager.PrimitiveManager;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// トリミングプリミティブリソースをバイナリに書き出すクラスです。
    /// </summary>
    public class TrimmingPrimitiveResourceWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">トリミングプリミティブリソース</param>
        public TrimmingPrimitiveResourceWriter(TrimmingPrimitiveResourceData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The trimming primitive resource data must be null.");
            }

            this.DataModel = data;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var primitiveRes = this.DataModel as TrimmingPrimitiveResourceData;
            if (primitiveRes == null)
            {
                return false;
            }

            this.Position = stream.Position;

            EffectMaker.Foundation.Model.ModelData data;
            LoadModelResults result =
                MyPrimitiveManager.Instance.CreateTrimmingPrimitiveWithData(
                primitiveRes.TexturePath,
                true,
                out data);

            // Failed to find model, bail out.
            if (result != LoadModelResults.Success)
            {
                return false;
            }

            PrimitiveResourceWriter.WriteCore(stream, data, MyPrimitiveManager.Instance.GetGuid(primitiveRes.TexturePath));

            this.Size = stream.Position - this.Position;
            context.AddBinaryWriter(this);

            // 正常終了.
            return true;
        }
    }
}
