﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.BusinessLogic.IO;

using MyPrimitiveManager = EffectMaker.BusinessLogic.Manager.PrimitiveManager;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// g3dのプリミティブテーブルライタです。
    /// </summary>
    public class G3dPrimitiveTableWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="emitterSets">The emitterSet</param>
        public G3dPrimitiveTableWriter(G3dPrimitiveArrayData primitiveArray)
        {
            this.DataModel = primitiveArray.TableData;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the primitive resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var primitiveArray = this.DataModel as G3dPrimitiveTableData;
            if (primitiveArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            if (primitiveArray.PrimitivePaths.Any())
            {
                // uint64_t guid;          //!< ハッシュコードと衝突カウンタからなるGUID
                // uint32_t nextOffset;    //!< 次のレコードへのオフセット
                // uint32_t nameLength;    //!< 名前の長さ(今は利用していません。バイナリの互換性保持のためにあります。)
                // uint8_t  positionIndex; //!< posデータのインデックス
                // uint8_t  normalIndex;   //!< 法線データのインデックス
                // uint8_t  tangentIndex;  //!< 接線データのインデックス
                // uint8_t  colorIndex;    //!< 頂点カラーのインデックス
                // uint8_t  uv0Index;      //!< UV0のインデックス
                // uint8_t  uv1Index;      //!< UV1のインデックス
                // char     dummy[2];      //!< アライメント

                var primitivePaths = primitiveArray.PrimitivePaths.Select((value, index) => new { value, index });

                foreach (var primitivePath in primitivePaths)
                {
                    int lengthWithNull = 8;
                    int nextOffset = primitivePath.index + 1 == primitivePaths.Count() ? 0 : 24;

                    BinaryConversionUtility.ForResource.WriteStream(
                        stream,
                        Manager.PrimitiveManager.Instance.GetGuid(primitivePath.value));
                    BinaryConversionUtility.ForResource.WriteStream(stream, nextOffset);
                    BinaryConversionUtility.ForResource.WriteStream(stream, lengthWithNull);

                    var data = MyPrimitiveManager.Instance.LoadModel(primitivePath.value);
                    var modelData = data.ModelData;

                    byte positionAttributeIndex = (modelData.Position.ElementData.Length > 0 ? (byte)modelData.Position.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, positionAttributeIndex);

                    byte normalAttributeIndex = (modelData.Normal.ElementData.Length > 0 ? (byte)modelData.Normal.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, normalAttributeIndex);

                    byte tangentAttributeIndex = (modelData.Tangent.ElementData.Length > 0 ? (byte)modelData.Tangent.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, tangentAttributeIndex);

                    byte colorAttributeIndex = (modelData.Color.ElementData.Length > 0 ? (byte)modelData.Color.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, colorAttributeIndex);

                    byte texCoord0AttributeIndex = (modelData.TexCoord0.ElementData.Length > 0 ? (byte)modelData.TexCoord0.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, texCoord0AttributeIndex);

                    byte texCoord1AttributeIndex = (modelData.TexCoord1.ElementData.Length > 0 ? (byte)modelData.TexCoord1.AttributeIndex : byte.MaxValue);
                    BinaryConversionUtility.ForResource.WriteStream(stream, texCoord1AttributeIndex);

                    BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, 2));
                }
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
