﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Binary header helper for field data binary structures.
    /// </summary>
    public class FieldBinaryHeaderHelper : IBinaryHeaderHelper
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public FieldBinaryHeaderHelper()
        {
        }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return false; }
        }

        /// <summary>
        /// Get the alignment inserts before the next element.
        /// </summary>
        public uint AlignmentToNext
        {
            get { return 0; }
        }

        /// <summary>
        /// Check if the given data model is accepted for this binary header helper.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>True if accepted.</returns>
        public bool IsAccepted(DataModelBase dataModel)
        {
            if (dataModel is FieldDataBase)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Get tag string for the binary header.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The tag string.</returns>
        public string GetTag(DataModelBase dataModel)
        {
            if (dataModel is AddLocationData)
            {
                return "FPAD";
            }
            else if (dataModel is CollisionData)
            {
                return "FCOL";
            }
            else if (dataModel is ConvergeData)
            {
                return "FCOV";
            }
            else if (dataModel is CurlNoiseData)
            {
                return "FCLN";
            }
            else if (dataModel is MagnetData)
            {
                return "FMAG";
            }
            else if (dataModel is RandomData)
            {
                return "FRND";
            }
            else if (dataModel is RandomFe1Data)
            {
                return "FRN1";
            }
            else if (dataModel is SpinData)
            {
                return "FSPN";
            }
            else if (dataModel is CustomFieldData)
            {
                return "FCSF";
            }
            else
            {
                throw new ArgumentException("Unrecognized data model type.");
            }
        }

        /// <summary>
        /// Getting the next of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The next data model or null if the given is the last.</returns>
        public DataModelBase GetNext(DataModelBase current)
        {
            // フィールドを取得
            var field = current as FieldDataBase;
            if (field == null)
            {
                Logger.Log(LogLevels.Debug, "currentが不正");
                return null;
            }

            // 親エミッタを取得
            var emitter = field.Parent as EmitterData;
            if (emitter == null)
            {
                Logger.Log(LogLevels.Debug, "Parentが不正");
                return null;
            }

            // 親エミッタが持っているフィールドのリストを取得
            IEnumerable<FieldDataBase> list = emitter.ActiveFieldList;
            if (list == null)
            {
                Logger.Log(LogLevels.Debug, "ActiveFieldListが不正");
                return null;
            }

            // フィールドのインデックスを取得
            int index = 0;
            foreach (FieldDataBase item in list)
            {
                if (item == field)
                {
                    break;
                }

                ++index;
            }

            // 次のフィールドがあればそれを返す
            if (index < list.Count() - 1)
            {
                return list.ElementAt(index + 1);
            }

            // なければAnimeを返す
            return EmitterAnimationBinaryHeaderHelper.GetFirstEnableAnime(emitter);
        }

        /// <summary>
        /// Getting the first child of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first child data model or null if the child does not exist.</returns>
        public DataModelBase GetChild(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the first data model of the given data model's sub tree.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first data model of the sub tree or null there is no sub tree.</returns>
        public DataModelBase GetSub(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the offset from the binary header to the binary data.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <param name="headerPos">The position of the binary header.</param>
        /// <returns>The offset from the binary header to the binary data.</returns>
        public uint GetOffset(DataModelBase current, uint headerPos)
        {
            return BinaryStructHeader.Size;
        }
    }
}
