﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.BinaryHeaders
{
    /// <summary>
    /// Binary structure header class.
    /// </summary>
    public class BinaryStructHeader
    {
        /// <summary>An empty binary header.</summary>
        private static readonly BinaryStructHeader EmptyHeader = new BinaryStructHeader();

        /// <summary>The tag characters.</summary>
        private char[] tag = new char[] { '\0', '\0', '\0', '\0' };

        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryStructHeader()
        {
            this.Reset();
        }

        /// <summary>
        /// Get an empty binary header.
        /// </summary>
        public static BinaryStructHeader Empty
        {
            get { return EmptyHeader; }
        }

        /// <summary>
        /// Get the size of the binary header.
        /// </summary>
        public static uint Size
        {
            get { return 32; }
        }

        /// <summary>
        /// Get or set the tag of the binary header.
        /// </summary>
        /// <remarks>
        /// The tag always consists of 4 characters.
        /// If the string that being set is longer than 4 characters, it will be clamped.
        /// If the string is shorter than 4 characters, the rest will be filled in with '\0'.
        /// </remarks>
        public string Tag
        {
            get
            {
                return new string(this.tag);
            }

            set
            {
                var str = string.Empty;
                if (string.IsNullOrEmpty(value) == false)
                {
                    str = value.Substring(0, Math.Min(4, value.Length));
                }

                for (int i = 0; i < 4; ++i)
                {
                    if (i < str.Length)
                    {
                        this.tag[i] = str[i];
                    }
                    else
                    {
                        this.tag[i] = '\0';
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the size of the binary data.
        /// </summary>
        public uint BinarySize { get; set; }

        /// <summary>
        /// Get or set the offset from the beginning of the header to the child binary data.
        /// </summary>
        public uint Child { get; set; }

        /// <summary>
        /// Get or set the offset from the beginning of the header to the next binary data.
        /// </summary>
        public uint Next { get; set; }

        /// <summary>
        /// Get or set the offset from the beginning of the header to the sub tree binary data.
        /// </summary>
        public uint Sub { get; set; }

        /// <summary>
        /// Get or set the offset from the beginning of the header to the binary data.
        /// </summary>
        public uint Offset { get; set; }

        /// <summary>
        /// バイナリツリー全体の通し番号が入ります。当面はテクスチャリソースのID用として使います。
        /// </summary>
        public uint Guid { get; set; }

        /// <summary>
        /// 子バイナリの個数です。
        /// </summary>
        public ushort ChildNum { get; set; }

        /// <summary>
        /// Reset the data to the initial states.
        /// </summary>
        public void Reset()
        {
            this.tag[0] = '\0';
            this.tag[1] = '\0';
            this.tag[2] = '\0';
            this.tag[3] = '\0';

            this.BinarySize = 0;
            this.Child = 0xFFFFFFFF;
            this.Next = 0xFFFFFFFF;
            this.Sub = 0xFFFFFFFF;
            this.Guid = 0;
            this.ChildNum = 0;

            // The size of the header is 32 bytes. (with 8 bytes of dummy data)
            this.Offset = 0x00000020;
        }

        /// <summary>
        /// Write the header to the given stream.
        /// </summary>
        /// <param name="stream">The stream to write to.</param>
        public void Write(Stream stream)
        {
            BinaryConversionUtility.ForResource.WriteStream(stream, this.tag);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.BinarySize);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.Child);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.Next);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.Sub);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.Offset);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.Guid);
            BinaryConversionUtility.ForResource.WriteStream(stream, this.ChildNum);

            // フリップフラグはこちらから送り出す時は常に0;
            BinaryConversionUtility.ForResource.WriteStream(stream, (byte)0);

            // エンディアンはバイナリ全体のCpuEndianと同一
            var endian = (byte)(BinaryConversionUtility.ForResource.IsLittleEndian ? 0 : 1);
            BinaryConversionUtility.ForResource.WriteStream(stream, endian);
        }
    }
}
