﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using Microsoft.Win32;

namespace EffectMaker.Foundation.Win32
{
    /// <summary>
    /// ユーティリティクラス。
    /// </summary>
    public static class Utility
    {
        /// <summary>
        /// 下位16ビットを取得。
        /// </summary>
        /// <param name="n">32ビット値</param>
        /// <returns>下位16ビット</returns>
        public static int LOWORD(int n)
        {
            return n & 0xffff;
        }

        /// <summary>
        /// 下位16ビットを取得。
        /// </summary>
        /// <param name="p">32ビット値</param>
        /// <returns>下位16ビット</returns>
        public static int LOWORD(IntPtr p)
        {
            return LOWORD((int)(long)p);
        }

        /// <summary>
        /// 上位16ビットを取得。
        /// </summary>
        /// <param name="n">32ビット値</param>
        /// <returns>上位16ビット</returns>
        public static int HIWORD(int n)
        {
            return (n >> 0x10) & 0xffff;
        }

        /// <summary>
        /// 上位16ビットを取得。
        /// </summary>
        /// <param name="p">32ビット値</param>
        /// <returns>上位16ビット</returns>
        public static int HIWORD(IntPtr p)
        {
            return HIWORD((int)(long)p);
        }

        /// <summary>
        /// 符号付き下位16ビットを取得。
        /// </summary>
        /// <param name="n">32ビット値</param>
        /// <returns>符号付き下位16ビット</returns>
        public static int SignedLOWORD(int n)
        {
            return (short)(n & 0xffff);
        }

        /// <summary>
        /// 符号付き下位16ビットを取得。
        /// </summary>
        /// <param name="p">32ビット値</param>
        /// <returns>符号付き下位16ビット</returns>
        public static int SignedLOWORD(IntPtr p)
        {
            return SignedLOWORD((int)(long)p);
        }

        /// <summary>
        /// 符号付き上位16ビットを取得。
        /// </summary>
        /// <param name="n">32ビット値</param>
        /// <returns>符号付き上位16ビット</returns>
        public static int SignedHIWORD(int n)
        {
            return (short)((n >> 0x10) & 0xffff);
        }

        /// <summary>
        /// 符号付き上位16ビットを取得。
        /// </summary>
        /// <param name="p">32ビット値</param>
        /// <returns>符号付き上位16ビット</returns>
        public static int SignedHIWORD(IntPtr p)
        {
            return SignedHIWORD((int)(long)p);
        }

        /// <summary>
        /// LONG 値を作成。
        /// </summary>
        /// <param name="low">下位16ビット</param>
        /// <param name="high">上位16ビット</param>
        /// <returns>LONG 値</returns>
        public static int MAKELONG(int low, int high)
        {
            return (high << 0x10) | (low & 0xffff);
        }

        /// <summary>
        /// LParam を作成。
        /// </summary>
        /// <param name="low">下位16ビット</param>
        /// <param name="high">上位16ビット</param>
        /// <returns>LParam</returns>
        public static IntPtr MAKELPARAM(int low, int high)
        {
            return (IntPtr)((high << 0x10) | (low & 0xffff));
        }

        /// <summary>
        /// LParam を位置情報に変換。
        /// LParam に位置情報が送られるウィンドウメッセージで使用します。
        /// </summary>
        /// <param name="lparam">位置情報が入ったLParam</param>
        /// <returns>位置情報</returns>
        public static Point LParamToPoint(IntPtr lparam)
        {
            return new Point(LOWORD(lparam), HIWORD(lparam));
        }

        /// <summary>
        /// LParam を符号付き位置情報に変換。
        /// LParam に位置情報が送られるウィンドウメッセージで使用します。
        /// </summary>
        /// <param name="lparam">位置情報が入ったLParam</param>
        /// <returns>符号付き位置情報</returns>
        public static Point LParamToSignedPoint(IntPtr lparam)
        {
            return new Point(SignedLOWORD(lparam), SignedHIWORD(lparam));
        }

        /// <summary>
        /// OSのテーマが有効になっているか否かを取得します。
        /// クラシックスタイルを利用している場合はテーマが無効となり、
        /// タブのデザインを変更しようとすると例外になるため、このプロパティで処理を分岐します。
        /// </summary>
        public static bool IsThemeActive
        {
            get
            {
                var reg = Registry.CurrentUser.OpenSubKey(@"Software\Microsoft\Windows\CurrentVersion\ThemeManager");
                if (reg != null)
                {
                    var val = (string)reg.GetValue("ThemeActive");
                    return !val.Equals("0");
                }

                return false;
            }
        }
    }
}
