﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
// <auto-generated />
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// 連続的な値変更イベントハンドラデリゲート。
    /// </summary>
    /// <param name="sender">The sender of the event.</param>
    /// <param name="e">The event arguments.</param>
    public delegate void SequentialValueChangedEventHandler(object sender, SequentialValueChangedEventArgs e);

    /// <summary>
    /// カラー成分のモード。
    /// </summary>
    public enum ColorModeType : int
    {
        /// <summary>RGBA設定。</summary>
        RGBA,
        /// <summary>RGB設定。</summary>
        RGB,
        /// <summary>sRGBA設定。</summary>
        sRGBA,
        /// <summary>sRGB設定。</summary>
        sRGB,
        /// <summary>HSVA設定。</summary>
        HSVA,
        /// <summary>HSV設定。</summary>
        HSV,
        /// <summary>A設定。</summary>
        A
    };

    /// <summary>
    /// 色要素対象
    /// </summary>
    public enum ColorElementType
    {
        /// <summary>
        /// 色
        /// </summary>
        Color,

        /// <summary>
        /// アルファ
        /// </summary>
        Alpha,

        /// <summary>
        /// 色
        /// </summary>
        GlobalColor,

        /// <summary>
        /// アルファ
        /// </summary>
        GlobalAlpha,
    }


    /// <summary>
    /// チャンネルフラグ
    /// </summary>
    [Flags]
    public enum ColorChannelFlags
    {
        /// <summary>
        /// None
        /// </summary>
        None = 0x00,

        /// <summary>
        /// R
        /// </summary>
        R = 0x01,

        /// <summary>
        /// G
        /// </summary>
        G = 0x02,

        /// <summary>
        /// B
        /// </summary>
        B = 0x04,

        /// <summary>
        /// アルファ
        /// </summary>
        A = 0x08,

        /// <summary>
        /// RGB
        /// </summary>
        Rgb = R | G | B,

        /// <summary>
        /// RGBA
        /// </summary>
        Rgba = R | G | B | A
    }

    /// <summary>
    /// 連続的な値変更イベントデータクラス。
    /// </summary>
    public sealed class SequentialValueChangedEventArgs : EventArgs
    {
        /// <summary>
        /// 変更中.
        /// </summary>
        private readonly bool changing;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="changing">変更中であれば true.</param>
        public SequentialValueChangedEventArgs(bool changing)
        {
            this.changing = changing;
        }

        /// <summary>
        /// 変更中かどうか。
        /// </summary>
        public bool Changing
        {
            get { return this.changing; }
        }

        /// <summary>
        /// 要素インデックス
        /// </summary>
        public uint ElementBits { get; set; }
    }

    /// <summary>
    /// Class providing utility methods for color manipulations.
    /// </summary>
    public static class ColorUtility
    {
        #region Static member variables

        /// <summary>
        /// gammaEnabled
        /// </summary>
        private static bool gammaEnabled = false;

        /// <summary>
        /// gamma
        /// </summary>
        private static double gamma = 0.0;

        /// <summary>
        /// gammaRampTable
        /// </summary>
        private static byte[] gammaRampTable = new byte[256];

        /// <summary>
        /// invGammaRampTable
        /// </summary>
        private static byte[] invGammaRampTable = new byte[256];

        /// <summary>
        /// textBoxSelectAllOnClick
        /// </summary>
        private static bool textBoxSelectAllOnClick = true;

        #endregion

        #region Static properties

        /// <summary>
        /// Get or set the gamma value.
        /// </summary>
        public static double Gamma
        {
            get
            {
                return gamma;
            }

            set
            {
                if (gamma == value)
                {
                    return;
                }

                gamma = value;
                CreateGammaRampTable(gamma);
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether gamma correction is enabled.
        /// </summary>
        public static bool IsGammaCorrectionEnabled
        {
            get { return gammaEnabled; }
            set { gammaEnabled = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to select all text while
        /// clicked on the text box.
        /// </summary>
        public static bool TextBoxSelectAllOnClick
        {
            get { return textBoxSelectAllOnClick; }
            set { textBoxSelectAllOnClick = value; }
        }

        #endregion

        #region Gamma correction
        /// <summary>
        /// Compute gamma corrected color with the given color value and gamma.
        /// </summary>
        /// <param name="origValue">The original color value.</param>
        /// <param name="gamma">The gamma.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ComputeGamma(double origValue, double gamma)
        {
            double value = (255.0 * System.Math.Pow(origValue / 255.0, 1.0 / gamma)) + 0.5;

            if (value < 0.5)
            {
                return 0;
            }

            if (value >= 255.5)
            {
                return 255;
            }

            return (byte)value;
        }

        /// <summary>
        /// Compute inversely gamma corrected color with the given color value and gamma.
        /// </summary>
        /// <param name="origValue">元の値</param>
        /// <param name="gamma">ガンマ値。</param>
        /// <returns>InvGamma を返します。</returns>
        public static byte ComputeInvGamma(double origValue, double gamma)
        {
            double value = (255.0 * System.Math.Pow(origValue / 255.0, gamma)) + 0.5;

            if (value < 0.5)
            {
                return 0;
            }

            if (value >= 255.5)
            {
                return 255;
            }

            return (byte)value;
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorGamma(byte color, bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return color;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return color;
                }
            }

            return gammaRampTable[color];
        }

        /// <summary>
        /// Apply inverse gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorInvGamma(byte color, bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return color;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return color;
                }
            }

            return invGammaRampTable[color];
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorGamma(int color, bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return (byte)ClampByte(color);
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return (byte)ClampByte(color);
                }
            }

            return gammaRampTable[ClampByte(color)];
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorGamma(float color, bool forceGamma = false)
        {
            byte colorIdx = (byte)ClampByte(color);

            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return colorIdx;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return colorIdx;
                }
            }

            return gammaRampTable[colorIdx];
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorInvGamma(float color, bool forceGamma = false)
        {
            byte colorIdx = (byte)ClampByte(color);

            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return colorIdx;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return colorIdx;
                }
            }

            return invGammaRampTable[colorIdx];
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value.</returns>
        public static byte ApplyColorGamma(double color, bool forceGamma = false)
        {
            byte colorIdx = (byte)ClampByte(color);

            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return colorIdx;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return colorIdx;
                }
            }

            return gammaRampTable[colorIdx];
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="r">The color r value to apply gamma correction.</param>
        /// <param name="g">The color g value to apply gamma correction.</param>
        /// <param name="b">The color b value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static int ApplyColorGamma(
                          byte r,
                          byte g,
                          byte b,
                          bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return ((int)r << 16) | ((int)g << 8) | (int)b;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return ((int)r << 16) | ((int)g << 8) | (int)b;
                }
            }

            return ((int)gammaRampTable[r] << 16) |
                   ((int)gammaRampTable[g] << 8) |
                   ((int)gammaRampTable[b]);
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="r">The color r value to apply gamma correction.</param>
        /// <param name="g">The color g value to apply gamma correction.</param>
        /// <param name="b">The color b value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static int ApplyColorGamma(
                          int r,
                          int g,
                          int b,
                          bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return (ClampByte(r) << 16) | (ClampByte(g) << 8) | ClampByte(b);
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return (ClampByte(r) << 16) | (ClampByte(g) << 8) | ClampByte(b);
                }
            }

            return ((int)gammaRampTable[ClampByte(r)] << 16) |
                   ((int)gammaRampTable[ClampByte(g)] << 8) |
                   ((int)gammaRampTable[ClampByte(b)]);
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="r">The color r value to apply gamma correction.</param>
        /// <param name="g">The color g value to apply gamma correction.</param>
        /// <param name="b">The color b value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static int ApplyColorGamma(
                          float r, float g, float b, bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return ((int)ClampByte(r) << 16) | ((int)ClampByte(g) << 8) | (int)ClampByte(b);
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return ((int)ClampByte(r) << 16) | ((int)ClampByte(g) << 8) | (int)ClampByte(b);
                }
            }

            return ((int)gammaRampTable[(int)ClampByte(r)] << 16) |
                   ((int)gammaRampTable[(int)ClampByte(g)] << 8) |
                   ((int)gammaRampTable[(int)ClampByte(b)]);
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="r">The color r value to apply gamma correction.</param>
        /// <param name="g">The color g value to apply gamma correction.</param>
        /// <param name="b">The color b value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static int ApplyColorGamma(
                          double r,
                          double g,
                          double b,
                          bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return ((int)ClampByte(r) << 16) | ((int)ClampByte(g) << 8) | (int)ClampByte(b);
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return ((int)ClampByte(r) << 16) | ((int)ClampByte(g) << 8) | (int)ClampByte(b);
                }
            }

            return ((int)gammaRampTable[(int)ClampByte(r)] << 16) |
                   ((int)gammaRampTable[(int)ClampByte(g)] << 8) |
                   ((int)gammaRampTable[(int)ClampByte(b)]);
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="alpha">The alpha value to apply gamma correction.</param>
        /// <param name="rgb">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static System.Drawing.Color ApplyColorGamma(
                                           byte alpha,
                                           System.Drawing.Color rgb,
                                           bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return System.Drawing.Color.FromArgb(alpha, rgb);
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return System.Drawing.Color.FromArgb(alpha, rgb);
                }
            }

            return System.Drawing.Color.FromArgb(
                                        alpha,
                                        gammaRampTable[rgb.R],
                                        gammaRampTable[rgb.G],
                                        gammaRampTable[rgb.B]);
        }

        /// <summary>
        /// Apply gamma correction to the given color with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="color">The color value to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <returns>The corrected color value, encoded to an 32bits integer.</returns>
        public static ColorRgba ApplyColorGamma(
                                ColorRgba color,
                                bool forceGamma = false)
        {
            // Has the gamma ramp table being created yet?
            if (gamma <= 0.0)
            {
                return color;
            }

            // Are we applying any gamma correction?
            if (forceGamma == false)
            {
                if (IsGammaCorrectionEnabled == false || gamma == 1.0)
                {
                    return color;
                }
            }

            return new ColorRgba(
                       (float)gammaRampTable[(byte)ClampByte(color.R * 255.0f)] / 255.0f,
                       (float)gammaRampTable[(byte)ClampByte(color.G * 255.0f)] / 255.0f,
                       (float)gammaRampTable[(byte)ClampByte(color.B * 255.0f)] / 255.0f,
                       color.A);
        }

        /// <summary>
        /// Apply gamma correction to the given bitmap with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <param name="applyToAlpha">Apply to the alpha color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true,
                           bool applyToAlpha = false)
        {
            if (img.PixelFormat == System.Drawing.Imaging.PixelFormat.Format24bppRgb)
            {
                return ApplyRGBImageGamma(
                       img,
                       forceGamma,
                       applyToRed,
                       applyToGreen,
                       applyToBlue);
            }
            else if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format32bppArgb)
            {
                return ApplyARGBImageGamma(
                       img,
                       forceGamma,
                       applyToRed,
                       applyToGreen,
                       applyToBlue,
                       applyToAlpha);
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Apply gamma correction to the given bitmap with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyRGBImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true)
        {
            if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format24bppRgb)
            {
                return false;
            }

            // Lock the image to edit the pixels.
            System.Drawing.Imaging.BitmapData bmpData =
                img.LockBits(
                    new System.Drawing.Rectangle(0, 0, img.Width, img.Height),
                    System.Drawing.Imaging.ImageLockMode.ReadWrite,
                    System.Drawing.Imaging.PixelFormat.Format24bppRgb);

            if (bmpData == null)
            {
                return false;
            }

#if false // TODO:コード復帰
            unsafe
            {
                byte* buffer = (byte*)(void*)bmpData.Scan0;

                int iNumBytes = img.Width * img.Height * 3;
                for (int i = 0; i < iNumBytes; ++i)
                {
                    if ((i % 3 == 0 && applyToRed == true) ||
                         (i % 3 == 1 && applyToGreen == true) ||
                         (i % 3 == 2 && applyToBlue == true))
                    {
                        buffer[i] = ApplyColorGamma(buffer[i], true);
                    }
                }
            }
#endif

            // Commit the modification to the pixels.
            img.UnlockBits(bmpData);

            return false;
        }

        /// <summary>
        /// Apply gamma correction to the given bitmap with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <param name="applyToAlpha">Apply to the alpha color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyARGBImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true,
                           bool applyToAlpha = false)
        {
            if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format32bppArgb)
            {
                return false;
            }

            // Lock the image to edit the pixels.
            System.Drawing.Imaging.BitmapData bmpData =
                img.LockBits(
                    new System.Drawing.Rectangle(0, 0, img.Width, img.Height),
                    System.Drawing.Imaging.ImageLockMode.ReadWrite,
                    System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            if (bmpData == null)
            {
                return false;
            }

#if false // TODO:コード復帰
            unsafe
            {
                byte* buffer = (byte*)(void*)bmpData.Scan0;

                int iNumBytes = img.Width * img.Height * 4;
                for (int i = 0; i < iNumBytes; ++i)
                {
                    if ((i % 4 == 0 && applyToAlpha == true) ||
                         (i % 4 == 1 && applyToRed == true) ||
                         (i % 4 == 2 && applyToGreen == true) ||
                         (i % 4 == 3 && applyToBlue == true))
                    {
                        buffer[i] = ApplyColorGamma(buffer[i], true);
                    }
                }
            }
#endif

            // Commit the modification to the pixels.
            img.UnlockBits(bmpData);

            return false;
        }

        /// <summary>
        /// Apply inverse gamma correction to the given bitmap with
        /// the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply inverse gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <param name="applyToAlpha">Apply to the alpha color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyInvImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true,
                           bool applyToAlpha = false)
        {
            if (img.PixelFormat == System.Drawing.Imaging.PixelFormat.Format24bppRgb)
            {
                return ApplyInvRGBImageGamma(
                       img,
                       forceGamma,
                       applyToRed,
                       applyToGreen,
                       applyToBlue);
            }
            else if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format32bppArgb)
            {
                return ApplyInvARGBImageGamma(
                       img,
                       forceGamma,
                       applyToRed,
                       applyToGreen,
                       applyToBlue,
                       applyToAlpha);
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Apply inverse gamma correction to the given bitmap with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply inverse gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyInvRGBImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true)
        {
            if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format24bppRgb)
            {
                return false;
            }

            // Lock the image to edit the pixels.
            System.Drawing.Imaging.BitmapData bmpData =
                img.LockBits(
                    new System.Drawing.Rectangle(0, 0, img.Width, img.Height),
                    System.Drawing.Imaging.ImageLockMode.ReadWrite,
                    System.Drawing.Imaging.PixelFormat.Format24bppRgb);

            if (bmpData == null)
            {
                return false;
            }

#if false // TODO:コード復帰
            unsafe
            {
                byte* buffer = (byte*)(void*)bmpData.Scan0;

                int iNumBytes = img.Width * img.Height * 3;
                for (int i = 0; i < iNumBytes; ++i)
                {
                    if ((i % 3 == 0 && applyToRed == true) ||
                         (i % 3 == 1 && applyToGreen == true) ||
                         (i % 3 == 2 && applyToBlue == true))
                    {
                        buffer[i] = ApplyColorInvGamma(buffer[i], true);
                    }
                }
            }
#endif

            // Commit the modification to the pixels.
            img.UnlockBits(bmpData);

            return false;
        }

        /// <summary>
        /// Apply inverse gamma correction to the given bitmap with the precomputed gamma ramp table.
        /// </summary>
        /// <param name="img">The image to apply inverse gamma correction.</param>
        /// <param name="forceGamma">True to force enabling gamma correction.</param>
        /// <param name="applyToRed">Apply to the red color byte of the pixels.</param>
        /// <param name="applyToGreen">Apply to the green color byte of the pixels.</param>
        /// <param name="applyToBlue">Apply to the blue color byte of the pixels.</param>
        /// <param name="applyToAlpha">Apply to the alpha color byte of the pixels.</param>
        /// <returns>True on success.</returns>
        public static bool ApplyInvARGBImageGamma(
                           System.Drawing.Bitmap img,
                           bool forceGamma = false,
                           bool applyToRed = true,
                           bool applyToGreen = true,
                           bool applyToBlue = true,
                           bool applyToAlpha = true)
        {
            if (img.PixelFormat != System.Drawing.Imaging.PixelFormat.Format32bppArgb)
            {
                return false;
            }

            // Lock the image to edit the pixels.
            System.Drawing.Imaging.BitmapData bmpData =
                img.LockBits(
                    new System.Drawing.Rectangle(0, 0, img.Width, img.Height),
                    System.Drawing.Imaging.ImageLockMode.ReadWrite,
                    System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            if (bmpData == null)
            {
                return false;
            }

#if false // TODO:コード復帰
            unsafe
            {
                byte* buffer = (byte*)(void*)bmpData.Scan0;

                int iNumBytes = img.Width * img.Height * 4;
                for (int i = 0; i < iNumBytes; ++i)
                {
                    if ((i % 4 == 0 && applyToAlpha == true) ||
                         (i % 4 == 1 && applyToRed == true) ||
                         (i % 4 == 2 && applyToGreen == true) ||
                         (i % 4 == 3 && applyToBlue == true))
                    {
                        buffer[i] = ApplyColorInvGamma(buffer[i], true);
                    }
                }
            }
#endif

            // Commit the modification to the pixels.
            img.UnlockBits(bmpData);

            return false;
        }

        /// <summary>
        /// Create gamma ramp table with the specified gamma.
        /// </summary>
        /// <param name="gamma">The gamma.</param>
        public static void CreateGammaRampTable(double gamma)
        {
            if (gamma == 1.0)
            {
                for (int i = 0; i < 256; ++i)
                {
                    gammaRampTable[i] = (byte)i;
                    invGammaRampTable[i] = (byte)i;
                }
            }
            else
            {
                for (int i = 0; i < 256; ++i)
                {
                    gammaRampTable[i] = ComputeGamma((double)i, gamma);
                    invGammaRampTable[i] = ComputeInvGamma((double)i, gamma);
                }
            }
        }

        #endregion

        #region Math utility

        /// <summary>
        /// Clamp the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static double Clamp(
                             double value,
                             double min,
                             double max)
        {
            return System.Math.Min(System.Math.Max(value, min), max);
        }

        /// <summary>
        /// Clamp the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static float Clamp(
                            float value,
                            float min,
                            float max)
        {
            return System.Math.Min(System.Math.Max(value, min), max);
        }

        /// <summary>
        /// Clamp the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static int Clamp(int value, int min, int max)
        {
            return System.Math.Min(System.Math.Max(value, min), max);
        }

        /// <summary>
        /// Clamp the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static byte Clamp(byte value, byte min, byte max)
        {
            return System.Math.Min(System.Math.Max(value, min), max);
        }

        /// <summary>
        /// Clamp the value to 0 - 255.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <returns>The clamped value.</returns>
        public static double ClampByte(double value)
        {
            return System.Math.Min(System.Math.Max(value, 0.0), 255.0);
        }

        /// <summary>
        /// Clamp the value to 0 - 255.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <returns>The clamped value.</returns>
        public static float ClampByte(float value)
        {
            return System.Math.Min(System.Math.Max(value, 0.0f), 255.0f);
        }

        /// <summary>
        /// Clamp the value to 0 - 255.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <returns>The clamped value.</returns>
        public static int ClampByte(int value)
        {
            return System.Math.Min(System.Math.Max(value, 0), 255);
        }

        /// <summary>
        /// Clamp the value to 0 - 255.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <returns>The clamped value.</returns>
        public static byte ClampByte(byte value)
        {
            return System.Math.Min(System.Math.Max(value, (byte)0), (byte)255);
        }

        /// <summary>
        /// Convert color to another format.
        /// </summary>
        /// <param name="color">The color to convert.</param>
        /// <returns>The converted color.</returns>
        public static int ConvertColor(ColorRgba color)
        {
            int r = (int)ClampByte(color.R * 255.0f);
            int g = (int)ClampByte(color.G * 255.0f);
            int b = (int)ClampByte(color.B * 255.0f);
            return (r << 16) | (g << 8) | b;
        }

        #endregion

        /// <summary>
        /// Convert color to a windows native color format.
        /// </summary>
        /// <param name="color">The color to convert.</param>
        /// <returns>The converted color.</returns>
        public static System.Drawing.Color ToWinColor(ColorRgba color)
        {
            return System.Drawing.Color.FromArgb(
                                        (byte)ClampByte(color.A * 255.0f),
                                        (byte)ClampByte(color.R * 255.0f),
                                        (byte)ClampByte(color.G * 255.0f),
                                        (byte)ClampByte(color.B * 255.0f));
        }

        /// <summary>
        /// ColorHsva から ColorRgba に変換します。
        /// </summary>
        /// <param name="src">変換対象の値。</param>
        /// <returns>変換された値。</returns>
        public static ColorRgba ToColorRgba(float h, float s, float v, float a)
        {
            ColorRgba rgba = new ColorRgba();

            if (s == 0.0f)
            {
                rgba.Set(v, v, v, a);
                return rgba;
            }

            if (h < 0.0f)
            {
                h = 0.0f;
            }
            else if (1.0f < h)
            {
                h = 1.0f;
            }

            if (h == 1.0f)
            {
                h = 0.0f;
            }

            h = h * 6.0f;
            int hi = ((int)h) % 6;

            float f = h - hi;
            float p = v * (1 - s);
            float q = v * (1 - (f * s));
            float t = v * (1 - ((1 - f) * s));

            switch (hi)
            {
                case 0:
                    rgba.Set(v, t, p, a);
                    return rgba;
                case 1:
                    rgba.Set(q, v, p, a);
                    return rgba;
                case 2:
                    rgba.Set(p, v, t, a);
                    return rgba;
                case 3:
                    rgba.Set(p, q, v, a);
                    return rgba;
                case 4:
                    rgba.Set(t, p, v, a);
                    return rgba;
                case 5:
                    rgba.Set(v, p, q, a);
                    return rgba;
                default:
                    rgba.Set(0.0f, 0.0f, 0.0f, a);
                    return rgba;
            }
        }

        /// <summary>
        /// ColorHsva から ColorRgba に変換します。
        /// </summary>
        /// <param name="src">変換対象の値。</param>
        /// <returns>変換された値。</returns>
        public static ColorRgba ToColorRgba(ColorHsva src)
        {
            return ToColorRgba(src.H, src.S, src.V, src.A);
        }

        /// <summary>
        /// Convert color to a floating point color format.
        /// </summary>
        /// <param name="color">The color to convert.</param>
        /// <returns>The converted color.</returns>
        public static ColorRgba ToColorRgba(System.Drawing.Color color)
        {
            return new ColorRgba(
                       (float)color.R / 255.0f,
                       (float)color.G / 255.0f,
                       (float)color.B / 255.0f,
                       (float)color.A / 255.0f);
        }

        /// <summary>
        /// ColorHsva から ColorRgba に変換します。
        /// </summary>
        /// <param name="src">変換対象の値。</param>
        /// <returns>変換された値。</returns>
        public static ColorHsva ToColorHsva(ColorRgba src)
        {
            ColorHsva hsva = new ColorHsva();

            float r = src.R;
            float b = src.B;
            float g = src.G;
            float a = src.A;

            // HDR 対応のため 0 - 1 範囲へ収める処理はしない
            //r = Math.Max(0.0f, Math.Min(1.0f, r));
            //g = Math.Max(0.0f, Math.Min(1.0f, g));
            //b = Math.Max(0.0f, Math.Min(1.0f, b));

            float max = Math.Max(Math.Max(r, g), b);
            float min = Math.Min(Math.Min(r, g), b);

            if (max == 0.0f)
            {
                hsva.Set(0.0f, 0.0f, 0.0f, a);
                return hsva;
            }

            if (max == min)
            {
                hsva.Set(0.0f, 0.0f, max, a);
                return hsva;
            }

            float h = 0.0f;

            if (max == r)
            {
                h = ((g - b) / (max - min)) + 0.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }
            else if (max == g)
            {
                h = ((b - r) / (max - min)) + 2.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }
            else
            {
                h = ((r - g) / (max - min)) + 4.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }

            float s = (max - min) / max;
            float v = max;

            hsva.Set(h, s, v, a);

            return hsva;
        }

        /// <summary>
        /// RGBの中で1より大きな値のチャンネルがあったら除算して正規化します
        /// </summary>
        /// <param name="color">カラー</param>
        /// <returns>正規化されたカラー</returns>
        public static ColorRgba NormalizeColor(ColorRgba color)
        {
            var maxElem = Math.Max(color.R, Math.Max(color.G, color.B));
            if (maxElem > 1.0f)
            {
                return new ColorRgba(color.R / maxElem, color.G / maxElem, color.B / maxElem, color.A);
            }

            return color;
        }

        /// <summary>
        /// 色要素対象タイプを作る
        /// </summary>
        /// <param name="src">元色対象タイプ</param>
        /// <returns>色要素対象タイプ</returns>
        public static ColorElementType MakeColorElementType(ColorTargetType src)
        {
            switch(src)
            {
                case ColorTargetType.Color0:
                case ColorTargetType.Color1:
                    return ColorElementType.Color;

                case ColorTargetType.Alpha0:
                case ColorTargetType.Alpha1:
                    return ColorElementType.Alpha;

                case ColorTargetType.GlobalColor0:
                case ColorTargetType.GlobalColor1:
                    return ColorElementType.GlobalColor;

                case ColorTargetType.GlobalAlpha0:
                case ColorTargetType.GlobalAlpha1:
                    return ColorElementType.GlobalAlpha;
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// ビットマップに対してガンマ変換を行います。
        /// </summary>
        /// <param name="src">元画像</param>
        /// <param name="matrix">カラー行列</param>
        /// <param name="isRgb">RGBチャンネルを処理対象に含むか</param>
        /// <param name="isTexSrgb">テクスチャがSRGBフォーマットか否か</param>
        /// <returns></returns>
        public static Bitmap ConvertBitmapWithGamma(Bitmap src, ColorMatrix matrix, bool isRgb, bool isTexSrgb)
        {
            var image = new Bitmap(src.Width, src.Height, PixelFormat.Format32bppArgb);

            using (var ia = new ImageAttributes())
            using (var g = Graphics.FromImage(image))
            {
                ia.SetColorMatrix(matrix);

                if (isRgb)
                {
                    if (isTexSrgb)
                    {
                        if (ColorUtility.IsGammaCorrectionEnabled == false)
                        {
                            ia.SetGamma((float)ColorUtility.Gamma);
                        }
                    }
                    else
                    {
                        if (ColorUtility.IsGammaCorrectionEnabled)
                        {
                            ia.SetGamma((float)(1.0 / ColorUtility.Gamma));
                        }
                    }
                }

                g.DrawImage(
                    src,
                    new Rectangle(0, 0, src.Width, src.Height),
                    0,
                    0,
                    src.Width,
                    src.Height,
                    GraphicsUnit.Pixel,
                    ia);
            }

            return image;
        }

        /// <summary>
        /// カラーを表す文字列を返します。
        /// </summary>
        /// <returns>カラーを表す文字列です。</returns>
        public static string ToString(ColorRgba rgba)
        {
            return string.Format("R:{0:0.00}, G:{1:0.00}, B:{2:0.00}, A:{3:0.00}", rgba.R, rgba.G, rgba.B, rgba.A);
        }

        /// <summary>
        /// sRGB 値を Linear RGB 値に変換します。 (IEC 61966-2-1:1999)
        /// </summary>
        /// <param name="srgb">sRGB 値</param>
        /// <returns>0 以上 1 以下の Linear RGB 値を返します。</returns>
        public static float SrgbToLinear(float srgb)
        {
            float linear = 0.0f;

            if (srgb <= 0.04045f)
            {
                linear = srgb / 12.92f;
            }
            else
            {
                linear = (float)Math.Pow(((srgb + 0.055) / 1.055), 2.4);
            }

            return Math.Min(Math.Max(0.0f, linear), 1.0f);
        }

        /// <summary>
        /// sRGB カラーを Linear RGB カラーに変換します。
        /// </summary>
        /// <param name="src">sRGB カラー</param>
        /// <returns>Linear RGB カラーを返します。</returns>
        public static ColorRgba SrgbToLinear(ColorRgba src)
        {
            float max = Math.Max(src.R, Math.Max(src.G, src.B));

            ColorRgba color;

            if (max <= 1.0f)
            {
                color = new ColorRgba(SrgbToLinear(src.R), SrgbToLinear(src.G), SrgbToLinear(src.B), src.A);
            }
            else
            {
                color = new ColorRgba(SrgbToLinear(src.R / max) * max, SrgbToLinear(src.G / max) * max, SrgbToLinear(src.B / max) * max, src.A);
            }

            return color;
        }

        /// <summary>
        /// Linear RGB 値を sRGB 値に変換します。 (IEC 61966-2-1:1999)
        /// </summary>
        /// <param name="linear">Linear RGB 値</param>
        /// <returns>0 以上 1 以下の sRGB 値を返します。</returns>
        public static float LinearToSrgb(float linear)
        {
            float srgb = 0.0f;

            if (linear <= 0.0031308f)
            {
                srgb = 12.92f * linear;
            }
            else
            {
                srgb = (1.055f * (float)Math.Pow(linear, 1.0 / 2.4) - 0.055f);
            }

            return Math.Min(Math.Max(0.0f, srgb), 1.0f);
        }

        /// <summary>
        /// Linear RGB カラーを sRGB カラーに変換します。
        /// </summary>
        /// <param name="src">Linear RGB カラー</param>
        /// <returns>sRGB カラーを返します。</returns>
        public static ColorRgba LinearToSrgb(ColorRgba src)
        {
            float max = Math.Max(src.R, Math.Max(src.G, src.B));

            ColorRgba color;

            if (max <= 1.0f)
            {
                color = new ColorRgba(LinearToSrgb(src.R), LinearToSrgb(src.G), LinearToSrgb(src.B), src.A);
            }
            else
            {
                color = new ColorRgba(LinearToSrgb(src.R / max) * max, LinearToSrgb(src.G / max) * max, LinearToSrgb(src.B / max) * max, src.A);
            }

            return color;
        }
    }
}
