﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// XML document serializer.
    /// </summary>
    public class XmlDocSerializer
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public XmlDocSerializer()
        {
        }

        /// <summary>
        /// Serialize the IXmlSerializable object to the provided stream.
        /// </summary>
        /// <param name="stream">The stream.</param>
        /// <param name="obj">The object to serialize.</param>
        /// <returns>True on success.</returns>
        public bool Serialize(Stream stream, IXmlDocSerializable obj)
        {
            const string XmlNamespaceUri = "http://www.w3.org/2000/xmlns/";
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";
            const string XsdNamespaceUri = "http://www.w3.org/2001/XMLSchema";

            var doc = new XmlDocument();

            this.Serialize(doc, obj);
            if (doc.DocumentElement == null)
            {
                return false;
            }

            // Insert XML declaration.
            var xmldecl = doc.CreateXmlDeclaration("1.0", string.Empty, string.Empty);
            doc.InsertBefore(xmldecl, doc.DocumentElement);

            // Add xsi and xsd namespaces to the root element.
            var xsiNamespaceDeclAttr = doc.CreateAttribute("xmlns", "xsi", XmlNamespaceUri);
            xsiNamespaceDeclAttr.Value = XsiNamespaceUri;

            var xsdNamespaceDeclAttr = doc.CreateAttribute("xmlns", "xsd", XmlNamespaceUri);
            xsdNamespaceDeclAttr.Value = XsdNamespaceUri;

            doc.DocumentElement.SetAttributeNode(xsiNamespaceDeclAttr);
            doc.DocumentElement.SetAttributeNode(xsdNamespaceDeclAttr);

            // Save the XML document the object is serialized to.
            doc.Save(stream);

            return true;
        }

        /// <summary>
        /// Serialize the IXmlSerializable object to the provided stream.
        /// </summary>
        /// <param name="doc">The XML document.</param>
        /// <param name="obj">The object to serialize.</param>
        /// <returns>True on success.</returns>
        public bool Serialize(XmlDocument doc, IXmlDocSerializable obj)
        {
            Type objType = obj.GetType();

            doc.AppendChild(doc.CreateElement(objType.Name));

            // Create the serialization context.
            var context = new XmlDocSerializationContext(doc);

            // Serialize.
            context.ReportPreSerialize(obj);
            obj.WriteXml(context);
            context.ReportPostSerialize(obj);

            // Write object type to the root element.
            XmlDocSerializationHelper.WriteElementType(doc.DocumentElement, objType);

            return true;
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <typeparam name="T">The type to deserialize the stream contents to.</typeparam>
        /// <param name="stream">The stream to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public T Deserialize<T>(
            Stream stream,
            IEnumerable<Type> extraTypes = null) where T : IXmlDocSerializable
        {
            if (extraTypes.IsNullOrEmpty() == false)
            {
                var map = new Dictionary<string, Type>();
                extraTypes.ForEach(t => map.Add(t.Name, t));
                return this.Deserialize<T>(stream, map);
            }
            else
            {
                return this.Deserialize<T>(stream, (Dictionary<string, Type>)null);
            }
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <typeparam name="T">The type to deserialize the stream contents to.</typeparam>
        /// <param name="stream">The stream to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public T Deserialize<T>(
            Stream stream,
            Dictionary<string, Type> extraTypes) where T : IXmlDocSerializable
        {
            var doc = new XmlDocument();
            doc.Load(stream);

            return this.Deserialize<T>(doc, extraTypes);
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <typeparam name="T">The type to deserialize the stream contents to.</typeparam>
        /// <param name="doc">The XML document to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public T Deserialize<T>(
            XmlDocument doc,
            IEnumerable<Type> extraTypes) where T : IXmlDocSerializable
        {
            if (extraTypes.IsNullOrEmpty() == false)
            {
                var map = new Dictionary<string, Type>();
                extraTypes.ForEach(t => map.Add(t.Name, t));
                return this.Deserialize<T>(doc, map);
            }
            else
            {
                return this.Deserialize<T>(doc, (Dictionary<string, Type>)null);
            }
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <typeparam name="T">The type to deserialize the stream contents to.</typeparam>
        /// <param name="doc">The XML document to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public T Deserialize<T>(
            XmlDocument doc,
            Dictionary<string, Type> extraTypes) where T : IXmlDocSerializable
        {
            // Create the serialization context.
            var context = new XmlDocSerializationContext(doc, extraTypes);

            Type realType = typeof(T);

            // Read the type written in the root element when it was serialized.
            Type serializedType =
                XmlDocSerializationHelper.ReadElementType(context, doc.DocumentElement);
            if (serializedType != null && serializedType != realType)
            {
                realType = serializedType;
            }

            // Create instance for deserialization.
            var obj = (T)Activator.CreateInstance(realType);

            // Deserialize.
            if (obj.ReadXml(context) == false)
            {
                return default(T);
            }

            // Call the post-deserialization handlers.
            context.ReportPostDeserialize(obj);

            return obj;
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="stream">The stream to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public IXmlDocSerializable Deserialize(
            Type type,
            Stream stream,
            IEnumerable<Type> extraTypes = null)
        {
            if (extraTypes.IsNullOrEmpty() == false)
            {
                var map = new Dictionary<string, Type>();
                extraTypes.ForEach(t => map.Add(t.Name, t));
                return this.Deserialize(type, stream, map);
            }
            else
            {
                return this.Deserialize(type, stream, (Dictionary<string, Type>)null);
            }
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="stream">The stream to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public IXmlDocSerializable Deserialize(
            Type type,
            Stream stream,
            Dictionary<string, Type> extraTypes)
        {
            var doc = new XmlDocument();
            doc.Load(stream);

            return this.Deserialize(type, doc, extraTypes);
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="doc">The XML document to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public IXmlDocSerializable Deserialize(
            Type type,
            XmlDocument doc,
            IEnumerable<Type> extraTypes)
        {
            if (extraTypes.IsNullOrEmpty() == false)
            {
                var map = new Dictionary<string, Type>();
                extraTypes.ForEach(t => map.Add(t.Name, t));
                return this.Deserialize(type, doc, map);
            }
            else
            {
                return this.Deserialize(type, doc, (Dictionary<string, Type>)null);
            }
        }

        /// <summary>
        /// Deserialize the contents in the stream to an object of the specified type.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="doc">The XML document to deserialize from.</param>
        /// <param name="extraTypes">The extra types for deserialization.</param>
        /// <returns>The deserialized object.</returns>
        public IXmlDocSerializable Deserialize(
            Type type,
            XmlDocument doc,
            Dictionary<string, Type> extraTypes)
        {
            // Create the serialization context.
            var context = new XmlDocSerializationContext(doc, extraTypes);

            Type realType = type;

            // Read the type written in the root element when it was serialized.
            Type serializedType =
                XmlDocSerializationHelper.ReadElementType(context, doc.DocumentElement);
            if (serializedType != null && serializedType != realType)
            {
                realType = serializedType;
            }

            // Create instance for deserialization.
            var obj = Activator.CreateInstance(realType) as IXmlDocSerializable;

            // Deserialize.
            if (obj.ReadXml(context) == false)
            {
                return null;
            }

            // Call the post-deserialization handlers.
            context.ReportPostDeserialize(obj);

            return obj;
        }
    }
}
