﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Class as a serializable wrapper of the .net generic Dictionary.
    /// </summary>
    /// <typeparam name="TKey">The type of the keys.</typeparam>
    /// <typeparam name="TValue">The type of the values.</typeparam>
    public class SerializableDictionary<TKey, TValue> :
        Dictionary<TKey, TValue>,
        IXmlSerializable,
        IXmlDocSerializable
    {
        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public System.Xml.Schema.XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            var element = context.CurrentNode as XmlElement;
            if (element == null)
            {
                return false;
            }

            // First check if the element is supposed to be null.
            if (XmlDocSerializationHelper.IsNilElement(element) == true)
            {
                return false;
            }

            // First clear the items.
            this.Clear();

            Type keyType = typeof(TKey);
            Type valueType = typeof(TValue);

            // Read the items.
            foreach (XmlNode child in element.ChildNodes)
            {
                if (child.NodeType == XmlNodeType.Element &&
                    child.Name == "item")
                {
                    var itemElement = (XmlElement)child;

                    context.PushCurrentNode(itemElement);

                    TKey key = default(TKey);
                    TValue value = default(TValue);

                    var keyElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                        itemElement, "key").FirstOrDefault();
                    if (keyElement != null)
                    {
                        context.PushCurrentNode(keyElement);

                        Type overrideType =
                            XmlDocSerializationHelper.ReadElementType(context, keyElement);

                        key = this.ReadElement(
                            context,
                            keyElement.FirstChild as XmlElement,
                            default(TKey),
                            overrideType);

                        context.PopCurrentNode();
                    }

                    var valueElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                        itemElement, "value").FirstOrDefault();
                    if (valueElement != null)
                    {
                        context.PushCurrentNode(valueElement);

                        Type overrideType =
                            XmlDocSerializationHelper.ReadElementType(context, valueElement);

                        value = this.ReadElement(
                            context,
                            valueElement.FirstChild as XmlElement,
                            default(TValue),
                            overrideType);

                        context.PopCurrentNode();
                    }

                    this.Add(key, value);

                    context.PopCurrentNode();
                }
            }

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            foreach (KeyValuePair<TKey, TValue> pair in this)
            {
                // Create the item XML element and add it to the current node.
                var itemElement = context.XmlDocument.CreateElement("item");
                context.CurrentNode.AppendChild(itemElement);

                context.PushCurrentNode(itemElement);

                // Create the key XML element and add it to the current node.
                var keyElement = context.XmlDocument.CreateElement("key");
                context.CurrentNode.AppendChild(keyElement);

                context.PushCurrentNode(keyElement);

                if (pair.Key == null)
                {
                    this.WriteElement<TKey>(context, typeof(TKey).Name, pair.Key);
                }
                else
                {
                    this.WriteElement(context, pair.Key.GetType().Name, pair.Key);
                }

                context.PopCurrentNode();

                // Create the value XML element and add it to the current node.
                var valueElement = context.XmlDocument.CreateElement("value");
                context.CurrentNode.AppendChild(valueElement);

                context.PushCurrentNode(valueElement);

                if (pair.Value == null)
                {
                    this.WriteElement<TValue>(context, typeof(TValue).Name, pair.Value);
                }
                else
                {
                    this.WriteElement(context, pair.Value.GetType().Name, pair.Value);
                }

                context.PopCurrentNode();

                context.PopCurrentNode();
            }

            return true;
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            XmlSerializer keySerializer = new XmlSerializer(typeof(TKey));
            XmlSerializer valueSerializer = new XmlSerializer(typeof(TValue));

            bool wasEmpty = reader.IsEmptyElement;
            reader.Read();

            if (wasEmpty == true)
            {
                return;
            }

            while (reader.NodeType != System.Xml.XmlNodeType.EndElement)
            {
                reader.ReadStartElement("item");

                reader.ReadStartElement("key");
                TKey key = (TKey)keySerializer.Deserialize(reader);
                reader.ReadEndElement();

                reader.ReadStartElement("value");
                TValue value = (TValue)valueSerializer.Deserialize(reader);
                reader.ReadEndElement();

                this.Add(key, value);

                reader.ReadEndElement();
                reader.MoveToContent();
            }

            reader.ReadEndElement();
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            XmlSerializer keySerializer = new XmlSerializer(typeof(TKey));
            XmlSerializer valueSerializer = new XmlSerializer(typeof(TValue));

            foreach (TKey key in this.Keys)
            {
                writer.WriteStartElement("item");

                writer.WriteStartElement("key");
                keySerializer.Serialize(writer, key);
                writer.WriteEndElement();

                writer.WriteStartElement("value");
                TValue value = this[key];
                valueSerializer.Serialize(writer, value);
                writer.WriteEndElement();

                writer.WriteEndElement();
            }
        }
    }
}
