﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Net;
using System.Security;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Extension methods for IXmlDocSerializable.
    /// </summary>
    public static class IXmlDocSerializableExtensions
    {
        /// <summary>
        /// Check if the current element contains an attribute with the specified name.
        /// </summary>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="attributeName">The attribute name.</param>
        /// <returns>True if the attribute exists.</returns>
        public static bool HasAttribute(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string attributeName)
        {
            var element = context.CurrentNode as XmlElement;
            if (element == null)
            {
                return false;
            }

            return element.HasAttribute(attributeName);
        }

        /// <summary>
        /// Check if the current element contains an child element with the specified name.
        /// </summary>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="elementName">The child element name.</param>
        /// <returns>True if the child element exists.</returns>
        public static bool HasElement(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string elementName)
        {
            var element = context.CurrentNode as XmlElement;
            if (element == null)
            {
                return false;
            }

            foreach (XmlNode child in element.ChildNodes)
            {
                if (child is XmlElement && child.Name == elementName)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Write value to XML attribute with the specified tag name.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="attributeName">The name of the XML element.</param>
        /// <param name="value">The value to write.</param>
        /// <returns>True on success.</returns>
        public static bool WriteAttribute<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string attributeName,
            T value)
        {
            Type valueType = value != null ? value.GetType() : typeof(T);

            // If the type can directly convert from string, convert it.
            var converter = TypeDescriptor.GetConverter(valueType);
            if (converter == null)
            {
                return false;
            }

            // The current node must be an element to have a child attribute.
            var currElement = context.CurrentNode as XmlElement;
            if (currElement == null)
            {
                return false;
            }

            if (valueType == typeof(string))
            {
                var tmpStr = value as string;
                currElement.SetAttribute(attributeName, SecurityElement.Escape(tmpStr));
            }
            else
            {
                // Write the attribute to the current element.
                currElement.SetAttribute(
                    attributeName,
                    converter.ConvertToInvariantString(value));
            }

            return true;
        }

        /// <summary>
        /// Read value from the given XML attribute.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="attributeName">The name of the XML attribute to read from.</param>
        /// <param name="originalValue">The original value.</param>
        /// <param name="overrideType">The override type for the value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static T ReadAttribute<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string attributeName,
            T originalValue,
            Type overrideType = null)
        {
            // The current node must be an element to have a child attribute.
            var currElement = context.CurrentNode as XmlElement;
            if (currElement == null)
            {
                return originalValue;
            }

            // Get the attribute node.
            var attr = currElement.GetAttributeNode(attributeName);
            if (attr == null)
            {
                return originalValue;
            }

            Type valueType = overrideType != null ? overrideType : typeof(T);

            if (valueType == typeof(string))
            {
                // 文字列だったらアンエスケープしてから格納
                originalValue = (T)(object)UnescapeString(attr.Value);
            }
            else
            {
                // If the type can directly convert from string, convert it.
                var converter = TypeDescriptor.GetConverter(valueType);
                originalValue = (T)converter.ConvertFromInvariantString(attr.Value);
            }

            return originalValue;
        }

        /// <summary>
        /// Write value to XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="value">The value to write.</param>
        /// <returns>True on success.</returns>
        public static bool WriteElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            T value)
        {
            Type valueType = value != null ? value.GetType() : typeof(T);

            if (value == null)
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);
                XmlDocSerializationHelper.WriteNilElement(element);
            }
            else if (typeof(IXmlDocSerializable).IsAssignableFrom(valueType) == true)
            {
                // The value is an IXmlDocSerializable.

                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                context.PushCurrentNode(element);

                var serializable = (IXmlDocSerializable)value;

                context.ReportPreSerialize(serializable);
                serializable.WriteXml(context);
                context.ReportPostSerialize(serializable);

                // If the value is derived from T, we should write the type to the element,
                // so we can deserialize the element correctly.
                if (value.GetType() != typeof(T))
                {
                    XmlDocSerializationHelper.WriteElementType(element, value.GetType());
                }

                context.PopCurrentNode();
            }
            else if (typeof(IXmlSerializable).IsAssignableFrom(valueType) == true)
            {
                // The value is an IXmlSerializable.
                var nav = context.CurrentNode.CreateNavigator();
                using (var writer = nav.AppendChild())
                {
                    writer.WriteStartElement(tagName);

                    // If the value is derived from T, we should write the type to the element,
                    // so we can deserialize the element correctly.
                    if (value.GetType() != typeof(T))
                    {
                        XmlDocSerializationHelper.WriteElementType(writer, value.GetType());
                    }

                    var serializable = (IXmlSerializable)value;
                    serializable.WriteXml(writer);
                    writer.WriteEndElement();
                }
            }
            else if (valueType == typeof(Rectangle))
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                context.PushCurrentNode(element);

                var rect = (Rectangle)(object)value;

                XmlDocSerializationHelper.WriteSimpleElement(context, "X", rect.X.ToString());
                XmlDocSerializationHelper.WriteSimpleElement(context, "Y", rect.Y.ToString());
                XmlDocSerializationHelper.WriteSimpleElement(context, "Width", rect.Width.ToString());
                XmlDocSerializationHelper.WriteSimpleElement(context, "Height", rect.Height.ToString());

                XmlDocSerializationHelper.WriteElementType(element, typeof(Rectangle));

                context.PopCurrentNode();
            }
            else if (valueType == typeof(Size))
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                context.PushCurrentNode(element);

                var rect = (Size)(object)value;

                XmlDocSerializationHelper.WriteSimpleElement(context, "Width", rect.Width.ToString());
                XmlDocSerializationHelper.WriteSimpleElement(context, "Height", rect.Height.ToString());

                XmlDocSerializationHelper.WriteElementType(element, typeof(Size));

                context.PopCurrentNode();
            }
            else if (valueType == typeof(string))
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                // エスケープ処理を通して書き出し
                var tmpStr = value as string;
                element.InnerXml = SecurityElement.Escape(tmpStr);
            }
            else
            {
                // If the type can directly convert from string, convert it.
                var converter = TypeDescriptor.GetConverter(valueType);

                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                element.InnerXml = converter.ConvertToInvariantString(value);
            }

            return true;
        }

        /// <summary>
        /// Write System.Collections.Generic.List to XML element with the specified tag name.
        /// If the tagName is null or empty string, the items will be directly written to the
        /// current node saved in the context.
        /// </summary>
        /// <typeparam name="T">The list element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="value">The value to write.</param>
        /// <param name="overrideItemName">Use the specified tag name for the items in the enumearble.</param>
        /// <returns>True on success.</returns>
        public static bool WriteEnumerableElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            IEnumerable<T> value,
            string overrideItemName = null)
        {
            // If the tagName is null or empty string, do not create the XML element.
            bool createNewElement = string.IsNullOrEmpty(tagName) == false;

            if (createNewElement == true)
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                context.PushCurrentNode(element);
            }

            foreach (T item in value)
            {
                string itemTagName;
                if (item == null)
                {
                    itemTagName = typeof(T).Name;
                }
                else
                {
                    itemTagName = item.GetType().Name;
                }

                if (string.IsNullOrEmpty(overrideItemName) == false)
                {
                    itemTagName = overrideItemName;
                }

                WriteElement(self, context, itemTagName, item);
            }

            if (createNewElement == true)
            {
                context.PopCurrentNode();
            }

            return true;
        }

        /// <summary>
        /// Write System.Collections.Generic.Dictionary to XML element with the specified tag name.
        /// If the tagName is null or empty string, the items will be directly written to the
        /// current node saved in the context.
        /// </summary>
        /// <typeparam name="TKey">The dictionary key type.</typeparam>
        /// <typeparam name="TValue">The dictionary value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="value">The value to write.</param>
        /// <returns>True on success.</returns>
        public static bool WriteDictionaryElement<TKey, TValue>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            Dictionary<TKey, TValue> value)
        {
            // If the tagName is null or empty string, do not create the XML element.
            bool createNewElement = string.IsNullOrEmpty(tagName) == false;

            if (createNewElement == true)
            {
                // Create the XML element and add it to the current node.
                var element = context.XmlDocument.CreateElement(tagName);
                context.CurrentNode.AppendChild(element);

                context.PushCurrentNode(element);
            }

            foreach (KeyValuePair<TKey, TValue> pair in value)
            {
                // Create the item XML element and add it to the current node.
                var itemElement = context.XmlDocument.CreateElement("item");
                context.CurrentNode.AppendChild(itemElement);

                context.PushCurrentNode(itemElement);

                // Create the key XML element and add it to the current node.
                var keyElement = context.XmlDocument.CreateElement("key");
                context.CurrentNode.AppendChild(keyElement);

                context.PushCurrentNode(keyElement);

                if (pair.Key == null)
                {
                    WriteElement<TKey>(self, context, typeof(TKey).Name, pair.Key);
                }
                else
                {
                    WriteElement(self, context, pair.Key.GetType().Name, pair.Key);
                }

                context.PopCurrentNode();

                // Create the value XML element and add it to the current node.
                var valueElement = context.XmlDocument.CreateElement("value");
                context.CurrentNode.AppendChild(valueElement);

                context.PushCurrentNode(valueElement);

                if (pair.Value == null)
                {
                    WriteElement<TValue>(self, context, typeof(TValue).Name, pair.Value);
                }
                else
                {
                    WriteElement(self, context, pair.Value.GetType().Name, pair.Value);
                }

                context.PopCurrentNode();

                context.PopCurrentNode();
            }

            if (createNewElement == true)
            {
                context.PopCurrentNode();
            }

            return true;
        }

        /// <summary>
        /// Read value from XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static T ReadElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            T originalValue)
        {
            var firstElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                context.CurrentNode, tagName).FirstOrDefault();

            return ReadElement(self, context, firstElement, originalValue);
        }

        /// <summary>
        /// Read System.Collections.Generic.List from XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="T">The list element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static List<T> ReadListElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            List<T> originalValue)
        {
            var firstElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                context.CurrentNode, tagName).FirstOrDefault();

            return ReadListElement(self, context, firstElement, originalValue);
        }

        /// <summary>
        /// Read array from XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="T">The array element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static T[] ReadArrayElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            T[] originalValue)
        {
            var firstElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                context.CurrentNode, tagName).FirstOrDefault();

            return ReadArrayElement(self, context, firstElement, originalValue);
        }

        /// <summary>
        /// Read System.Collections.Generic.Dictionary from XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="TKey">The dictionary key type.</typeparam>
        /// <typeparam name="TValue">The dictionary value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static Dictionary<TKey, TValue> ReadDictionaryElement<TKey, TValue>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName,
            Dictionary<TKey, TValue> originalValue)
        {
            var firstElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                context.CurrentNode, tagName).FirstOrDefault();

            return ReadDictionaryElement(self, context, firstElement, originalValue);
        }

        /// <summary>
        /// Read value from the given XML element.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="element">The XML element to read from.</param>
        /// <param name="originalValue">The original value.</param>
        /// <param name="overrideType">The override type for the value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static T ReadElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            XmlElement element,
            T originalValue,
            Type overrideType = null)
        {
            if (element == null)
            {
                return originalValue;
            }

            // First check if the element is supposed to be null.
            if (XmlDocSerializationHelper.IsNilElement(element) == true)
            {
                return default(T);
            }

            // Determine the actual type.
            Type valueType;
            if (overrideType != null)
            {
                valueType = overrideType;
            }
            else
            {
                Type elementType = XmlDocSerializationHelper.ReadElementType(context, element);
                if (elementType == null)
                {
                    valueType = typeof(T);
                }
                else
                {
                    valueType = elementType;
                }
            }

            // We are unable to create an instance of a abstract type.
            if (valueType.IsAbstract == true)
            {
                return originalValue;
            }

            // Push XML element to the context for further processing.
            context.PushCurrentNode(element);

            if (typeof(IXmlDocSerializable).IsAssignableFrom(valueType) == true)
            {
                // The value is an IXmlDocSerializable.
                var serializable = (IXmlDocSerializable)originalValue;
                if (serializable == null)
                {
                    serializable = (IXmlDocSerializable)Activator.CreateInstance(valueType);
                }

                // Load the serializable.
                serializable.ReadXml(context);

                // Call the post-deserialization handlers.
                context.ReportPostDeserialize(serializable);

                originalValue = (T)serializable;
            }
            else if (typeof(IXmlSerializable).IsAssignableFrom(valueType) == true)
            {
                // The value is an IXmlSerializable.
                var serializable = (IXmlSerializable)originalValue;
                if (serializable == null)
                {
                    serializable = (IXmlSerializable)Activator.CreateInstance(valueType);
                }

                // Create the reader so the IXmlSerializable can read the XML node.
                var reader = new XmlNodeReader(context.CurrentNode);
                reader.Read();

                // Load the serializable.
                serializable.ReadXml(reader);

                originalValue = (T)serializable;
            }
            else if (valueType == typeof(Rectangle))
            {
                var nodeX = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "X").FirstOrDefault();

                var nodeY = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "Y").FirstOrDefault();

                var nodeW = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "Width").FirstOrDefault();

                var nodeH = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "Height").FirstOrDefault();

                originalValue = (T)(object)new Rectangle(
                    int.Parse(nodeX.InnerText),
                    int.Parse(nodeY.InnerText),
                    int.Parse(nodeW.InnerText),
                    int.Parse(nodeH.InnerText));
            }
            else if (valueType == typeof(Size))
            {
                var nodeW = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "Width").FirstOrDefault();

                var nodeH = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode, "Height").FirstOrDefault();

                originalValue = (T)(object)new Size(
                    int.Parse(nodeW.InnerText),
                    int.Parse(nodeH.InnerText));
            }
            else if (valueType == typeof(IPAddress))
            {
                originalValue = (T)(object)IPAddress.Parse(element.InnerXml);
            }
            else if (valueType == typeof(string))
            {
                // 文字列だったらアンエスケープしてから格納
                originalValue = (T)(object)UnescapeString(element.InnerXml);
            }
            else
            {
                // If the type can directly convert from string, convert it.
                var converter = TypeDescriptor.GetConverter(valueType);
                originalValue = (T)converter.ConvertFromInvariantString(element.InnerXml);
            }

            // Pop the current node to continue on processing the parent node.
            context.PopCurrentNode();

            return originalValue;
        }

        /// <summary>
        /// Read multiple XML elements with the specified tag name for enumeration.
        /// </summary>
        /// <typeparam name="T">The element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagName">The tag name of the XML element.</param>
        /// <returns>The values read from the XML elements.</returns>
        public static IEnumerable<T> ReadElementsByTagName<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            string tagName)
        {
            var elements = XmlDocSerializationHelper.EnumerateElementsByTagName(
                context.CurrentNode, tagName);

            foreach (XmlElement element in elements)
            {
                yield return ReadElement(self, context, element, default(T));
            }
        }

        /// <summary>
        /// Read multiple XML elements with the specified tag names for enumeration.
        /// </summary>
        /// <typeparam name="T">The element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="tagNames">The name of the XML elements to load.</param>
        /// <returns>The values read from the XML elements.</returns>
        public static IEnumerable<T> ReadElementsByTagNames<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            IEnumerable<string> tagNames)
        {
            var elements = XmlDocSerializationHelper.EnumerateElementsByTagNames(
                context.CurrentNode, tagNames);

            foreach (XmlElement element in elements)
            {
                yield return ReadElement(self, context, element, default(T));
            }
        }

        /// <summary>
        /// Read multiple XML elements with the specified tag names for enumeration.
        /// </summary>
        /// <typeparam name="T">The element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="nameTypeMap">
        /// The map of element name and their corresponding type to load from the elements.
        /// </param>
        /// <returns>The values read from the XML elements.</returns>
        public static IEnumerable<T> ReadElementsByTagNames<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            Dictionary<string, Type> nameTypeMap)
        {
            var elements = XmlDocSerializationHelper.EnumerateElementsByTagNames(
                context.CurrentNode, nameTypeMap.Keys);

            foreach (XmlElement element in elements)
            {
                yield return ReadElement(
                    self,
                    context,
                    element,
                    default(T),
                    nameTypeMap[element.Name]);
            }
        }

        /// <summary>
        /// Read System.Collections.Generic.List from the given XML element.
        /// </summary>
        /// <typeparam name="T">The list element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="element">The XML element to read from.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static List<T> ReadListElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            XmlElement element,
            List<T> originalValue)
        {
            if (element == null)
            {
                return originalValue;
            }

            // First check if the element is supposed to be null.
            if (XmlDocSerializationHelper.IsNilElement(element) == true)
            {
                return null;
            }

            Type itemType = typeof(T);

            // Push XML element to the context for further processing.
            context.PushCurrentNode(element);

            // Check if the list has already been created.
            List<T> newValue = new List<T>();
            if (originalValue != null)
            {
                newValue.AddRange(originalValue);
            }

            // Read the items.
            foreach (XmlNode child in element.ChildNodes)
            {
                if (child.NodeType == XmlNodeType.Element)
                {
                    var childElement = (XmlElement)child;

                    Type overrideType =
                        XmlDocSerializationHelper.ReadElementType(context, childElement);
                    if (overrideType == null &&
                        childElement.Name != typeof(T).Name)
                    {
                        overrideType = context.FindExtraType(childElement.Name);
                    }

                    newValue.Add(ReadElement(
                        self,
                        context,
                        childElement,
                        default(T),
                        overrideType));
                }
            }

            // Pop the current node to continue on processing the parent node.
            context.PopCurrentNode();

            return newValue;
        }

        /// <summary>
        /// Read array from the given XML element.
        /// </summary>
        /// <typeparam name="T">The array element type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="element">The XML element to read from.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static T[] ReadArrayElement<T>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            XmlElement element,
            T[] originalValue)
        {
            if (element == null)
            {
                return originalValue;
            }

            // First check if the element is supposed to be null.
            if (XmlDocSerializationHelper.IsNilElement(element) == true)
            {
                return null;
            }

            Type itemType = typeof(T);

            // Push XML element to the context for further processing.
            context.PushCurrentNode(element);

            // Read the items.
            var tmpList = new List<T>(element.ChildNodes.Count);
            foreach (XmlNode child in element.ChildNodes)
            {
                if (child is XmlElement)
                {
                    var childElement = (XmlElement)child;

                    Type overrideType =
                        XmlDocSerializationHelper.ReadElementType(context, childElement);
                    if (overrideType == null &&
                        childElement.Name != typeof(T).Name)
                    {
                        overrideType = context.FindExtraType(childElement.Name);
                    }

                    tmpList.Add(ReadElement(
                        self,
                        context,
                        childElement,
                        default(T),
                        overrideType));
                }
            }

            // Pop the current node to continue on processing the parent node.
            context.PopCurrentNode();

            return tmpList.ToArray();
        }

        /// <summary>
        /// Read System.Collections.Generic.Dictionary from XML element with the specified tag name.
        /// </summary>
        /// <typeparam name="TKey">The dictionary key type.</typeparam>
        /// <typeparam name="TValue">The dictionary value type.</typeparam>
        /// <param name="self">The IXmlDocSerializable.</param>
        /// <param name="context">The serialization context.</param>
        /// <param name="element">The XML element to read from.</param>
        /// <param name="originalValue">The original value.</param>
        /// <returns>The value read from the XML element.</returns>
        public static Dictionary<TKey, TValue> ReadDictionaryElement<TKey, TValue>(
            this IXmlDocSerializable self,
            XmlDocSerializationContext context,
            XmlElement element,
            Dictionary<TKey, TValue> originalValue)
        {
            if (element == null)
            {
                return originalValue;
            }

            // First check if the element is supposed to be null.
            if (XmlDocSerializationHelper.IsNilElement(element) == true)
            {
                return null;
            }

            Type keyType = typeof(TKey);
            Type valueType = typeof(TValue);

            // Push XML element to the context for further processing.
            context.PushCurrentNode(element);

            // Check if the list has already been created.
            var newValue = new Dictionary<TKey, TValue>();
            if (originalValue != null)
            {
                originalValue.ForEach(p => newValue.Add(p.Key, p.Value));
            }

            // Read the items.
            foreach (XmlNode child in element.ChildNodes)
            {
                if (child.NodeType == XmlNodeType.Element &&
                    child.Name == "item")
                {
                    var itemElement = (XmlElement)child;

                    context.PushCurrentNode(itemElement);

                    TKey key = default(TKey);
                    TValue value = default(TValue);

                    var keyElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                        itemElement, "key").FirstOrDefault();
                    if (keyElement != null)
                    {
                        context.PushCurrentNode(keyElement);

                        Type overrideType =
                            XmlDocSerializationHelper.ReadElementType(context, keyElement);

                        key = ReadElement(
                            self,
                            context,
                            keyElement.FirstChild as XmlElement,
                            default(TKey),
                            overrideType);

                        context.PopCurrentNode();
                    }

                    var valueElement = XmlDocSerializationHelper.EnumerateElementsByTagName(
                        itemElement, "value").FirstOrDefault();
                    if (valueElement != null)
                    {
                        context.PushCurrentNode(valueElement);

                        Type overrideType =
                            XmlDocSerializationHelper.ReadElementType(context, valueElement);

                        value = ReadElement(
                            self,
                            context,
                            valueElement.FirstChild as XmlElement,
                            default(TValue),
                            overrideType);

                        context.PopCurrentNode();
                    }

                    newValue.Add(key, value);

                    context.PopCurrentNode();
                }
            }

            // Pop the current node to continue on processing the parent node.
            context.PopCurrentNode();

            return newValue;
        }

        /// <summary>
        /// SecurityElement.Escape()の対になる関数がなかったので独自実装
        /// </summary>
        /// <param name="arg">エスケープ済みの文字列</param>
        /// <returns>アンエスケープした文字列</returns>
        private static string UnescapeString(string arg)
        {
            arg = arg.Replace("&amp;", "&");
            arg = arg.Replace("&apos;", "\'");
            arg = arg.Replace("&quot;", "\"");
            arg = arg.Replace("&gt;", ">");
            arg = arg.Replace("&lt;", "<");

            return arg;
        }
    }
}
