﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.Foundation.Animation;

namespace EffectMaker.Foundation.Render.ScrollBar
{
    /// <summary>
    /// Base class for the scroll bars.
    /// </summary>
    internal abstract class ScrollBarBase
    {
        /// <summary>Constant defines the scroll bar size.</summary>
        public const int ScrollBarSize = 10;

        /// <summary>Constant defines the arrow button size.</summary>
        protected const int ArrowButtonSize = 10;

        /// <summary>Constant defines the scroll bar handle size.</summary>
        protected const int ScrollHandleSize = 8;

        /// <summary>Constant defines the minimum scroll bar handle length.</summary>
        protected const int MinScrollHandleLength = 10;

        /// <summary>Constant represents the target background alpha for animation.</summary>
        private const byte BackgroundTargetAlpha = 64;

        /// <summary>Constant represents the target foreground alpha for animation.</summary>
        private const byte ForegroundTargetAlpha = 128;

        /// <summary>Size of the contents.</summary>
        private Size ownerContentSize = new Size(0, 0);

        /// <summary>Size of the view.</summary>
        private Size ownerViewSize = new Size(0, 0);

        /// <summary>The display rectangle for the scroll bar.</summary>
        private Rectangle scrollBarRect = new Rectangle(0, 0, 0, 0);

        /// <summary>The display rectangle for the scroll track.</summary>
        private Rectangle scrollTrackRect = new Rectangle(0, 0, 0, 0);

        /// <summary>The display rectangle for the scroll bar handle in compact mode.</summary>
        private Rectangle scrollHandleRect = new Rectangle(0, 0, 0, 0);

        /// <summary>The display rectangle for the decrease button.</summary>
        private Rectangle decreaseBtnRect = new Rectangle(0, 0, 0, 0);

        /// <summary>The display rectangle for the increase button.</summary>
        private Rectangle increaseBtnRect = new Rectangle(0, 0, 0, 0);

        /// <summary>The offset between the handle and the mouse cursor when start dragging.</summary>
        private Point mouseDragOffset = new Point();

        /// <summary>The scroll position.</summary>
        private int scrollPosition = 0;

        /// <summary>The flag indicating whether the scroll bar is enabled or not.</summary>
        private bool isEnabled = true;

        /// <summary>The flag indicating whether cursor is over the owner control.</summary>
        private bool isMouseOverOwner = false;

        /// <summary>The flag indicating whether cursor is over the scroll bar.</summary>
        private bool isMouseOverScrollBar = false;

        /// <summary>Background alpha animation controller for the vertical scroll bar.</summary>
        private AnimationControllerB backgroundAnimationController = new AnimationControllerB();

        /// <summary>Foreground alpha animation controller for the vertical scroll bar.</summary>
        private AnimationControllerB foregroundAnimationController = new AnimationControllerB();

        /// <summary>
        /// Constructor.
        /// </summary>
        public ScrollBarBase()
        {
            this.ForeColor = Color.Gray;
            this.BackColor = Color.Gray;

            this.IsEnabled = true;
            this.Value = 0;
            this.Increment = 1;

            this.IsMouseDragging = false;

            this.foregroundAnimationController.AnimationUpdated += (s, e) =>
            {
                this.ForegroundAlpha = (byte)e.Value;
                this.TriggerRenderRequiredEvent();
            };

            this.backgroundAnimationController.AnimationUpdated += (s, e) =>
            {
                this.BackgroundAlpha = (byte)e.Value;
                this.TriggerRenderRequiredEvent();
            };
        }

        /// <summary>
        /// Event triggered when rendering is required for this scroll bar.
        /// </summary>
        public event EventHandler RenderRequired = null;

        /// <summary>
        /// Event triggered when the scroll position is changed.
        /// </summary>
        public event EventHandler Scroll = null;

        /// <summary>Get the flag indicating if the scroll bar is enabled.</summary>
        public bool IsEnabled
        {
            get
            {
                return this.isEnabled;
            }

            set
            {
                if (this.isEnabled == value)
                {
                    return;
                }

                this.isEnabled = value;

                // Fire the event.
                if (this.RenderRequired != null)
                {
                    this.RenderRequired(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>Get the flag indicating if the scroll bar is visible.</summary>
        public abstract bool IsVisible { get; }

        /// <summary>Get or set the foreground color.</summary>
        public Color ForeColor { get; set; }

        /// <summary>Get or set the background color.</summary>
        public Color BackColor { get; set; }

        /// <summary>Get or set the increment when the increment/decrement button is pressed.</summary>
        public int Increment { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether cursor is over the owner control.
        /// </summary>
        public bool IsMouseOverOwner
        {
            get
            {
                return this.isMouseOverOwner;
            }

            private set
            {
                if (this.isMouseOverOwner == value)
                {
                    return;
                }

                this.isMouseOverOwner = value;
                if (value == false)
                {
                    // Also clear the IsMouseOverScrollBar flag.
                    this.IsMouseOverScrollBar = false;
                }

                if (value == true)
                {
                    // Fade in the arrows and scroll handle.
                    this.foregroundAnimationController.StartAnimation(
                        0.5,
                        25.0,
                        0,
                        ForegroundTargetAlpha,
                        this.ForegroundAlpha);
                }
                else
                {
                    // Fade out the arrows and scroll handle.
                    this.foregroundAnimationController.StartAnimation(
                        0.25,
                        25.0,
                        ForegroundTargetAlpha,
                        0,
                        this.ForegroundAlpha);
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether cursor is over the scroll bar.
        /// </summary>
        public bool IsMouseOverScrollBar
        {
            get
            {
                return this.isMouseOverScrollBar;
            }

            private set
            {
                if (this.isMouseOverScrollBar == value)
                {
                    return;
                }

                this.isMouseOverScrollBar = value;

                if (value == true)
                {
                    // Fade in the arrow button background and scroll track.
                    this.backgroundAnimationController.StartAnimation(
                        0.25,
                        25.0,
                        0,
                        BackgroundTargetAlpha,
                        this.BackgroundAlpha);
                }
                else
                {
                    // Fade out the arrow button background and scroll track.
                    this.backgroundAnimationController.StartAnimation(
                        0.25,
                        25.0,
                        BackgroundTargetAlpha,
                        0,
                        this.BackgroundAlpha);
                }
            }
        }

        /// <summary>
        /// Get the flag indicating whether if the scroll handle is being mouse dragged.
        /// </summary>
        public bool IsMouseDragging { get; private set; }

        /// <summary>Get or set the scroll value.</summary>
        public int Value
        {
            get
            {
                return this.scrollPosition;
            }

            set
            {
                int clamppedValue = this.ClampValue(value);
                if (this.scrollPosition == clamppedValue)
                {
                    return;
                }

                this.scrollPosition = clamppedValue;

                this.ComputeScrollHandle();

                // Fire the event.
                if (this.Scroll != null)
                {
                    this.Scroll(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Get the rectangle of the whole scroll bar.
        /// </summary>
        public Rectangle ScrollBarBounds
        {
            get { return this.scrollBarRect; }
        }

        /// <summary>
        /// Get the rectangle of the scroll track
        /// </summary>
        public Rectangle ScrollTrackBounds
        {
            get { return this.scrollTrackRect; }
        }

        /// <summary>
        /// Get the rectangle of the scroll bar handle.
        /// </summary>
        public Rectangle ScrollHandleBounds
        {
            get { return this.scrollHandleRect; }
        }

        /// <summary>
        /// Get the rectangle of the decrease (up) button.
        /// </summary>
        public Rectangle DecreaseButtonBounds
        {
            get { return this.decreaseBtnRect; }
        }

        /// <summary>
        /// Get the rectangle of the increase (down) button.
        /// </summary>
        public Rectangle IncreaseButtonBounds
        {
            get { return this.increaseBtnRect; }
        }

        /// <summary>
        /// Get or set the alpha value for the background.
        /// </summary>
        protected byte BackgroundAlpha { get; set; }

        /// <summary>
        /// Get or set the alpha value for the foreground.
        /// </summary>
        protected byte ForegroundAlpha { get; set; }

        /// <summary>
        /// Get the owner content size.
        /// </summary>
        protected Size OwnerContentSize
        {
            get { return this.ownerContentSize; }
        }

        /// <summary>
        /// Get the owner view size.
        /// </summary>
        protected Size OwnerViewSize
        {
            get { return this.ownerViewSize; }
        }

        /// <summary>
        /// Get the offset between the handle and the mouse cursor when start dragging.
        /// </summary>
        protected Point MouseDragOffset
        {
            get { return this.mouseDragOffset; }
        }

        /// <summary>Update layout.</summary>
        /// <param name="rectDisplay">The display rectangle of the owner view.</param>
        /// <param name="sizeShrink">How much size to shrink from the rectangle.</param>
        public abstract void UpdateLayout(Rectangle rectDisplay, int sizeShrink);

        /// <summary>
        /// Set the content and view size of the scroll bar owner.
        /// </summary>
        /// <remarks>
        /// The unit of this size can be the actual number of pixels
        /// or any other unit (e.g. line count in a text view),
        /// but the unit of the view size and content size has to
        /// be the same.
        /// </remarks>
        /// <param name="contentSize">The content size.</param>
        /// <param name="viewSize">The view size.</param>
        public void SetContentAndViewSize(Size contentSize, Size viewSize)
        {
            this.ownerContentSize = contentSize;
            this.ownerViewSize = viewSize;

            // Fire the event.
            if (this.RenderRequired != null)
            {
                this.RenderRequired(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle MouseEnter event from the owner control.
        /// </summary>
        public virtual void OnOwnerMouseEnter()
        {
            this.IsMouseOverOwner = true;
        }

        /// <summary>
        /// Handle MouseLeave event from the owner control.
        /// </summary>
        public virtual void OnOwnerMouseLeave()
        {
            this.IsMouseOverOwner = false;
        }

        /// <summary>
        /// Handle MouseMove event from the owner control.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public virtual void OnOwnerMouseMove(MouseEventArgs e)
        {
            this.IsMouseOverScrollBar = this.ScrollBarBounds.Contains(e.Location);
        }

        /// <summary>
        /// Handle MouseDown event from the owner control.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public virtual void OnOwnerMouseDown(MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            int origValue = this.Value;

            if (this.decreaseBtnRect.Contains(e.Location) == true)
            {
                this.Value -= this.Increment;
            }
            else if (this.increaseBtnRect.Contains(e.Location) == true)
            {
                this.Value += this.Increment;
            }
            else if (this.scrollHandleRect.Contains(e.Location) == true)
            {
                this.IsMouseDragging = true;
                this.mouseDragOffset.X = this.scrollHandleRect.X - e.X;
                this.mouseDragOffset.Y = this.scrollHandleRect.Y - e.Y;
            }
            else if (this.scrollTrackRect.Contains(e.Location) == true)
            {
                if (e.Location.Y < this.scrollHandleRect.Top)
                {
                    this.Value -= this.ownerViewSize.Height;
                }
                else if (e.Location.Y > this.scrollHandleRect.Bottom)
                {
                    this.Value += this.ownerViewSize.Height;
                }
            }

            // Trigger event.
            if ((origValue != this.Value) &&
                (this.Scroll != null))
            {
                this.Scroll(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle MouseUp event from the owner control.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public virtual void OnOwnerMouseUp(MouseEventArgs e)
        {
            this.IsMouseDragging = false;
        }

        /// <summary>
        /// Render the scroll bar.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        public virtual void Render(Graphics g)
        {
            if (this.IsVisible == false)
            {
                return;
            }

            using (var brush = new SolidBrush(this.BackColor))
            {
                if (this.BackgroundAlpha > 0)
                {
                    brush.Color = Color.FromArgb(this.BackgroundAlpha, this.BackColor);
                    g.FillRectangle(brush, this.scrollBarRect);
                }

                if (this.ForegroundAlpha > 0)
                {
                    brush.Color = Color.FromArgb(this.ForegroundAlpha, this.ForeColor);
                    g.FillRectangle(brush, this.scrollHandleRect);

                    System.Drawing.Drawing2D.SmoothingMode origSmoothingMode = g.SmoothingMode;
                    g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;

                    this.RenderDecreaseButton(g);
                    this.RenderIncreaseButton(g);

                    g.SmoothingMode = origSmoothingMode;
                }
            }
        }

        /// <summary>
        /// Clamp the given scroll value to the scroll range.
        /// </summary>
        /// <param name="value">The scroll value.</param>
        /// <returns>The clamped value.</returns>
        protected abstract int ClampValue(int value);

        /// <summary>
        /// Render the increase button of the scroll bar.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected abstract void RenderIncreaseButton(Graphics g);

        /// <summary>
        /// Render the decrease button of the scroll bar.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected abstract void RenderDecreaseButton(Graphics g);

        /// <summary>
        /// Compute scroll bar handle size and position.
        /// </summary>
        protected abstract void ComputeScrollHandle();

        /// <summary>
        /// Set the scroll bar rectangle.
        /// </summary>
        /// <param name="x">X position of the rectangle.</param>
        /// <param name="y">Y position of the rectangle.</param>
        /// <param name="width">Width of the rectangle.</param>
        /// <param name="height">Height of the rectangle.</param>
        protected void SetScrollBarRectangle(int x, int y, int width, int height)
        {
            this.scrollBarRect.X = x;
            this.scrollBarRect.Y = y;
            this.scrollBarRect.Width = width;
            this.scrollBarRect.Height = height;
        }

        /// <summary>
        /// Set the scroll track rectangle.
        /// </summary>
        /// <param name="x">X position of the rectangle.</param>
        /// <param name="y">Y position of the rectangle.</param>
        /// <param name="width">Width of the rectangle.</param>
        /// <param name="height">Height of the rectangle.</param>
        protected void SetScrollTrackRectangle(int x, int y, int width, int height)
        {
            this.scrollTrackRect.X = x;
            this.scrollTrackRect.Y = y;
            this.scrollTrackRect.Width = width;
            this.scrollTrackRect.Height = height;
        }

        /// <summary>
        /// Set the scroll bar handle rectangle.
        /// </summary>
        /// <param name="x">X position of the rectangle.</param>
        /// <param name="y">Y position of the rectangle.</param>
        /// <param name="width">Width of the rectangle.</param>
        /// <param name="height">Height of the rectangle.</param>
        protected void SetScrollHandleRectangle(int x, int y, int width, int height)
        {
            this.scrollHandleRect.X = x;
            this.scrollHandleRect.Y = y;
            this.scrollHandleRect.Width = width;
            this.scrollHandleRect.Height = height;
        }

        /// <summary>
        /// Set the decrease button rectangle.
        /// </summary>
        /// <param name="x">X position of the rectangle.</param>
        /// <param name="y">Y position of the rectangle.</param>
        /// <param name="width">Width of the rectangle.</param>
        /// <param name="height">Height of the rectangle.</param>
        protected void SetDecreaseButtonRectangle(int x, int y, int width, int height)
        {
            this.decreaseBtnRect.X = x;
            this.decreaseBtnRect.Y = y;
            this.decreaseBtnRect.Width = width;
            this.decreaseBtnRect.Height = height;
        }

        /// <summary>
        /// Set the increase button rectangle.
        /// </summary>
        /// <param name="x">X position of the rectangle.</param>
        /// <param name="y">Y position of the rectangle.</param>
        /// <param name="width">Width of the rectangle.</param>
        /// <param name="height">Height of the rectangle.</param>
        protected void SetIncreaseButtonRectangle(int x, int y, int width, int height)
        {
            this.increaseBtnRect.X = x;
            this.increaseBtnRect.Y = y;
            this.increaseBtnRect.Width = width;
            this.increaseBtnRect.Height = height;
        }

        /// <summary>
        /// Trigger RenderRequired event.
        /// </summary>
        protected void TriggerRenderRequiredEvent()
        {
            // Fire the event.
            if (this.RenderRequired != null)
            {
                this.RenderRequired(this, EventArgs.Empty);
            }
        }
    }
}
