﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;

using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that holds data for rendering.
    /// </summary>
    public class RenderContext : ISettable, ICloneable
    {
        /// <summary>The transformation for rendering.</summary>
        private Transformation transformation = new Transformation();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="viewport">The view port.</param>
        public RenderContext(Viewport viewport)
        {
            this.Viewport = viewport;
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source to copy from.</param>
        public RenderContext(RenderContext src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the graphics object for rendering.
        /// </summary>
        public Graphics Graphics { get; set; }

        /// <summary>
        /// Get the transformation.
        /// </summary>
        public Transformation Transformation
        {
            get { return this.transformation; }
        }

        /// <summary>
        /// Get or set the scale.
        /// </summary>
        public PointF Scale
        {
            get { return this.transformation.Scale; }
            set { this.transformation.Scale = value; }
        }

        /// <summary>
        /// Get or set the translation.
        /// </summary>
        public PointF Translation
        {
            get { return this.transformation.Translation; }
            set { this.transformation.Translation = value; }
        }

        /// <summary>
        /// Get the transformation matrix.
        /// </summary>
        public Matrix TransformationMatrix
        {
            get { return this.transformation.TransformationMatrix; }
        }

        /// <summary>
        /// Get the view port.
        /// </summary>
        public Viewport Viewport { get; private set; }

        /// <summary>
        /// Get or set the rectangle that represents the view area.
        /// Anything outside of the rectangle should be clipped so
        /// it won't be rendered.
        /// </summary>
        public Rectangle ViewRectangle { get; set; }

        /// <summary>
        /// Get the flag indicating whether the view rectangle is empty.
        /// (Empty view rectangle means view clipping is disabled.)
        /// </summary>
        public bool IsViewRenctangleEmpty
        {
            get { return this.ViewRectangle.IsEmpty; }
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="transformation">Translation to apply.</param>
        public void ApplyTransformation(Transformation transformation)
        {
            this.transformation.ApplyTransformation(transformation);
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="translateX">Translation on X axis.</param>
        /// <param name="translateY">Translation on Y axis.</param>
        /// <param name="scaleX">Scale on X axis.</param>
        /// <param name="scaleY">Scale on Y axis.</param>
        public void ApplyTransformation(
            float translateX,
            float translateY,
            float scaleX = 1.0f,
            float scaleY = 1.0f)
        {
            this.transformation.ApplyTransformation(translateX, translateY, scaleX, scaleY);
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public virtual bool Set(object src)
        {
            var srcContext = src as RenderContext;
            if (srcContext == null)
            {
                return false;
            }

            this.Viewport = srcContext.Viewport;
            this.ViewRectangle = srcContext.ViewRectangle;
            this.transformation.Set(srcContext.transformation);

            return true;
        }

        /// <summary>
        /// Clone this render context.
        /// </summary>
        /// <returns>The new render context.</returns>
        public virtual object Clone()
        {
            return new RenderContext(this);
        }
    }

    /// <summary>
    /// Class to save the render context settings and restore them on dispose.
    /// </summary>
    public class PushPopRenderContext : IDisposable
    {
        /// <summary>Reference to the render context, to be restored on dispose.</summary>
        private RenderContext origRenderContext = null;

        /// <summary>The cloned render context to restore to the original one on dispose.</summary>
        private RenderContext clonedRenderContext = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="context">The render context to save.</param>
        public PushPopRenderContext(RenderContext context)
        {
            this.origRenderContext = context;
            this.clonedRenderContext = context.Clone() as RenderContext;
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            this.origRenderContext.Set(this.clonedRenderContext);
        }
    }
}
