﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;

using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a line segment.
    /// </summary>
    public class LineSegment : RenderableBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public LineSegment()
        {
            this.Vertex1 = PointF.Empty;
            this.Vertex2 = PointF.Empty;
            this.BorderColor = Color.Black;
            this.BorderThickness = 1.0f;
            this.PickThreshold = 3.0f;
            this.DashStyle = DashStyle.Solid;
        }

        /// <summary>
        /// Get or set the boundary of the rectangle.
        /// </summary>
        public new RectangleF Bounds
        {
            get
            {
                return new RectangleF(
                    Math.Min(this.Vertex1.X, this.Vertex2.X),
                    Math.Min(this.Vertex1.Y, this.Vertex2.Y),
                    Math.Abs(this.Vertex1.X - this.Vertex2.X),
                    Math.Abs(this.Vertex1.Y - this.Vertex2.Y));
            }
        }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF Vertex1 { get; set; }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF Vertex2 { get; set; }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF TransformedVertex1 { get; protected set; }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF TransformedVertex2 { get; protected set; }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF ClippedVertex1 { get; protected set; }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF ClippedVertex2 { get; protected set; }

        /// <summary>
        /// Get or set the flag indicating if the line segment is in the view rectangle.
        /// </summary>
        public bool IsInViewRectangle { get; protected set; }

        /// <summary>
        /// Get or set the threshold for mouse picking.
        /// </summary>
        public float PickThreshold { get; set; }

        /// <summary>
        /// Get or set the fill color of the item.
        /// </summary>
        public new Color FillColor
        {
            get { return Color.Transparent; }
        }

        /// <summary>
        /// Gets or sets the dash style.
        /// </summary>
        public DashStyle DashStyle { get; set; }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public override bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            // Compute the distance from the point.
            float distance = MathUtility.ComputePointLineDistance(
                p,
                this.TransformedVertex1,
                this.TransformedVertex2);

            return distance <= this.PickThreshold;
        }

        /// <summary>
        /// Render the item.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>False when the item is clipped and need not to be rendered.</returns>
        public override bool Draw(Graphics g)
        {
            if (this.Visible == false)
            {
                return false;
            }

            if (this.BorderThickness <= 0.0f)
            {
                return false;
            }

            if ((int)this.TransformedVertex1.X == (int)this.TransformedVertex2.X &&
                (int)this.TransformedVertex1.Y == (int)this.TransformedVertex2.Y)
            {
                // The line segment's length is zero, don't render it.
                return false;
            }

            if (this.IsInViewRectangle == false)
            {
                // The line segment does not lie in the view rectangle, don't render it.
                return false;
            }

            if (this.IsUpdateAndRenderingSuspended == false)
            {
                this.SuspendUpdateAndRendering();

                this.DrawBackground(g);
                this.DrawForeground(g);
                this.DrawBorder(g);
                this.DrawChildren(g);

                this.ResumeUpdateAndRendering();

                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            base.UpdateSelf(context);

            PointF[] points = { this.Vertex1, this.Vertex2 };

            // Transform the vertices with the transformation matrix.
            context.TransformationMatrix.TransformPoints(points);

            this.TransformedVertex1 = points[0];
            this.TransformedVertex2 = points[1];

            this.TransformedBounds =
                new RectangleF(
                    Math.Min(this.TransformedVertex1.X, this.TransformedVertex2.X),
                    Math.Min(this.TransformedVertex1.Y, this.TransformedVertex2.Y),
                    Math.Abs(this.TransformedVertex1.X - this.TransformedVertex2.X),
                    Math.Abs(this.TransformedVertex1.Y - this.TransformedVertex2.Y));

            // Do clipping with the view rectangle.
            PointF p1 = this.TransformedVertex1;
            PointF p2 = this.TransformedVertex2;
            if (context.ViewRectangle.IsEmpty == false)
            {
                this.IsInViewRectangle = RenderUtility.ClipLineSegment(
                    context.ViewRectangle,
                    this.TransformedVertex1,
                    this.TransformedVertex2,
                    out p1,
                    out p2);
            }
            else
            {
                this.IsInViewRectangle = false;
            }

            this.ClippedVertex1 = p1;
            this.ClippedVertex2 = p2;

            this.ClippedBounds =
                new RectangleF(
                    Math.Min(this.ClippedVertex1.X, this.ClippedVertex2.X),
                    Math.Min(this.ClippedVertex1.Y, this.ClippedVertex2.Y),
                    Math.Abs(this.ClippedVertex1.X - this.ClippedVertex2.X),
                    Math.Abs(this.ClippedVertex1.Y - this.ClippedVertex2.Y));
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            // Draw the line segment.
            using (Pen pen = new Pen(this.BorderColor, this.BorderThickness))
            {
                pen.DashStyle = this.DashStyle;
                if (this.DashStyle == DashStyle.Dash)
                {
                    pen.DashPattern = new float[] { 5.0f, 5.0f };
                }

                g.DrawLine(pen, this.ClippedVertex1, this.ClippedVertex2);
            }
        }
    }
}
