﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Schema;

using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Primitives
{
    /// <summary>
    /// The base abstract primitive class.
    /// </summary>
    /// <typeparam name="TValue">The value type.</typeparam>
    public abstract class PrimitiveBase<TValue> : IPrimitive, IEnumerable<TValue>
    {
        /// <summary>
        /// GetHashCode()で使用するハッシュテーブル.
        /// </summary>
        private static readonly int[] HashTable = { 0x77073096, 0x6e0e612c, 0x690951ba, 0x076dc419 };

        /// <summary>
        /// Constructor.
        /// </summary>
        protected PrimitiveBase()
        {
            this.Values = new TValue[this.Count];
            this.SetZero();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source to copy from.</param>
        protected PrimitiveBase(PrimitiveBase<TValue> src)
        {
            this.Values = new TValue[this.Count];
            this.Set(src);
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source array.</param>
        protected PrimitiveBase(TValue[] src)
        {
            this.Values = new TValue[this.Count];
            this.Set(src);
        }

        /// <summary>
        /// Get the number of values the primitive contains.
        /// </summary>
        public abstract int Count { get; }

        /// <summary>
        /// Get the type of the values the primitive contains.
        /// </summary>
        public abstract ValueTypes ValueType { get; }

        /// <summary>
        /// Get or set the elements of the primitive.
        /// </summary>
        protected TValue[] Values { get; set; }

        /// <summary>
        /// Get or set the value at the specified index.
        /// </summary>
        /// <param name="index">The index.</param>
        /// <returns>The value.</returns>
        public virtual TValue this[int index]
        {
            get
            {
                if (index < 0 || index >= this.Count)
                {
                    throw new IndexOutOfRangeException();
                }

                return this.Values[index];
            }

            set
            {
                if (index < 0 || index >= this.Count)
                {
                    throw new IndexOutOfRangeException();
                }

                this.Values[index] = value;
            }
        }

        /// <summary>
        /// Get or set the value at the specified index.
        /// </summary>
        /// <param name="index">The index.</param>
        /// <returns>The value.</returns>
        object IPrimitive.this[int index]
        {
            get
            {
                return this[index];
            }

            set
            {
                this[index] = (TValue)value;
            }
        }

        /// <summary>
        /// Get an array that contains all the values of the primitive.
        /// </summary>
        /// <returns>The array contains all the values.</returns>
        public virtual TValue[] ToArray()
        {
            return (TValue[])this.Values.Clone();
        }

        /// <summary>
        /// Set value from the given source.
        /// The set method accepts a primitive or an array with same element type.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public virtual bool Set(object src)
        {
            if (src is TValue[])
            {
                // It's an array of the correct element type.
                TValue[] typedSrc = src as TValue[];
                for (int i = 0; i < typedSrc.Length && i < this.Count; ++i)
                {
                    this.Values[i] = typedSrc[i];
                }

                return true;
            }
            else if (src is PrimitiveBase<TValue>)
            {
                // It's a primitive of the correct element type.
                var typedSrc = src as PrimitiveBase<TValue>;
                if ((typedSrc == null) ||
                    (typedSrc.Count != this.Count))
                {
                    return false;
                }

                for (int i = 0; i < this.Count; ++i)
                {
                    this.Values[i] = typedSrc[i];
                }

                return true;
            }

            return false;
        }

        /// <summary>
        /// Set all the values of the primitive to zero.
        /// </summary>
        public abstract void SetZero();

        /// <summary>
        /// Set all the values of the primitive to one.
        /// </summary>
        public abstract void SetOne();

        /// <summary>
        /// Clone this object.
        /// </summary>
        /// <returns>The created instance.</returns>
        public virtual object Clone()
        {
            var newInstance = Activator.CreateInstance(this.GetType()) as IPrimitive;

            newInstance.Set(this);

            return newInstance;
        }

        /// <summary>
        /// Get an enumerator.
        /// </summary>
        /// <returns>The primitive enumerator.</returns>
        public IEnumerator<TValue> GetEnumerator()
        {
            return new PrimitiveEnumerator(this);
        }

        /// <summary>
        /// Get an enumerator.
        /// </summary>
        /// <returns>The primitive enumerator.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return new PrimitiveEnumerator(this);
        }

        /// <summary>
        /// Compare with the given object and check if they are equal.
        /// </summary>
        /// <param name="other">The object to compare with.</param>
        /// <returns>True if the instances are equal.</returns>
        public override bool Equals(object other)
        {
            var typedSrc = other as PrimitiveBase<TValue>;
            if ((typedSrc == null) ||
                (typedSrc.Count != this.Count))
            {
                return false;
            }

            for (int i = 0; i < this.Count; ++i)
            {
                if (this.Values[i].Equals(typedSrc[i]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュコードを取得する.
        /// </summary>
        /// <returns>ハッシュコードの値.</returns>
        /// <remarks>異なる成分を持つが、同じハッシュ値が生成される
        /// 可能性があるため、利用を薦めない.</remarks>
        public override int GetHashCode()
        {
            int num = 42;
            unchecked
            {
                for (int i = 0; i < this.Count; ++i)
                {
                    int hash = this.Values[i].GetHashCode();
                    num = (num * 37) + hash;
                    num ^= hash >> 16;
                    num ^= HashTable[(hash >> 30) & 3];
                }
            }

            return num;
        }

        /// <summary>
        /// Convert the object to a string representation.
        /// </summary>
        /// <returns>The converted string.</returns>
        public override string ToString()
        {
            // Basically, there will only be primitives with elements less then four.
            switch (this.Count)
            {
                case 1:
                    return string.Format("{0}", this.Values[0]);

                case 2:
                    return string.Format(
                                         "{0}, {1}",
                                         this.Values[0],
                                         this.Values[1]);

                case 3:
                    return string.Format(
                                         "{0}, {1}, {2}",
                                         this.Values[0],
                                         this.Values[1],
                                         this.Values[2]);

                case 4:
                    return string.Format(
                                         "{0}, {1}, {2}, {3}",
                                         this.Values[0],
                                         this.Values[1],
                                         this.Values[2],
                                         this.Values[3]);
            }

            // If the element count is more than 4.
            string output = string.Empty;
            for (int i = 0; i < this.Count; ++i)
            {
                if (i > 0)
                {
                    output += ", ";
                }

                output += this.Values[i].ToString();
            }

            return output;
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            var lineInfo = reader as IXmlLineInfo;

            string content = reader.ReadElementContentAsString();

            // Check if the XML element value is valid.
            if (string.IsNullOrEmpty(content) == true)
            {
                if (lineInfo != null)
                {
                    throw new XmlException(
                        "The XML node is empty and cannot be read.",
                        null,
                        lineInfo.LineNumber,
                        lineInfo.LinePosition);
                }
                else
                {
                    throw new XmlException("The XML node is empty and cannot be read.");
                }
            }

            // Split the comma-separated string.
            string[] tokens = content.Split(',');

            int i = 0;
            foreach (string token in tokens)
            {
                // Parse the elements and store them.
                TValue value;
                if (this.ParseElement(token.Trim(), out value) == true)
                {
                    this.Values[i] = value;

                    ++i;
                    if (i >= this.Count)
                    {
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteValue(this.ToString());
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Parse the given string to a value of the element type.
        /// </summary>
        /// <param name="strElement">The string to parse.</param>
        /// <param name="value">The parsed value.</param>
        /// <returns>True on success.</returns>
        protected abstract bool ParseElement(
                                             string strElement,
                                             out TValue value);

        /// <summary>
        /// Enumerator class for the primitives.
        /// </summary>
        public class PrimitiveEnumerator : IEnumerator<TValue>
        {
            /// <summary>The enumerating index.</summary>
            private int currEnumeratingIndex = -1;

            /// <summary>The primitive that is currently enumerating.</summary>
            private PrimitiveBase<TValue> primitive;

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="primitive">The enumerating primitive.</param>
            public PrimitiveEnumerator(PrimitiveBase<TValue> primitive)
            {
                this.primitive = primitive;
            }

            /// <summary>
            /// Get the current item.
            /// </summary>
            object System.Collections.IEnumerator.Current
            {
                get { return this.Current; }
            }

            /// <summary>
            /// Get the current item with type.
            /// </summary>
            public TValue Current
            {
                get { return this.primitive[this.currEnumeratingIndex]; }
            }

            /// <summary>
            /// Dispose the enumerator.
            /// </summary>
            public void Dispose()
            {
                this.Reset();
            }

            /// <summary>
            /// Move to the next item.
            /// </summary>
            /// <returns>True on success.</returns>
            public bool MoveNext()
            {
                ++this.currEnumeratingIndex;
                if (this.currEnumeratingIndex < 0 ||
                    this.currEnumeratingIndex >= this.primitive.Count)
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }

            /// <summary>
            /// Reset the enumerator.
            /// </summary>
            public void Reset()
            {
                this.currEnumeratingIndex = 0;
            }
        }
    }
}
