﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Drawing;

namespace EffectMaker.Foundation.Primitives
{
    /// <summary>
    /// class for RGBA color.
    /// </summary>
    public class ColorRgba : Primitivef
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public ColorRgba()
        {
            this.SetOne();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source to copy from.</param>
        public ColorRgba(Primitivef src)
            : base(src)
        {
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source array.</param>
        public ColorRgba(float[] src)
            : base(src)
        {
        }

        /// <summary>
        /// Setter constructor.
        /// </summary>
        /// <param name="r">The R color value to set.</param>
        /// <param name="g">The G color value to set.</param>
        /// <param name="b">The B color value to set.</param>
        /// <param name="a">The A color value to set.</param>
        public ColorRgba(
                         float r,
                         float g,
                         float b,
                         float a) :
            this()
        {
            this.Set(r, g, b, a);
        }

        /// <summary>
        /// Get the number of value elements the primitive contains.
        /// </summary>
        public override int Count
        {
            get { return 4; }
        }

        /// <summary>
        /// Get or set the red element of the color.
        /// </summary>
        public float R
        {
            get { return this.Values[0]; }
            set { this.Values[0] = value; }
        }

        /// <summary>
        /// Get or set the green element of the color.
        /// </summary>
        public float G
        {
            get { return this.Values[1]; }
            set { this.Values[1] = value; }
        }

        /// <summary>
        /// Get or set the blue element of the color.
        /// </summary>
        public float B
        {
            get { return this.Values[2]; }
            set { this.Values[2] = value; }
        }

        /// <summary>
        /// Get or set the alpha element of the color.
        /// </summary>
        public float A
        {
            get { return this.Values[3]; }
            set { this.Values[3] = value; }
        }

        /// <summary>
        /// Convert from Windows color structure.
        /// </summary>
        /// <param name="color">The Windows color structure.</param>
        /// <returns>The created ColorRgba instance.</returns>
        public static ColorRgba FromWinColor(Color color)
        {
            return new ColorRgba(
                (float)color.R / 255.0f,
                (float)color.G / 255.0f,
                (float)color.B / 255.0f,
                (float)color.A / 255.0f);
        }

        /// <summary>
        /// Set value from the given source.
        /// The set method accepts a primitive or an array with same element type.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            if (src is Color)
            {
                var color = (Color)src;
                this.Set(
                    (float)color.R / 255.0f,
                    (float)color.G / 255.0f,
                    (float)color.B / 255.0f,
                    (float)color.A / 255.0f);

                return true;
            }
            else
            {
                return base.Set(src);
            }
        }

        /// <summary>
        /// Set value to the instance.
        /// </summary>
        /// <param name="r">The R value to set.</param>
        /// <param name="g">The G value to set.</param>
        /// <param name="b">The B value to set.</param>
        /// <param name="a">The A value to set.</param>
        /// <returns>True on success.</returns>
        public bool Set(
                        float r,
                        float g,
                        float b,
                        float a)
        {
            this.Values[0] = r;
            this.Values[1] = g;
            this.Values[2] = b;
            this.Values[3] = a;

            return true;
        }

        /// <summary>
        /// Convert to Windows color structure.
        /// </summary>
        /// <returns>The Windows color structure.</returns>
        public Color ToWinColor()
        {
            return Color.FromArgb(
                (byte)(Utility.MathUtility.Clamp(this.A, 0.0f, 1.0f) * 255.0f),
                (byte)(Utility.MathUtility.Clamp(this.R, 0.0f, 1.0f) * 255.0f),
                (byte)(Utility.MathUtility.Clamp(this.G, 0.0f, 1.0f) * 255.0f),
                (byte)(Utility.MathUtility.Clamp(this.B, 0.0f, 1.0f) * 255.0f));
        }
    }
}
