﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace EffectMaker.Foundation.Log
{
    /// <summary>
    /// ログレベルです。
    /// </summary>
    public enum LogLevels
    {
        /// <summary>
        /// Profiling
        /// </summary>
        Profile,

        /// <summary>
        /// Debug
        /// </summary>
        Debug,

        /// <summary>
        /// Information
        /// </summary>
        Information,

        /// <summary>
        /// Warning
        /// </summary>
        Warning,

        /// <summary>
        /// Error
        /// </summary>
        Error,

        /// <summary>
        /// Fatal error
        /// </summary>
        Fatal
    }

    /// <summary>
    /// ログを管理するクラスです。
    /// </summary>
    public static class Logger
    {
        /// <summary>ログ出力先を文字列で列挙するときのデリミタ</summary>
        private static char[] destinationDelimiters = new char[] { ',' };

        /// <summary>
        /// ログハンドラのリスト
        /// </summary>
        private static Dictionary<string, ILogHandler> logHandlers = new Dictionary<string, ILogHandler>();

        /// <summary>
        /// ロガー本体
        /// </summary>
        private static ThreadSafeLogger threadSafelogger = null;

        /// <summary>
        /// デフォルトのログレベルを取得または設定します。
        /// </summary>
        public static LogLevels DefaultLogLevel { get; set; }

        /// <summary>
        /// デフォルトのログ出力先を取得または設定します。
        /// </summary>
        public static string DefaultLogDestination { get; set; }

        /// <summary>
        /// 初期化処理を行います。
        /// </summary>
        public static void Initialize()
        {
            Debug.Assert(Logger.threadSafelogger == null);

            Logger.threadSafelogger = new ThreadSafeLogger();
            Logger.threadSafelogger.Initialize();
        }

        /// <summary>
        /// 破棄処理を行います。
        /// </summary>
        public static void Release()
        {
            if (Logger.threadSafelogger != null)
            {
                Logger.threadSafelogger.Release();
                Logger.threadSafelogger = null;
            }
        }

        /// <summary>
        /// ログハンドラを登録します。
        /// </summary>
        /// <param name="handler">ログハンドラ</param>
        public static void RegisterLogHandler(ILogHandler handler)
        {
            Debug.Assert(Logger.threadSafelogger != null);

            Logger.threadSafelogger.RegisterLogHandler(handler);
        }

        /// <summary>
        /// ログハンドラの登録を解除します。
        /// </summary>
        /// <param name="handler">ログハンドラ</param>
        public static void UnregisterLogHandler(ILogHandler handler)
        {
            Debug.Assert(Logger.threadSafelogger != null);

            Logger.threadSafelogger.UnregisterLogHandler(handler);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void Log(string format, params object[] args)
        {
            Logger.LogWithDepth(1, format, args);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void Log(LogLevels logLevel, string format, params object[] args)
        {
            Logger.LogWithDepth(1, logLevel, format, args);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void Log(string destination, LogLevels logLevel, string format, params object[] args)
        {
            Logger.LogWithDepth(1, destination, logLevel, format, args);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="depth">スタック深度</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void LogWithDepth(int depth, string format, params object[] args)
        {
            Logger.EnqueueMessage(depth + 1, Logger.DefaultLogDestination, Logger.DefaultLogLevel, format, args);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="depth">スタック深度</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void LogWithDepth(int depth, LogLevels logLevel, string format, params object[] args)
        {
            Logger.EnqueueMessage(depth + 1, Logger.DefaultLogDestination, logLevel, format, args);
        }

        /// <summary>
        /// ログメッセージを出力します。
        /// </summary>
        /// <param name="depth">スタック深度</param>
        /// <param name="destination">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        public static void LogWithDepth(int depth, string destination, LogLevels logLevel, string format, params object[] args)
        {
            Logger.EnqueueMessage(depth + 1, destination, logLevel, format, args);
        }

        /// <summary>
        /// 例外発生メッセージを出力します。
        /// </summary>
        /// <param name="e">例外情報</param>
        public static void LogException(Exception e)
        {
            Logger.EnqueueExceptionMessage(1, Logger.DefaultLogDestination, Logger.DefaultLogLevel, e);
        }

        /// <summary>
        /// 例外発生メッセージを出力します。
        /// </summary>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="e">例外情報</param>
        public static void LogException(LogLevels logLevel, Exception e)
        {
            Logger.EnqueueExceptionMessage(1, Logger.DefaultLogDestination, logLevel, e);
        }

        /// <summary>
        /// 例外発生メッセージを出力します。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="e">例外情報</param>
        public static void LogException(string destination, LogLevels logLevel, Exception e)
        {
            Logger.EnqueueExceptionMessage(1, destination, logLevel, e);
        }

        /// <summary>
        /// ログメッセージをキューに追加します。
        /// </summary>
        /// <param name="depth">スタック深度</param>
        /// <param name="destination">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="format">書式指定文字列</param>
        /// <param name="args">書式設定オブジェクト</param>
        private static void EnqueueMessage(int depth, string destinations, LogLevels logLevel, string format, params object[] args)
        {
            if (Logger.threadSafelogger == null) return;

            // コールスタックを取得
            StackFrame callStack = new StackFrame(depth + 1, true);

            string[] destinationList = new string[0];

            // 出力先リストを作成
            if (string.IsNullOrEmpty(destinations) == false)
            {
                destinationList = destinations.Split(destinationDelimiters, StringSplitOptions.RemoveEmptyEntries);

                if (destinationList == null)
                {
                    destinationList = new string[0];
                }
            }

            string message;

            // 出力メッセージを書式変換
            if (args == null || args.Length <= 0)
            {
                message = format;
            }
            else
            {
                message = string.Format(format, args);
            }

            // ログメッセージをキューに追加
            Logger.threadSafelogger.EnqueueMessage(destinationList, logLevel, message, callStack);
        }

        /// <summary>
        /// 例外発生メッセージをキューに追加します。
        /// </summary>
        /// <param name="depth">スタック深度</param>
        /// <param name="destination">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="e">例外情報</param>
        private static void EnqueueExceptionMessage(int depth, string destinations, LogLevels logLevel, Exception e)
        {
            if (Logger.threadSafelogger == null) return;

            // コールスタックを取得
            StackFrame callStack = new StackFrame(depth + 1, true);

            string[] destinationList = new string[0];

            // 出力先リストを作成
            if (string.IsNullOrEmpty(destinations) == false)
            {
                destinationList = destinations.Split(destinationDelimiters, StringSplitOptions.RemoveEmptyEntries);

                if (destinationList == null)
                {
                    destinationList = new string[0];
                }
            }

            StringBuilder message = new StringBuilder();

            // 例外情報メッセージを作成
            {
                string file = Path.GetFileNameWithoutExtension(callStack.GetFileName());
                int line = callStack.GetFileLineNumber();

                message.AppendLine(string.Format("{0} ({1}): Exception occurred.", file, line));
                message.AppendLine(e.Message);
            }

            // ログメッセージをキューに追加
            Logger.threadSafelogger.EnqueueMessage(destinationList, logLevel, message.ToString(), callStack);
        }
    }
}
