﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Input
{
    /// <summary>
    /// An Executable that aggregates several sub Executables.
    /// </summary>
    public class AggregateExecutable : IExecutable
    {
        /// <summary>
        /// Backing field for the IsEnabled property.
        /// </summary>
        private bool isEnabled = true;

        /// <summary>
        /// Stores the aggregated executable instances.
        /// </summary>
        private IEnumerable<IExecutable> executables;

        /// <summary>
        /// Initializes the AggregateExecutable instance.
        /// </summary>
        /// <param name="executables">The executables to aggregate.</param>
        public AggregateExecutable(IEnumerable<IExecutable> executables)
        {
            this.executables = executables;
        }

        /// <summary>
        /// Initializes the AggregateExecutable instance.
        /// </summary>
        /// <param name="executables">The executables to aggregate.</param>
        public AggregateExecutable(params IExecutable[] executables)
            : this((IEnumerable<IExecutable>)executables)
        {
        }

        /// <summary>
        /// Event raised when the executable availability changes.
        /// </summary>
        public event EventHandler CanExecuteChanged;

        /// <summary>
        /// Gets or sets whether the executable is available or not.
        /// </summary>
        public bool IsEnabled
        {
            get
            {
                return this.isEnabled;
            }

            set
            {
                if (this.isEnabled == value)
                {
                    return;
                }

                this.isEnabled = value;

                this.OnCanExecuteChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// Checks whether the current Executable can be executed or not.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        /// <returns>Returns true if can be executed, false otherwise.</returns>
        public virtual bool CanExecute(object parameter)
        {
            if (this.IsEnabled == false)
            {
                return false;
            }

            foreach (IExecutable exec in this.executables)
            {
                if (exec.CanExecute(parameter) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Executes the Executable.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public virtual void Execute(object parameter)
        {
            foreach (IExecutable exec in this.executables)
            {
                exec.Execute(parameter);
            }
        }

        /// <summary>
        /// Called when the CanExecute method would return a different value.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnCanExecuteChanged(EventArgs e)
        {
            var handler = this.CanExecuteChanged;

            if (handler != null)
            {
                handler(this, e);
            }
        }
    }
}
