﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Extensions methods for IHierarchyObject interface.
    /// </summary>
    public static class HierarchyObjectExtensions
    {
        /// <summary>
        /// Find the parent of a given type.
        /// </summary>
        /// <typeparam name="T">The type of parent to find.</typeparam>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <param name="findNearest">Tells whether to find the
        /// nearest or the farthest parent.</param>
        /// <returns>Returns the parent if found, or default value of T otherwise.</returns>
        public static T FindParentOfType<T>(this IHierarchyObject origin, bool findNearest)
            where T : IHierarchyObject
        {
            T found = default(T);

            while (origin != null)
            {
                if (origin.Parent is T)
                {
                    found = (T)origin.Parent;
                    if (findNearest)
                    {
                        break;
                    }
                }

                origin = origin.Parent;
            }

            return found;
        }

        /// <summary>
        /// Find the nearest parent of a given type.
        /// </summary>
        /// <typeparam name="T">The type of parent to find.</typeparam>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <returns>Returns the parent if found, or default value of T otherwise.</returns>
        public static T FindNearestParentOfType<T>(this IHierarchyObject origin)
            where T : IHierarchyObject
        {
            return FindParentOfType<T>(origin, true);
        }

        /// <summary>
        /// Find the farthest parent of a given type.
        /// </summary>
        /// <typeparam name="T">The type of parent to find.</typeparam>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <returns>Returns the parent if found, or default value of T otherwise.</returns>
        public static T FindFarthestParentOfType<T>(this IHierarchyObject origin)
        where T : IHierarchyObject
        {
            return FindParentOfType<T>(origin, false);
        }

        /// <summary>
        /// Find the parent with a given TagAttribute.
        /// </summary>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <param name="tagIdentifier">The ID set in the TagAttribute.</param>
        /// <param name="findNearest">Tells whether to find the
        /// nearest or the farthest parent.</param>
        /// <returns>Returns the parent if found, or null otherwise.</returns>
        public static IHierarchyObject FindParentByTag(
            this IHierarchyObject origin,
            string tagIdentifier,
            bool findNearest)
        {
            IHierarchyObject found = null;

            if (origin == null)
            {
                return null;
            }

            while (origin.Parent != null)
            {
                TagAttribute[] tags = origin.Parent
                    .GetType()
                    .GetCustomAttributes(typeof(TagAttribute), false)
                    .Cast<TagAttribute>()
                    .ToArray();

                if (tags.FindIndex(t => t.ID == tagIdentifier) >= 0)
                {
                    found = origin.Parent;
                    if (findNearest)
                    {
                        break;
                    }
                }

                origin = origin.Parent;
            }

            return found;
        }

        /// <summary>
        /// Find the nearest parent with a given TagAttribute.
        /// </summary>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <param name="tagIdentifier">The ID set in the TagAttribute.</param>
        /// <returns>Returns the parent if found, or null otherwise.</returns>
        public static IHierarchyObject FindNearestParentByTag(
            this IHierarchyObject origin,
            string tagIdentifier)
        {
            return FindParentByTag(origin, tagIdentifier, true);
        }

        /// <summary>
        /// Find the farthest parent with a given TagAttribute.
        /// </summary>
        /// <param name="origin">The origin node from where to start to search.</param>
        /// <param name="tagIdentifier">The ID set in the TagAttribute.</param>
        /// <returns>Returns the parent if found, or null otherwise.</returns>
        public static IHierarchyObject FindFarthestParentByTag(
            this IHierarchyObject origin,
            string tagIdentifier)
        {
            return FindParentByTag(origin, tagIdentifier, false);
        }

        /// <summary>
        /// 子要素を再帰的に探索し、指定した条件に合致する要素を見つけて取得します。
        /// </summary>
        /// <param name="src">探しだすオブジェクト</param>
        /// <param name="target">探索対象のオブジェクト</param>
        /// <param name="func">一致条件を評価する関数</param>
        /// <returns>発見したオブジェクト(見つからなかったらnull)</returns>
        public static IHierarchyObject FindChildrenRecursive(
            IHierarchyObject src,
            IHierarchyObject target,
            Func<object, object, bool> func)
        {
            foreach (var child in target.Children)
            {
                if (func(src, child))
                {
                    return child;
                }

                var result = FindChildrenRecursive(src, child, func);
                if (result != null)
                {
                    return result;
                }
            }

            return null;
        }
    }
}
