﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.Foundation.Core;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Static class that contains all extension methods related to IEnumerable and IEnumerable(T)
    /// </summary>
    public static class EnumerableExtensions
    {
        /// <summary>
        /// Checks whether only one item matches the predicate.
        /// </summary>
        /// <typeparam name="T">The type of elements in the sequence.</typeparam>
        /// <param name="source">The source sequence.</param>
        /// <param name="predicate">The predicate to check instances upon.</param>
        /// <returns>Returns true if one and only one element matches the predicate,
        /// false otherwise.</returns>
        public static bool One<T>(this IEnumerable<T> source, Func<T, bool> predicate)
        {
            if (source == null)
            {
                throw new ArgumentNullException("source");
            }

            return source.Count(predicate) == 1;
        }

        /// <summary>
        /// Checks whether the array is null or empty.
        /// </summary>
        /// <param name="source">The source array to check.</param>
        /// <returns>Returns true if the array instance is null or if it is empty,
        /// false otherwise.</returns>
        public static bool IsNullOrEmpty(this Array source)
        {
            return source == null || source.Length == 0;
        }

        /// <summary>
        /// Checks whether the sequence is null or empty.
        /// </summary>
        /// <param name="source">The source sequence to check.</param>
        /// <returns>Returns true if the sequence instance is null or if it is empty,
        /// false otherwise.</returns>
        public static bool IsNullOrEmpty(this IEnumerable source)
        {
            if (source == null)
            {
                return true;
            }

            return source.GetEnumerator().MoveNext() == false;
        }

        /// <summary>
        /// Checks whether the sequence contains more than one element.
        /// (At least two elements)
        /// </summary>
        /// <param name="source">The source sequence to check.</param>
        /// <returns>Returns true if the sequence contains more than one element,
        /// false otherwise.</returns>
        public static bool MoreThanOne(this IEnumerable source)
        {
            if (source == null)
            {
                throw new ArgumentNullException("source");
            }

            var e = source.GetEnumerator();

            return e.MoveNext() && e.MoveNext();
        }

        /// <summary>
        /// Returns distinct element from a sequence.
        /// </summary>
        /// <typeparam name="T">The type of each element of the sequence.</typeparam>
        /// <param name="source">The source sequence.</param>
        /// <param name="getHashCode">The method to compute the hash code of each element.</param>
        /// <param name="equals">The method to determine equality of each element.</param>
        /// <returns>Returns a new sequence of distinct elements.</returns>
        public static IEnumerable<T> Distinct<T>(
            this IEnumerable<T> source,
            Func<T, int> getHashCode,
            Func<T, T, bool> equals)
        {
            if (source == null)
            {
                throw new ArgumentNullException("source");
            }

            if (equals == null)
            {
                throw new ArgumentNullException("equals");
            }

            if (getHashCode == null)
            {
                throw new ArgumentNullException("getHashCode");
            }

            return source.Distinct(new AnonymousEqualityComparer<T>(getHashCode, equals));
        }

        /// <summary>
        /// Runs a method for each item of the sequence.
        /// </summary>
        /// <typeparam name="T">Type of the item in the sequence.</typeparam>
        /// <param name="sequence">The sequence on which to run the method.</param>
        /// <param name="action">The method that run for each item and receives it as parameter.</param>
        /// <returns>Returns the input sequence for composability purpose.</returns>
        public static IEnumerable<T> ForEach<T>(this IEnumerable<T> sequence, Action<T> action)
        {
            if (sequence == null)
            {
                throw new ArgumentNullException("sequence");
            }

            if (action == null)
            {
                throw new ArgumentNullException("action");
            }

            foreach (var item in sequence)
            {
                action(item);
            }

            return sequence;
        }

        /// <summary>
        /// Determines whether two sequences are equal by comparing the elements
        /// by using the default equality comparer for their type.
        /// The order of each elements does not matter.
        /// </summary>
        /// <typeparam name="T">The type of the elements.</typeparam>
        /// <param name="self">The first sequence.</param>
        /// <param name="other">The other sequence.</param>
        /// <returns>True if the sequences contain the same elements.</returns>
        public static bool UnorderedSequenceEqual<T>(
            this IEnumerable<T> self,
            IEnumerable<T> other)
        {
            return self.All(e1 => other.Any(e2 => object.Equals(e1, e2)));
        }

        /// <summary>
        /// Find the index of a item in a sequence.
        /// </summary>
        /// <param name="sequence">The sequence of items.</param>
        /// <param name="item">The item to look for the index.</param>
        /// <returns>Returns the zero-based index of the item if found, or -1 otherwise.</returns>
        public static int FindIndex(this IEnumerable sequence, object item)
        {
            int index = 0;

            foreach (object current in sequence)
            {
                if (current == item)
                {
                    return index;
                }

                index++;
            }

            return -1;
        }

        /// <summary>
        /// Find the index of a item in a sequence.
        /// </summary>
        /// <typeparam name="T">Type of item.</typeparam>
        /// <param name="sequence">The sequence of items.</param>
        /// <param name="predicate">Describes the way to identify item.</param>
        /// <returns>Returns the zero-based index of the item if found, or -1 otherwise.</returns>
        public static int FindIndex<T>(this IEnumerable<T> sequence, Predicate<T> predicate)
        {
            int index = 0;

            foreach (T current in sequence)
            {
                if (predicate(current))
                {
                    return index;
                }

                index++;
            }

            return -1;
        }

        /// <summary>
        /// Wraps this object instance into an IEnumerable&lt;T&gt;
        /// consisting of a single item.
        /// </summary>
        /// <typeparam name="T">Type of the wrapped object.</typeparam>
        /// <param name="item">The object to wrap.</param>
        /// <returns>
        /// An IEnumerable&lt;T&gt; consisting of a single item.
        /// </returns>
        public static IEnumerable<T> Yield<T>(this T item)
        {
            yield return item;
        }
    }
}
