﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.Foundation.Debugging.Profiling
{
    /// <summary>
    /// Instrumentation class to measure and timely report actions.
    /// </summary>
    public class Instrumentation : IInstrumentation
    {
        /// <summary>
        /// Tells whether the debugger breaks on dispose.
        /// </summary>
        private bool breakDebuggerWhenDone;

        /// <summary>
        /// Strores the time measurement instance.
        /// </summary>
        private Stopwatch stopWatch;

        /// <summary>
        /// Stores the information of measured actions.
        /// </summary>
        private List<IInstrumentationStepInfo> privateInstrumentationSteps =
            new List<IInstrumentationStepInfo>();

        /// <summary>
        /// Stores a publically exposable sequence of instrumentation steps.
        /// </summary>
        private ReadOnlyCollection<IInstrumentationStepInfo> publicInstrumentationSteps;

        /// <summary>
        /// Initializes the Instrumentation instance, and start the stopwatch.
        /// </summary>
        /// <param name="breakDebuggerWhenDone">Tells whether the debugger
        /// should break at the end.</param>
        public Instrumentation(bool breakDebuggerWhenDone = false)
        {
            this.breakDebuggerWhenDone = breakDebuggerWhenDone;
            this.stopWatch = Stopwatch.StartNew();
            this.publicInstrumentationSteps =
                new ReadOnlyCollection<IInstrumentationStepInfo>(
                    this.privateInstrumentationSteps);
        }

        /// <summary>
        /// Gets the instrumentation steps.
        /// </summary>
        public IEnumerable<IInstrumentationStepInfo> Steps
        {
            get
            {
                return this.privateInstrumentationSteps;
            }
        }

        /// <summary>
        /// Restart the stopwatch.
        /// </summary>
        public void Restart()
        {
            this.stopWatch.Restart();
        }

        /// <summary>
        /// Record an instrumentation step.
        /// </summary>
        /// <param name="stepDescription">Description of the past step.</param>
        public void Step(string stepDescription)
        {
            long time = this.stopWatch.ElapsedTicks;

            if (string.IsNullOrWhiteSpace(stepDescription))
            {
                throw new ArgumentException("stepDescription");
            }

            this.privateInstrumentationSteps.Add(
                new InstrumentationStepInfo(stepDescription, time));

            this.stopWatch.Restart();
        }

        /// <summary>
        /// Accumulate elapsed time to the last step with same description.
        /// </summary>
        /// <param name="stepDescription">Description of the step.</param>
        public void Accumulate(string stepDescription)
        {
            long time = this.stopWatch.ElapsedTicks;

            if (string.IsNullOrWhiteSpace(stepDescription))
            {
                throw new ArgumentException("stepDescription");
            }

            var found = (InstrumentationStepInfo)this.privateInstrumentationSteps
                .LastOrDefault(s => s.Description == stepDescription);

            if (found == null)
            {
                this.privateInstrumentationSteps.Add(
                    new InstrumentationStepInfo(stepDescription, time));
            }
            else
            {
                found.Ticks += time;
                found.CallCount++;
            }

            this.stopWatch.Restart();
        }

        /// <summary>
        /// Shutdowns the instrumentation.
        /// </summary>
        public void Dispose()
        {
            this.Step("<last action>");
            this.stopWatch.Reset();

            if (this.breakDebuggerWhenDone)
            {
                var view = new WeakReference(this.Steps);
                Debugger.Break();
            }
        }

        /// <summary>
        /// Transform the instrumentation steps into string representation.
        /// </summary>
        /// <returns>Returns a string representation of the instrumentation step.</returns>
        public override string ToString()
        {
            return this.ToString(InstrumentationTimeUnit.Milliseconds, items => items);
        }

        /// <summary>
        /// Produce a report in a textual form.
        /// </summary>
        /// <param name="timeUnit">Unit of time for the current instrumentation.</param>
        /// <param name="hook">The hook delegate to transform the
        /// displayed instrumentation steps.</param>
        /// <returns>Returns the textually formatted report.</returns>
        public string ToString(
            InstrumentationTimeUnit timeUnit,
            Func<IEnumerable<IInstrumentationStepInfo>, IEnumerable<IInstrumentationStepInfo>> hook)
        {
            var sb = new StringBuilder();

            var totalTicks = (double)this.privateInstrumentationSteps.Sum(s => s.Ticks);

            double frequency = 0.0;
            if (timeUnit == InstrumentationTimeUnit.Milliseconds)
            {
                frequency = Stopwatch.Frequency / 1000.0;
            }
            else
            {
                frequency = Stopwatch.Frequency / (1000.0 * 1000.0);
            }

            string timeFormat = "us";
            if (timeUnit == InstrumentationTimeUnit.Milliseconds)
            {
                timeFormat = "ms";
            }

            foreach (IInstrumentationStepInfo step in hook(this.privateInstrumentationSteps))
            {
                double time = step.Ticks / frequency;
                double percent = (step.Ticks / totalTicks) * 100.0;

                string callCount = string.Empty;
                if (step.CallCount > 1)
                {
                    callCount = string.Format(" [called {0} times]", step.CallCount);
                }

                sb.AppendLine(string.Format(
                    "{0} : {1:0.##} {2} ({3:0.##} %){4}",
                    step.Description,
                    time,
                    timeFormat,
                    percent,
                    callCount));
            }

            return sb.ToString();
        }
    }
}
