﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel.Composition.Hosting;
using System.ComponentModel.Composition.Primitives;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Composition
{
    /// <summary>
    /// A composer class that can aggregate
    /// </summary>
    /// <typeparam name="T">The specific type of modules to compose.</typeparam>
    public class Composer<T>
    {
        /// <summary>
        /// Stores the thread synchronization object.
        /// </summary>
        private readonly object syncRoot = new object();

        /// <summary>
        /// Stores the paths used to loop for modules to compose.
        /// </summary>
        private readonly List<PathLookupInfo> paths = new List<PathLookupInfo>();

        /// <summary>
        /// Add a path to look for modules to compose.
        /// </summary>
        /// <param name="path">An absolute base path for lookup.</param>
        public void AddPathLookup(string path)
        {
            lock (this.syncRoot)
            {
                this.paths.Add(new PathLookupInfo(path));
            }
        }

        /// <summary>
        /// Add a path to look for modules to compose.
        /// </summary>
        /// <param name="path">An absolute base path for lookup.</param>
        /// <param name="includeSubDirectories">Tells whether sub directories
        /// are included or not.</param>
        /// <param name="searchPatterns">Search patterns used for lookup.</param>
        public void AddPathLookup(string path, bool includeSubDirectories, params string[] searchPatterns)
        {
            lock (this.syncRoot)
            {
                this.paths.Add(new PathLookupInfo(path, includeSubDirectories, searchPatterns));
            }
        }

        /// <summary>
        /// Add a PathLookupInfo to look for modules to compose.
        /// </summary>
        /// <param name="path">Instance of PathLookupInfo to add.</param>
        public void AddPathLookup(PathLookupInfo path)
        {
            lock (this.syncRoot)
            {
                this.paths.Add(path);
            }
        }

        /// <summary>
        /// Run composition process and get an array of modules.
        /// </summary>
        /// <returns>Returns an array of composed modules.</returns>
        public T[] Compose()
        {
            return this.Compose(null);
        }

        /// <summary>
        /// Run composition asynchronously and get an array of modules.
        /// </summary>
        /// <returns>Returns a task of array of composed modules.</returns>
        public Task<T[]> ComposeAsync()
        {
            return this.ComposeAsync(null);
        }

        /// <summary>
        /// Run composition asynchronously and get an array of modules.
        /// </summary>
        /// <param name="errorReporter">An exception reporter to send
        /// exceptions that occured to.</param>
        /// <returns>Returns a task of array of composed modules.</returns>
        public Task<T[]> ComposeAsync(IExceptionReporter errorReporter)
        {
            return Task.Factory.StartNew(() => this.Compose(errorReporter));
        }

        /// <summary>
        /// Run composition and get an array of modules.
        /// </summary>
        /// <param name="reporter">An exception reporter to send
        /// exceptions that occured to.</param>
        /// <returns>Returns an array of composed modules.</returns>
        public T[] Compose(IExceptionReporter reporter)
        {
            var catalogs = new List<ComposablePartCatalog>();

            catalogs.AddRange(this.GetAssemblies(reporter));

            var aggregateCatalog = new AggregateCatalog(catalogs);
            var compositionContainer = new CompositionContainer(aggregateCatalog);

            try
            {
                return compositionContainer.GetExportedValues<T>().ToArray();
            }
            catch (Exception ex)
            {
                if (reporter != null)
                {
                    reporter.Report(ex);
                    return new T[0];
                }

                throw;
            }
        }

        /// <summary>
        /// Creates an AssemblyCatalog instance from an assembly absolute filename.
        /// </summary>
        /// <param name="file">The assembly absolute filename.</param>
        /// <param name="reporter">An exception reporter to
        /// send exceptions that occured to.</param>
        /// <returns>Returns an AssemblyCatalog instance, or null otherwise.</returns>
        private static AssemblyCatalog CreateAssemblyCatalog(
            string file,
            IExceptionReporter reporter)
        {
            try
            {
                return new AssemblyCatalog(file);
            }
            catch (Exception ex)
            {
                if (reporter != null)
                {
                    reporter.Report(ex);
                }

                return null;
            }
        }

        /// <summary>
        /// Retrieves a sequence of files based on the defined lookup information.
        /// </summary>
        /// <returns>Returns a sequence of file absolute path.</returns>
        private IEnumerable<string> RetrieveFileList()
        {
            PathLookupInfo[] coldPaths;

            lock (this.syncRoot)
            {
                coldPaths = this.paths.ToArray();
            }

            return PathLookupInfoUtility.RetrieveFiles(coldPaths).ToArray();
        }

        /// <summary>
        /// Get a sequence of assembly based on files defined by lookup information.
        /// </summary>
        /// <param name="reporter">An exception reporter to
        /// send exceptions that occured to.</param>
        /// <returns>Returns a sequence of AssemblyCatalog
        /// containing assembly information.</returns>
        private IEnumerable<AssemblyCatalog> GetAssemblies(IExceptionReporter reporter)
        {
            var assemblyCatalogs = this.RetrieveFileList()
                .Select(f => CreateAssemblyCatalog(f, reporter))
                .Where(ac => ac != null);

            foreach (var assemblyCatalog in assemblyCatalogs)
            {
                try
                {
                    var dummy = assemblyCatalog.Parts.Any();
                }
                catch (Exception ex)
                {
                    if (reporter != null)
                    {
                        reporter.Report(ex);
                    }

                    continue;
                }

                yield return assemblyCatalog;
            }
        }
    }
}
