﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;

using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.Foundation.Command
{
    /// <summary>
    /// コマンドの実行タイプ.
    /// </summary>
    public enum ExecutionType
    {
        /// <summary>実行.</summary>
        Do,

        /// <summary>元に戻す.</summary>
        Undo,

        /// <summary>やり直し.</summary>
        Redo,
    }

    /// <summary>
    /// コマンドを管理するクラス.
    /// </summary>
    public class CommandManager
    {
        #region Member variables

        /// <summary>コマンドディクショナリ.</summary>
        private static Dictionary<string, Type> commandRegistry
            = new Dictionary<string, Type>();

        /// <summary>The command stack list.</summary>
        private static List<CommandStack> commandStack
            = new List<CommandStack>();

        /// <summary>The global command list.</summary>
        private static List<CommandHistoryData> globalCommandList
            = new List<CommandHistoryData>();

        /// <summary>アクティブターゲット.</summary>
        private static CommandStack activeStack = null;

        /// <summary>コマンド実行中を示すフラグ.</summary>
        private static bool isCommandExecuting = false;

        /// <summary>
        /// Undo/Redoバッファのサイズの最大数.
        /// -1の場合は、サイズは無限
        /// </summary>
        private static int undoRedoMaximum = -1;

        /// <summary>
        /// The is step command.
        /// </summary>
        //// private static KeepingMode isSequential = KeepingMode.NotSpecified;

        #endregion // Member variables

        /// <summary>
        /// コマンドデリゲート.
        /// </summary>
        /// <param name="e">コマンド.</param>
        public delegate void CommandEventHandler(CommandEventArgs e);

        /// <summary>
        /// アクティブターゲット変更イベントハンドラ.
        /// </summary>
        /// <param name="e">コマンド</param>
        public delegate void ActiveTargetChangedCommandEventHandler(CommandActiveTargetEventArgs e);

        /// <summary>
        /// 実行前通知イベント.
        /// </summary>
        public static event CommandEventHandler BeforeExecution = null;

        /// <summary>
        /// 実行後通知イベント.
        /// </summary>
        public static event CommandEventHandler CommandExecuted = null;

        /// <summary>
        /// アクティブターゲットの変更通知イベント.
        /// </summary>
        public static event ActiveTargetChangedCommandEventHandler ActiveTargetChanged = null;

        /// <summary>
        /// Undo/Redoバッファの最大サイズ, 無限On/Offを設定する.
        /// unlimitedがtrueだと、maximumにどのような値を渡してもUndo/Redoバッファのサイズは無限になる.
        /// </summary>
        /// <param name="maximum">Undo/Redoバッファの最大サイズ.</param>
        /// <param name="unlimited">Undo/Redoバッファサイズの無限On/Off</param>
        public static void SetUndoRedoParameters(int maximum, bool unlimited)
        {
            // バッファサイズが無限だったら、undoRedoMaximumを-1にする
            if (unlimited == true)
            {
                undoRedoMaximum = -1;
            }
            else
            {
                undoRedoMaximum = maximum;
            }
        }

        /// <summary>
        /// コマンドクラスを登録する.
        /// </summary>
        /// <param name="commandType">The type of the command.</param>
        /// <returns>True on success.</returns>
        public static bool RegisterCommand(Type commandType)
        {
            // Get our command name attribute from the type of the command.
            var attributes = commandType.GetCustomAttributes(typeof(AliasAttribute), false);
            bool result = (attributes.Length > 0) && (attributes[0] is AliasAttribute);
            Debug.Assert(
                result, "CommandManager.RegisterCommand : Failed to get the custome attribute data.");

            var commandName = ((AliasAttribute)attributes[0]).Alias;

            // 重複は許可しないため、キーがすでに登録されているかチェックする.
            Debug.Assert(
                commandRegistry.ContainsKey(commandName) == false,
                "CommandManager.RegisterCommand : 指定したコマンド名のクラスはすでに存在するため、登録に失敗しました。");

            // コマンド名をキーに、ディクショナリへ登録する.
            commandRegistry[commandName] = commandType;

            return true;
        }

        /// <summary>
        /// アクティブターゲットを設定する.
        /// </summary>
        /// <param name="target">ターゲット.</param>
        /// <returns>成功した場合、trueを返す.</returns>
        public static bool SetActiveTarget(object target)
        {
            // propertyOwnerから最上位のtargetを取得する.
            IHierarchyObject targetObj = target as IHierarchyObject;

            activeStack = null;

            // 遡ってスタックを見つける.
            var cmdTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(targetObj);
            if (cmdTarget != null)
            {
                CommandStack stack = FindCommandStackWithTarget(cmdTarget);
                if (stack != null)
                {
                    activeStack = stack;
                }
                else
                {
                    // コマンドスタックがない場合、新規作成する.
                    stack = CreateCommandStack(cmdTarget);
                    commandStack.Add(stack);

                    // アクティブなスタックへ設定する.
                    activeStack = stack;
                }
            }

            // スタック変更のイベントを発行する.
            if (ActiveTargetChanged != null)
            {
                ActiveTargetChanged(new CommandActiveTargetEventArgs()
                {
                    ActiveTargetObj = cmdTarget,
                    ActiveStack = activeStack,
                });
            }

            return false;
        }

        /// <summary>
        /// アクティブなコマンドスタックを取得する.
        /// </summary>
        /// <returns>アクティブなコマンドスタック.</returns>
        public static CommandStack GetActiveStack()
        {
            return activeStack;
        }

        /// <summary>
        /// コマンドを実行する.
        /// </summary>
        /// <param name="commandName">コマンド名.</param>
        /// <param name="args">コマンドへ渡す引数配列.</param>
        /// <returns>True on success.</returns>
        public static bool Execute(string commandName, object[] args)
        {
            if (isCommandExecuting == true)
            {
                return true;
            }

            // コマンドを作成する
            CommandBase cmd = CreateCommand(commandName);
            if (cmd == null)
            {
                return false;
            }

            // コマンドを初期化する
            bool result = InitializeCommand(cmd, args);
            if (result == false)
            {
                return false;
            }

            // コマンドを実行する
            result = Execute(cmd);

            return result;
        }

        /// <summary>
        /// コマンドを実行する.
        /// </summary>
        /// <param name="cmd">コマンド</param>
        /// <returns>True on success.</returns>
        public static bool Execute(CommandBase cmd)
        {
            if (isCommandExecuting == true)
            {
                return true;
            }

            // 多重実行禁止フラグを立てる.
            isCommandExecuting = true;

            try
            {
                if (cmd.IsInitialized == false)
                {
                    return false;
                }

                CommandEventArgs cmdArgs = new CommandEventArgs()
                {
                    ExecType   = ExecutionType.Do,
                    CancelFlag = false,
                };

                // 実行前イベントを発行.
                if (BeforeExecution != null)
                {
                    BeforeExecution(cmdArgs);
                }

                // 実行前イベントでキャンセルフラグが[true]にセットされて
                // いた場合、コマンドの実行を中止する.
                if (cmdArgs.CancelFlag == true)
                {
                    return true;
                }

                bool result = ExecuteInternal(cmd);
                if (result == false)
                {
                    return false;
                }

                // 実行済後イベントを発行.
                if (CommandExecuted != null)
                {
                    CommandExecuted(cmdArgs);
                }
            }
            finally
            {
                // 多重実行禁止フラグを解除.
                isCommandExecuting = false;
            }

            return true;
        }

        /// <summary>
        /// 元に戻す処理.
        /// </summary>
        /// <returns>True on success.</returns>
        public static bool Undo()
        {
            const ExecutionType ExecType = ExecutionType.Undo;

            if (isCommandExecuting == true)
            {
                return true;
            }

            // 多重実行禁止フラグを立てる.
            isCommandExecuting = true;

            CommandBase cmd = null;

            try
            {
                CommandStack cmdStack = activeStack;
                if (cmdStack == null)
                {
                    return true;
                }

                // スタック操作を行い、コマンドを取得する.
                cmd = cmdStack.GetCommandForUndo();

                if (cmd == null)
                {
                    return true; // スタックにコマンドがない.
                }

                CommandEventArgs cmdArgs = new CommandEventArgs()
                {
                    ExecType = ExecType,
                    CancelFlag = false,
                };

                // 実行前イベントを発行.
                if (BeforeExecution != null)
                {
                    BeforeExecution(cmdArgs);
                }

                // 実行前イベントでキャンセルフラグが[true]にセットされて
                // いた場合、コマンドの実行を中止する.
                if (cmdArgs.CancelFlag == true)
                {
                    return true;
                }

                bool result = cmd.PrepareExecution(ExecType);
                if (result == false)
                {
                    return true;
                }

                // コマンドヒストリーへ記録.
                globalCommandList.Add(
                    new CommandHistoryData()
                    {
                        Command = cmd,
                        Type = ExecType,
                    });

                // 元に戻す操作.
                result = cmd.Undo();
                if (result == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "CommandManager.Undo : Failed to call Undo method.");
                    return false;
                }

                cmd.FinalizeExecution(ExecType);

                // 実行後イベントを発行.
                if (CommandExecuted != null)
                {
                    CommandExecuted(cmdArgs);
                }
            }
            finally
            {
                // 多重実行禁止フラグを解除.
                isCommandExecuting = false;
            }

            // スタック操作を行わずに、コマンドを取得する.
            var nextCmd = activeStack.GetNextUndoCommand();
            if (nextCmd == null)
            {
                return true; // スタックにコマンドがない.
            }

            // コマンドの実行を継続する場合、再帰的にUndo()を呼び出す.
            if (nextCmd.KeepExecuting)
            {
                return Undo();
            }

            return true;
        }

        /// <summary>
        /// やり直し処理.
        /// </summary>
        /// <returns>True on success.</returns>
        public static bool Redo()
        {
            if (isCommandExecuting == true)
            {
                return true;
            }

            // 多重実行禁止フラグを立てる.
            isCommandExecuting = true;

            const ExecutionType ExecType = ExecutionType.Redo;

            CommandBase cmd = null;

            try
            {
                CommandStack cmdStack = activeStack;
                if (cmdStack == null)
                {
                    return true;
                }

                // スタック操作を行い、コマンドを取得する.
                cmd = cmdStack.GetCommandForRedo();

                if (cmd == null)
                {
                    return true; // スタックにコマンドがない.
                }

                CommandEventArgs cmdArgs = new CommandEventArgs()
                {
                    ExecType = ExecType,
                    CancelFlag = false,
                };

                // 実行前イベントを発行.
                if (BeforeExecution != null)
                {
                    BeforeExecution(cmdArgs);
                }

                // 実行前イベントでキャンセルフラグが[true]にセットされて
                // いた場合、コマンドの実行を中止する.
                if (cmdArgs.CancelFlag == true)
                {
                    return true;
                }

                bool result = cmd.PrepareExecution(ExecType);
                if (result == false)
                {
                    return true;
                }

                // コマンドヒストリーへ記録.
                globalCommandList.Add(
                    new CommandHistoryData()
                    {
                        Command = cmd,
                        Type = ExecType,
                    });

                // やり直し操作.
                result = cmd.Redo();
                if (result == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "CommandManager.Redo : Failed to call Redo method.");
                    return false;
                }

                cmd.FinalizeExecution(ExecType);

                // 実行後イベントを発行.
                if (CommandExecuted != null)
                {
                    CommandExecuted(cmdArgs);
                }
            }
            finally
            {
                // 多重実行禁止フラグを解除.
                isCommandExecuting = false;
            }

            // コマンドの実行を継続する場合、再帰的にRedo()を呼び出す.
            if (cmd.KeepExecuting)
            {
                return Redo();
            }

            return true;
        }

        /// <summary>
        /// コマンドスタックをクリアする.
        /// </summary>
        public static void ClearCommandStack()
        {
            // コマンドスタックをクリア.
            foreach (CommandStack item in commandStack)
            {
                item.ClearRedoBuffer();
                item.ClearUndoBuffer();
            }

            UpdateCommandHistoryView();
        }

        /// <summary>
        /// 履歴ウィンドウの更新
        /// </summary>
        public static void UpdateCommandHistoryView()
        {
            // 履歴ウィンドウを更新させるため実行後イベントを発行.
            CommandEventArgs cmdArgs = new CommandEventArgs()
            {
                ExecType = ExecutionType.Do,
                CancelFlag = false,
            };

            if (CommandExecuted != null)
            {
                CommandExecuted(cmdArgs);
            }
        }

        /// <summary>
        /// コマンドを作成する.
        /// </summary>
        /// <param name="commandName">コマンド名.</param>
        /// <returns>成功時にtrueを返す.</returns>
        protected static CommandBase CreateCommand(string commandName)
        {
            Type commandType;
            if (commandRegistry.TryGetValue(commandName, out commandType) == false)
            {
                Logger.Log(
                    LogLevels.Error,
                    "CommandManager.CreateCommand : The command is not found.");
                return null;
            }

            // コマンド名から、コマンドのインスタンスを作成し初期化する.
            var cmd = Activator.CreateInstance(commandType) as CommandBase;
            return cmd;
        }

        /// <summary>
        /// コマンドの初期化.
        /// </summary>
        /// <param name="cmd">コマンド.</param>
        /// <param name="args">引数.</param>
        /// <returns>成功時にtrueを返す.</returns>
        protected static bool InitializeCommand(CommandBase cmd, object[] args)
        {
            if (cmd.Initialize(args) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "CommandManager.InitializeCommand : Failed to call Initialize method.");
                return false;
            }

            return true;
        }

        /// <summary>
        /// コマンド実行(内部処理).
        /// </summary>
        /// <param name="cmd">コマンド.</param>
        /// <returns>成功時にtrueを返す.</returns>
        protected static bool ExecuteInternal(CommandBase cmd)
        {
            const ExecutionType ExecType = ExecutionType.Do;

            // Execute the command.
            if (cmd.PrepareExecution(ExecType) == false)
            {
                return true;
            }

            if (cmd.CanUndo)
            {
                // コマンドスタックインスタンスを作成し、コマンドを登録する.
                // ターゲットが[null]の場合、スタックへコマンドを追加しない.
                object target = cmd.GetTarget();
                if (target != null)
                {
                    CommandStack cmdStack = FindCommandStackWithTarget(target);
                    if (cmdStack == null)
                    {
                        // コマンドスタックがない場合、新規作成する.
                        cmdStack = CreateCommandStack(target);
                        commandStack.Add(cmdStack);

                        // アクティブなスタックへ設定する.
                        activeStack = cmdStack;
                    }

                    cmdStack.PushCommand(cmd);
                }
            }

            // グローバルコマンドヒストリーへ記録する.
            globalCommandList.Add(
                new CommandHistoryData()
                {
                    Command = cmd,
                    Type = ExecType,
                });

            // コマンドを実行.
            bool result = cmd.Do();
            if (result == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "CommandManager.ExecuteInternal : Failed to call Do method.");
                return false;
            }

            cmd.FinalizeExecution(ExecType);

            return true;
        }

        /// <summary>
        /// ターゲットオブジェクトを参照しているコマンドスタックを取得する.
        /// </summary>
        /// <param name="activeObject">検索対象のターゲットオブジェクト.</param>
        /// <returns>コンテナにある場合、有効なコマンドターゲットを返す.</returns>
        private static CommandStack FindCommandStackWithTarget(object activeObject)
        {
            foreach (CommandStack stack in commandStack)
            {
                if (object.ReferenceEquals(stack.GetTargetObject(), activeObject))
                {
                    return stack;
                }
            }

            return null;
        }

        /// <summary>
        /// コマンドスタックを作成する.
        /// </summary>
        /// <param name="targetObject">ターゲットオブジェクト.</param>
        /// <returns>新規作成されたコマンドスタック.</returns>
        private static CommandStack CreateCommandStack(object targetObject)
        {
            return new CommandStack(targetObject, undoRedoMaximum);
        }

        /// <summary>
        /// コマンドイベント.
        /// </summary>
        public class CommandEventArgs : EventArgs
        {
            /// <summary>
            /// 実行タイプ.
            /// </summary>
            public ExecutionType ExecType { get; set; }

            /// <summary>
            /// キャンセルフラグ。実行をキャンセルする場合、
            /// このフラグをtrueにセットする.
            /// </summary>
            public bool CancelFlag { get; set; }
        }

        /// <summary>
        /// アクティブターゲットの変更コマンドイベント.
        /// </summary>
        public class CommandActiveTargetEventArgs : EventArgs
        {
            /// <summary>
            /// アクティブターゲット.
            /// </summary>
            public IHierarchyObject ActiveTargetObj { get; set; }

            /// <summary>
            /// アクティブスタック.
            /// </summary>
            public object ActiveStack { get; set; }
        }

        /// <summary>
        /// グローバルコマンドヒストリーのためのクラス.
        /// </summary>
        private class CommandHistoryData
        {
            /// <summary>
            /// The command string.
            /// </summary>
            public object Command { get; set; }

            /// <summary>
            /// The type of execution.
            /// </summary>
            public ExecutionType Type { get; set; }
        }
    }
}
