﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef Common_H_
#define Common_H_

// 定数
// RStatus RTimeMeasure RXMLElement
// RDataStream RBinDataStream
// RUserData
// RUnicodeCArray
// RFileBuf

//=============================================================================
// include
//=============================================================================
#include <cassert>
#include <cmath>
#include <ctime>
#include <iomanip>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <map>

#include <windows.h>
#include <shlwapi.h>

// Cafe headers
#include <types.h>
#include <cafe/gx2/gx2Constant.h>
#include <cafe/gx2/gx2Enum.h>
#pragma warning(push)
    #pragma warning(disable: 4100) // 引数は関数の本体部で 1 度も参照されません。
    #include <cafe/gx2/gx2Misc.h>
#pragma warning(pop)

//=============================================================================
//! @brief ビット数を指定した数値を表す型の定義です。
//=============================================================================
#ifndef __TYPES_H__
typedef __int8  s8;
typedef __int16 s16;
typedef __int32 s32;
typedef __int64 s64;
typedef unsigned __int8  u8;
typedef unsigned __int16 u16;
typedef unsigned __int32 u32;
typedef unsigned __int64 u64;
typedef float  f32;
typedef double f64;
#endif

//=============================================================================
// nwFtx ネームスペースを開始します。
//=============================================================================
namespace nwFtx {

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define R_INTERMEDIATE_3_0_0_SW // 中間ファイルバージョン 3.0.0 を出力するなら定義します。

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// color
const int R_RGB_COUNT  = 3; //!< RGB の成分数です。
const int R_RGBA_COUNT = 4; //!< RGBA の成分数です。

const int R_RGB_BYTES  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int R_RGBA_BYTES = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int R_RGB_FLOAT16_BYTES  = 3 * sizeof(u16); //!< RGB 各 16 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT16_BYTES = 4 * sizeof(u16); //!< RGBA 各 16 ビット浮動小数点数のピクセルのバイト数です。

const int R_RGB_FLOAT_BYTES  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT_BYTES = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

//-----------------------------------------------------------------------------
// math
const double R_SAME_TOLERANCE   = 1.0e-6;  //!< double 型の値を比較する際の許容値です。
const float  R_SAME_TOLERANCE_F = 1.0e-6f; //!< float 型の値を比較する際の許容値です。

const double R_M_PI = 3.14159265358979323846; //!< 円周率です。
const double R_M_RAD_TO_DEG = 180.0 / R_M_PI; //!< ラジアンを度数に変換するためのスケールです。
const double R_M_DEG_TO_RAD = R_M_PI / 180.0; //!< 度数をラジアンに変換するためのスケールです。

const int R_XYZ_COUNT  = 3; //!< XYZ の成分数です。
const int R_XYZW_COUNT = 4; //!< XYZW の成分数です。

//-----------------------------------------------------------------------------
// output
#define R_ENDL "\r\n"

const int R_FLOAT_OUT_PRECISION = 6; //!< std::ostream::precision で指定する精度です。

const int R_BINARY_ALIGNMENT = 32; //!< バイナリ形式の中間ファイルのアライメントです。

//=============================================================================
//! @brief 未使用変数の警告を抑止するためのマクロです。
//=============================================================================
#define R_UNUSED_VARIABLE(Variable) (void)(&Variable);

//=============================================================================
//! @brief 配列型の定義です。
//=============================================================================

//! @brief char 型配列の定義です。
typedef std::vector<char> RCharArray;

//! @brief unsigned char 型配列の定義です。
typedef std::vector<unsigned char> RUCharArray;

//! @brief short 型配列の定義です。
typedef std::vector<short> RShortArray;

//! @brief unsigned short 型配列の定義です。
typedef std::vector<unsigned short> RUShortArray;

//! @brief int 型配列の定義です。
typedef std::vector<int> RIntArray;

//! @brief unsigned int 型配列の定義です。
typedef std::vector<unsigned int> RUIntArray;

//! @brief float 型配列の定義です。
typedef std::vector<float> RFloatArray;

//! @brief double 型配列の定義です。
typedef std::vector<double> RDoubleArray;

//! @brief 文字列型配列の定義です。
typedef std::vector<std::string> RStringArray;

//! @brief ユニコード文字列型配列の定義です。
typedef std::vector<std::wstring> RWStringArray;

//! @brief bool 型配列の定義です。
typedef std::vector<bool> RBoolArray;

//=============================================================================
// 一般的な数学関連の関数です。
//=============================================================================

//! @brief [0, 360) の範囲に正規化した角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [0, 360) の範囲に正規化した角度を返します。
//!
template <typename T>
inline T RNormalizeAngle(T ang)
{
    if (ang >= (T)360)
    {
        const int ishift = static_cast<int>(ang / (T)360);
        ang -= ishift * (T)360;
    }
    else if (ang < (T)0)
    {
        const int ishift = static_cast<int>(-ang / (T)360) + 1;
        ang += ishift * (T)360;
    }
    return ang;
}

//! @brief [-360, 360] の範囲にシフトした角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [-360, 360] の範囲にシフトした角度を返します。
//!
template <typename T>
inline T RShiftAngle(T ang)
{
    if (ang < (T)-360 || ang > (T)360)
    {
        int ishift = static_cast<int>(ang / (T)360);
        ang -= ishift * (T)360;
    }
    return ang;
}

inline bool RIsSame(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    const double c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSame(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    const float c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    double c = a - b;
    if (a != 0.0 && b != 0.0)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    float c = a - b;
    if (a != 0.0f && b != 0.0f)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameAngle(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const double na = RNormalizeAngle(a);
    const double nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

inline bool RIsSameAngle(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const float na = RNormalizeAngle(a);
    const float nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

//! @brief 絶対値を返します。
template <typename T>
inline T RAbs(const T x)
{
    return (x >= (T)0) ? x : -x;
}

//! @brief 四捨五入した値を返します。
template <typename T>
inline int RRound(const T x)
{
    return (x >= (T)0) ? static_cast<int>(x + (T)0.5) : static_cast<int>(x - (T)0.5);
}

//! @brief 小さいほうの値を返します。
template <typename T>
inline T RMin(const T a, const T b)
{
    return (a < b) ? a : b;
}

//! @brief 大きいほうの値を返します。
template <typename T>
inline T RMax(const T a, const T b)
{
    return (a > b) ? a : b;
}

//! @brief double 型の 0 に近い値を 0 にした値を返します。
inline double RSnapToZero(const double v)
{
    if (-R_SAME_TOLERANCE < v && v < R_SAME_TOLERANCE)
    {
        return 0.0;
    }
    else
    {
        return v;
    }
}

//! @brief float 型の 0 に近い値を 0 にした値を返します。
inline float RSnapToZero(const float v)
{
    if (-R_SAME_TOLERANCE_F < v && v < R_SAME_TOLERANCE_F)
    {
        return 0.0f;
    }
    else
    {
        return v;
    }
}

//! @brief 指定した範囲にクランプした値を返します。
template <typename T>
inline T RClampValue(const T valMin, const T valMax, const T val)
{
    if (val < valMin)
    {
        return valMin;
    }
    else if (val > valMax)
    {
        return valMax;
    }
    else
    {
        return val;
    }
}

//! @brief アライメントした値を返します。
template <typename T>
inline T RAlignValue(T val, const int align)
{
    return (T)((val + align - 1) & (~(align - 1)));
}

//! @brief アライメントしたポインタを返します。
template <typename T>
inline T* RAlignPointer(T* val, const int align)
{
    #ifdef _WIN64
    return (T*)(((unsigned __int64)val + align - 1) & (~(align - 1)));
    #else
    return (T*)(((unsigned __int32)val + align - 1) & (~(align - 1)));
    #endif
}

//! @brief 2 つの値を交換します。
template <typename T>
inline void RSwapValue(T& r1, T& r2)
{
    T tmpVal = r1;
    r1 = r2;
    r2 = tmpVal;
}

//! @brief 2 のべき乗の数なら true を返します。
//!
//! @param[in] number 判定する数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 2 のべき乗の数なら true を返します。
//!
bool RIsPower2Number(const int number, const bool includeOne = true);

//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//!
//! @param[in] number 数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 指定した数以上で最小の 2 のべき乗の数を返します。
//!
int RGetPower2Number(const int number, const bool includeOne = true);

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 文字が半角スペースかタブなら true を返します。
inline bool RIsSpace(const char c)
{
    return (c == ' ' || c == '\t');
}

bool RIsShiftJisFirstByte(const unsigned char val);
std::string RGetUtf8FromShiftJis(const std::string& src);
std::string RGetShiftJisFromUtf8(const std::string& src);
std::wstring RGetUnicodeFromShiftJis(const std::string& src);
std::string RGetShiftJisFromUnicode(const std::wstring& src);

//! @brief ユニコード C 配列を RStringArray に変換します。
//!
//! @param[out] dsts 変換先の文字列配列です。
//! @param[in] srcs ユニコード C 配列です。最後に NULL を格納します。
//!
//! @return 文字列数を返します。
//!
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[]);

std::string RGetLowerCaseString(const std::string& src);
std::string RGetUpperCaseString(const std::string& src);
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2);
std::string RTrimLeftString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimRightString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimString(const std::string& src, const char* targets = " \t\r\n");
std::string RDequoteString(const std::string& src, const char quote = '\"');

//! @brief 文字列をトークンに分解します。
//!
//! @param[out] tokens トークン配列を格納します。
//! @param[in] input 文字列です。
//! @param[in] delims 区切り記号です。
//!
//! @return トークン数を返します。
//!
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims = " \t\r\n");

//! @brief int 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%d" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const int num, const char* format = NULL);

//! @brief u32 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%u" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const u32 num, const char* format = NULL);

//! @brief ポインタを文字列に変換します。
//!
//! @param[in] ptr ポインタです。
//! @param[in] format フォーマット文字列です。NULL なら "%p" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetPointerString(const void* ptr, const char* format = NULL);

//! @brief Shift-JIS 文字列を XML 用にエンコードします。
std::string REncodeXmlString(const std::string& src);

//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
std::string RDecodeXmlString(const std::string& src);

//! @brief u8 型のデータ文字列をデコードします。
//!
//! @param[out] pDst データを格納するバッファの先頭アドレスです。
//! @param[in] str スペースで 16 進数値を区切ったデータ文字列です。
//! @param[in] count デコードする値の個数です。
//!
void RDecodeDataStringU8(void* pDst, const std::string& str, const u32 count);

//! @brief f32 型のデータ文字列をデコードします。
//!
//! @param[out] pDst データを格納するバッファの先頭アドレスです。
//! @param[in] str スペースで f32 値を区切ったデータ文字列です。
//! @param[in] count デコードする値の個数です。
//!
void RDecodeDataStringF32(void* pDst, const std::string& str, const u32 count);

//=============================================================================
//! @brief ユニコード C 配列のクラスです。
//=============================================================================
class RUnicodeCArray
{
  protected:
    RWStringArray m_WStrings; //!< ユニコード文字列配列です。
    const wchar_t** m_CArray; //!< ユニコード C 配列です。

  public:
    //! コンストラクタです（RStringArray から生成）。
    RUnicodeCArray(const RStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(RGetUnicodeFromShiftJis(srcs[iStr]));
        }
        CreateCArray();
    }

    //! コンストラクタです（RWStringArray から生成）。
    RUnicodeCArray(const RWStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(srcs[iStr]);
        }
        CreateCArray();
    }

    //! ユニコード C 配列を作成します。
    void CreateCArray(void)
    {
        const int strCount = static_cast<int>(m_WStrings.size());
        m_CArray = new const wchar_t*[strCount + 1];
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_CArray[iStr] = m_WStrings[iStr].c_str();
        }
        m_CArray[strCount] = NULL; // 最終要素を NULL にします。
    }

    //! デストラクタです。
    virtual ~RUnicodeCArray()
    {
        //std::cerr << "~RUnicodeCArray(): " << m_WStrings.size() << std::endl;
        delete[] m_CArray;
    }

    //! ユニコード C 配列を返します。
    const wchar_t** GetCArray(void) const { return m_CArray; }
};

//=============================================================================
// ファイルパス関連の関数です。
//=============================================================================
std::string RGetWindowsFilePath(const std::string& path);
std::string RGetUnixFilePath(const std::string& path);
std::string RGetFileNameFromFilePath(const std::string& path);
std::string RGetFolderFromFilePath(const std::string& path);
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower = true);
std::string RGetNoExtensionFilePath(const std::string& path);
bool RIsFullFilePath(const std::string& path);
std::string RGetFullFilePath(const std::string& path, const bool isUnix);

//=============================================================================
// ファイル関連の関数です。
//=============================================================================
bool RFileExists(const std::string& path);
bool RFolderExists(const std::string& path);

//=============================================================================
// ファイル出力関連の関数です。
//=============================================================================
void RInitOutStreamFormat(std::ostream& os);
void ROutUtf8Bom(std::ostream& os);
void ROutUtf16Bom(std::ostream& os);
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr);
const char* RTab(const int size = 1);
const char* RBoolStr(const bool value);

//! @brief パディングを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] size サイズ（バイト数）です。
//! @param[in] value パディングに使用する値です。
//!
void ROutPadding(std::ostream& os, const int size, const char value = '\0');

//! @brief C 言語のソース形式でバイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 値のインデントに必要なタブの数です。
//! @param[in] array バイト型のデータ値配列です。
//! @param[in] column 列数です。
//!
void ROutCSourceData(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
);

//=============================================================================
// 配列関連の関数です。
//=============================================================================

//! @brief std::vector 配列から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
template <typename T>
int RFindValueInArray(const std::vector<T>& array, const T& value)
{
    const int size = static_cast<int>(array.size());
    for (int index = 0; index < size; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//=============================================================================
// メモリ関連の関数です。
//=============================================================================

//! @brief オブジェクトへのポインタが NULL でなければ delete して NULL を代入します。
//!
//! @param[in,out] ptr オブジェクトへのポインタです。
//!
template <typename T>
inline void RFreeAndClearObject(T*& ptr)
{
    if (ptr != NULL)
    {
        delete ptr;
        ptr = NULL;
    }
}

//! @brief 配列へのポインタが NULL でなければ delete[] して NULL を代入します。
//!
//! @param[in,out] ptr 配列へのポインタです。
//!
template <typename T>
inline void RFreeAndClearArray(T*& ptr)
{
    if (ptr != NULL)
    {
        delete[] ptr;
        ptr = NULL;
    }
}

//! @brief メモリからリトルエンディアンの値を取得します。
template <typename T>
void RGetMemLittle(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 0
    // swap for big endian
    u8* p1 = reinterpret_cast<u8*>(&value);
    u8* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリからビッグエンディアンの値を取得します。
template <typename T>
void RGetMemBig(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 1
    // swap for little endian
    u8* p1 = reinterpret_cast<u8*>(&value);
    u8* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリからリトルエンディアンの short 値を取得します。
inline short RGetMemLittleShort(const void* pMemory)
{
    short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemLittleUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの int 値を取得します。
inline int RGetMemLittleInt(const void* pMemory)
{
    int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemLittleUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの short 値を取得します。
inline short RGetMemBigShort(const void* pMemory)
{
    short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemBigUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの int 値を取得します。
inline int RGetMemBigInt(const void* pMemory)
{
    int value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemBigUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemBig(value, pMemory);
    return value;
}

//=============================================================================
// 一般的な Windows 関連の関数です。
//=============================================================================

//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//!
std::string RGetEnvVar(const char* name);

//! @brief PC のユーザー名を返します。
std::string RGetUserName(void);

//! @brief PC のコンピュータ名を返します。
std::string RGetComputerName(void);

//! @brief SYSTEMTIME から time_t を取得して返します。
time_t RGetTimeT(const SYSTEMTIME& st);

//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
std::string RGetDateTimeString(void);

//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//!
//! @param[in] hModule モジュールのハンドルです
//!                    NULL なら現在のプロセスを作成するために使われたファイルのパスを取得します。
//!
//! @return 実行ファイルのフルパスを返します。
//!
std::string RGetModuleFileName(HMODULE hModule);

//! @brief ユーザーが使用できるアドレス空間の空き容量を返します。
size_t RGetAvailableVirtualMemorySize(void);

//=============================================================================
// 引数関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief 引数のトークンの値を取得します。
//!        name=value または name="value" から value を抽出します。
//!
//! @param[in] token 引数のトークンです。
//!
//! @return トークンの値を返します。
//-----------------------------------------------------------------------------
std::string RGetArgTokenValue(const std::string& token);

//-----------------------------------------------------------------------------
//! @brief 引数のトークンからファイルのフルパスを取得します。
//!
//! @param[in] token 引数のトークンです。
//!
//! @return ファイルのフルパスを返します。
//-----------------------------------------------------------------------------
inline std::string RGetArgTokenPath(const std::string& token)
{
    return RGetFullFilePath(RGetArgTokenValue(token), true);
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[out] pMessage エラーメッセージを格納します。NULL なら格納しません。
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool RGetArgTokensFromFile(
    std::string* pMessage,
    RStringArray& tokens,
    const std::string& filePath
);

//=============================================================================
//! @brief 処理結果ステータスのクラスです。
//=============================================================================
class RStatus
{
  public:
    //! 処理結果ステータスコードを表す列挙型です。
    enum RStatusCode
    {
        SUCCESS = 0, //!< 成功です。
        FAILURE      //!< 失敗です。
    };

  private:
    RStatusCode m_Code; //!< 処理結果ステータスコードです。
    std::string m_Message; //!< 失敗メッセージ文字列です。

  public:
    RStatus() : m_Code(SUCCESS) {}
    RStatus(const RStatusCode code) : m_Code(code) {}
    RStatus(const RStatusCode code, const std::string& message) :
        m_Code(code), m_Message(message) {}
    operator bool() const { return (m_Code == SUCCESS); }
    bool operator==(const RStatus& other) const { return (m_Code == other.m_Code); }
    bool operator!=(const RStatus& other) const { return (m_Code != other.m_Code); }
    bool operator==(const RStatusCode code) const { return (m_Code == code); }
    bool operator!=(const RStatusCode code) const { return (m_Code != code); }
    std::string GetMessage(void) const { return m_Message; }
};

//-----------------------------------------------------------------------------
//! @brief 処理結果ステータスが成功でなければ return するマクロです。
//-----------------------------------------------------------------------------
#define RCheckStatus(status)		\
{									\
    if (status != RStatus::SUCCESS)	\
        return status;				\
}

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class RTimeMeasure
{
  protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

  public:
    //! @brief コンストラクタです。
    RTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset(void)
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSec(void) const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSec(void) const
    {
        return GetSec() * 1000.0f;
    }
};

//=============================================================================
//! @brief XML 要素のクラスです。
//=============================================================================
class RXMLElement
{
  public:
    typedef std::map<std::string, std::string> AttrMap;
    typedef void (*ErrorFunc)(const char* message, void* pUserData);

    std::string name;
    AttrMap attributes;
    std::string text;
    std::vector<RXMLElement> nodes;

    ErrorFunc m_ErrorFunc; //!< エラー表示関数です。
    void* m_pErrorUserData; //!< エラー表示関数に渡すユーザーデータのポインタです。

  private:
    //! @brief 属性を解析してノードに記録します。
    //!
    //! @param[in] src 要素の文字列です。
    //! @param[in] end 要素の終端の文字です。
    //!
    //! @return 要素の終端までの文字数を返します。
    //!
    int LoadAttributes(const char* src, const char end);

    //! @brief エラーを表示します。
    //!
    //! @param[in] message エラーメッセージです。
    //!
    void DisplayError(const std::string& message) const;

  public:
    //! コンストラクタです。
    RXMLElement(ErrorFunc errorFunc = NULL, void* pErrorUserData = NULL)
    : m_ErrorFunc(errorFunc),
      m_pErrorUserData(pErrorUserData)
    {
    }

    //! コピーコンストラクタです。
    RXMLElement(const RXMLElement &r)
    {
        operator=(r);
    }

    //! 再初期化します。
    void Clear()
    {
        name = "";
        text = "";
        attributes.clear();
        nodes.clear();
    }

    //! 代入演算子です。
    RXMLElement& operator=(const RXMLElement& r);

    //! @brief 下層ノードを新規作成します。
    //!
    //! @return 新規作成した要素のポインタを返します。
    //!
    RXMLElement* CreateElement()
    {
        nodes.push_back(RXMLElement(m_ErrorFunc, m_pErrorUserData));
        return &nodes[nodes.size() - 1];
    }

    //! @brief 指定のパスに該当する最初の要素のポインタを取得します。
    //!
    //! @param[in] path パスです。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 要素のポインタを返します。見つからなかった場合は NULL を返します。
    //!
    const RXMLElement* FindElement(const std::string& path, const bool showError = true) const
    {
        std::vector<const RXMLElement*> list;
        FindElements(list, path, showError, 1);
        return (list.size() > 0) ? list[0] : NULL;
    }

    //! @brief 指定のパスに該当する要素を指定個数分検索します。
    //!
    //! @param[out] list 要素のポインタ配列です。
    //! @param[in] path パスです。
    //! @param[in] maxCount 取得する最大要素数です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    void FindElements(
        std::vector<const RXMLElement*>& list,
        const std::string& path,
        const bool showError = true,
        const int maxCount = 0x7fffffff
    ) const;

    //! @brief 属性の値を取得します。
    //!
    //! @param[in] tag 属性名です。
    //! @param[in] alt 属性が見つからなかった場合に返す値です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 属性の値を返します。
    //!
    std::string GetAttribute(
        const std::string& tag,
        const char* alt = "",
        const bool showError = true
    ) const
    {
        AttrMap::const_iterator it = attributes.find(tag);
        if (showError && it == attributes.end())
        {
            DisplayError("Attribute is not found: " + tag + " in <" + name + ">");
        }
        return (it != attributes.end()) ? it->second : alt;
    }

    //! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
    //!
    //! @param[in,out] decl 宣言要素です。宣言要素の属性を取得しないなら NULL を指定します。
    //! @param[in] xmlDoc XML 文書です。
    //!
    void LoadDocument(RXMLElement* decl, const std::string& xmlDoc);

    //! @brief XML 文書を取得します。
    //!        要素、属性、平データ、下層ノードの内容を再帰的に文字列に出力します。
    //!
    //! @param[in] tabCount 要素のインデントに必要なタブの数です。
    //! @param[in] lf 改行コードです。
    //!
    //! @return XML 文書の文字列を返します。
    //!
    std::string GetDocument(const int tabCount = 0, const char* lf = "\n") const;

    //! @brief エラー表示関数を設定します。
    //!
    //! @param[in] func エラー表示関数です。
    //! @param[in] pUserData エラー表示関数に渡すユーザーデータのポインタです。
    //!
    void SetErrorFunc(ErrorFunc func, void* pUserData)
    {
        m_ErrorFunc = func;
        m_pErrorUserData = pUserData;
    }
};

//=============================================================================
// NW4F 関連の関数です。
//=============================================================================

//! @brief 環境変数 NW4F_G3D_ROOT または NW4F_ROOT の値を返します。
//!        どちらも定義されていなければ空文字を返します。
//!
//! @return 環境変数 NW4F_G3D_ROOT または NW4F_ROOT の値を返します。
//!
inline std::string RGetNw4fRoot(void)
{
    const std::string root = RGetEnvVar("NW4F_G3D_ROOT");
    return (!root.empty()) ? root : RGetEnvVar("NW4F_ROOT");
}

//! @brief NW4F G3D コマンドラインツールフォルダのパスを返します。
//!
//! @return NW4F G3D コマンドラインツールフォルダのパスを返します。
//!
inline std::string RGetNw4fG3dToolFolder(void)
{
    #ifdef _WIN64
    return RGetNw4fRoot() + "\\Tool\\G3dTool\\win64";
    #else
    return RGetNw4fRoot() + "\\Tool\\G3dTool\\win32";
    #endif
}

//! @brief NW4F G3D ツールの exe および dll ファイルのフルパスを返します。
//!        指定したモジュールの exe (dll) ファイルと同じフォルダに exe (dll) ファイルがあればそのパス、
//!        なければ G3D コマンドラインツールフォルダ下の exe (dll) ファイルのパスを返します。
//!
//! @param[in] fileName exe (dll) ファイル名です。
//! @param[in] hModule モジュールのハンドルです
//!                    NULL なら現在のプロセスを作成するために使われた exe ファイルのフォルダを参照します。
//!
//! @return exe (dll) ファイルのフルパスを返します。
//!
std::string RGetNw4fG3dExeDllFullPath(const std::string& fileName, HMODULE hModule);

//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//!        1.0.2 -> 100
//!        1.3.4 -> 103
//!
//! @return 中間ファイルバージョンを整数で返します。
//!
int RGetNw4f3difVerAsInt(const std::string& value);

//=============================================================================
// @brief データ列のクラスです。
//=============================================================================
class RDataStream
{
  public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    //! バイナリ形式のヘッダサイズです。
    enum { HEADER_SIZE = 0x20 };

    Type m_Type; //!< 値の型です。
    int m_Column; //!< アスキー出力時の列数です。
    RFloatArray m_FloatValues; //!< 浮動小数点数値の配列です。
    RIntArray m_IntValues; //!< 整数値の配列です。
    RUCharArray m_ByteValues; //!< バイト値の配列です。
    std::string m_StringValue; //!< 文字列値（Shift-JIS）です。

  public:
    //! @brief 浮動小数点数値用のコンストラクタです。
    //!
    //! @param[in] array 浮動小数点数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RFloatArray& array, const int column = 4)
    : m_Type(FLOAT),
      m_Column(column)
    {
        m_FloatValues = array;
    }

    //! @brief 整数値用のコンストラクタです。
    //!
    //! @param[in] array 整数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RIntArray& array, const int column = 4)
    : m_Type(INT),
      m_Column(column)
    {
        m_IntValues = array;
    }

    //! @brief バイト値用のコンストラクタです。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const u8* array, const int bytes, const int column = 4);

    //! @brief 文字列値のコンストラクタです。
    //!
    //! @param[in] str 文字列値（Shift-JIS）です。
    //! @param[in] isWstring ASCII 文字列として出力するなら false、
    //!                      Unicode 文字列として出力するなら true を指定します。
    //!
    RDataStream(const std::string& str, const bool isWstring = false)
    : m_Column(0)
    {
        m_Type = (isWstring) ? WSTRING : STRING;
        m_StringValue = str;
    }

    //! @brief Unicode 文字列値のコンストラクタです。
    //!
    //! @param[in] wstr Unicode 文字列値です。
    //!
    RDataStream(const std::wstring& wstr)
    : m_Type(WSTRING),
      m_Column(0)
    {
        m_StringValue = RGetShiftJisFromUnicode(wstr);
    }

    //! バイト値を追加します。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //!
    void Append(const u8* array, const int bytes);

    //! バイト値のパディングを追加します。
    //!
    //! @param[in] size パディングのバイト数です。
    //! @param[in] value パディングに使用する値です。
    //!
    void AppendPadding(const int bytes, const char value = '\0')
    {
        if (bytes > 0)
        {
            u8* buf = new u8[bytes];
            memset(buf, value, bytes);
            Append(buf, bytes);
            delete[] buf;
        }
    }

    //! 値の個数を取得します。
    const int GetCount(void) const
    {
        if      (m_Type == FLOAT) return static_cast<int>(m_FloatValues.size());
        else if (m_Type == INT  ) return static_cast<int>(m_IntValues.size()  );
        else if (m_Type == BYTE ) return static_cast<int>(m_ByteValues.size() );
        else                      return 1; // STRING, WSTRING
    }

    //! バイナリ形式の実データサイズ（バイト数）を取得します。
    const int GetSize(void) const
    {
        if      (m_Type == FLOAT ) return static_cast<int>(m_FloatValues.size()) * sizeof(float);
        else if (m_Type == INT   ) return static_cast<int>(m_IntValues.size()  ) * sizeof(int);
        else if (m_Type == BYTE  ) return static_cast<int>(m_ByteValues.size() );
        else if (m_Type == STRING) return static_cast<int>(m_StringValue.size()) + 1; // 終端文字を含むので +1 します。
        else // WSTRING
        {
            // 終端文字を含むので +1 します。
            return (static_cast<int>(RGetUnicodeFromShiftJis(m_StringValue).size()) + 1) * sizeof(wchar_t);
        }
    }

    //! ヘッダを含むバイナリ形式のアライメントしたデータサイズ（バイト数）を取得します。
    const int GetBinarySize(void) const
    {
        return RAlignValue(HEADER_SIZE + GetSize(), R_BINARY_ALIGNMENT);
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <stream> 要素のインデントに必要なタブの数です。
    //! @param[in] index <stream_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! @brief バイナリ形式で出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //!
    void OutBinary(std::ostream& os) const;
};

//! @brief データ列配列の定義です。
typedef std::vector<RDataStream> RDataStreamArray;

//! @brief データ列配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//!
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array);

//! @brief データ列配列をバイナリ形式で出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//! @param[in] outZeroPadding バイナリ形式の後半部の前のゼロパディングを出力するなら true を指定します。
//!
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
);

//! @brief 中間ファイルの先頭からバイナリデータまでのオフセットを取得します。
//!
//! @param[in] fileBuf 中間ファイルをリードしたメモリのポインタです。
//! @param[in] fileSize 中間ファイルのサイズ（バイト数）です。
//!
//! @return バイナリデータまでのオフセットを返します。
//!         バイナリデータがない場合は fileSize を返します。
//!
int RGetBinaryOffset(const void* fileBuf, const int fileSize);

//=============================================================================
// @brief バイナリデータ列のクラスです。
//=============================================================================
class RBinDataStream
{
  public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    char m_Identifier[8]; //!< 識別子です。
    int m_Type; //!< 値の型です。
    int m_Count; //!< 値の個数です。
    int m_Column; //!< アスキー出力時の列数です。
    int m_Size; //!< データサイズ（バイト数）です。文字列の場合は終端文字を含んだバイト数です。
    int m_Padding[2]; //!< 実データの先頭アドレスを 32 バイトアライメントにするためのパディングです。
    int m_Data[1]; //!< 実データです。実際は可変長です。

  public:
    //! @brief バイナリデータ列へのポインタを取得します（static 関数）。
    //!
    //! @param[in] binTop 中間ファイルのバイナリ部分の先頭アドレスです。
    //! @param[in] streamIdx バイナリデータ列のインデックスです。
    //!
    //! @return バイナリデータ列へのポインタを返します。
    //!         インデックスが不正な場合は NULL を返します。
    //!
    static const RBinDataStream* Get(const void* binTop, const int streamIdx);
};

//=============================================================================
// @brief ユーザーデータのクラスです。
//=============================================================================
class RUserData
{
  public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, STRING, WSTRING, STREAM };

    std::string m_Name; //!< 名前です。
    Type m_Type; //!< 値の型です。
    int m_NumberValueCount; //!< 浮動小数点数値または整数値の値の数です。
    std::string m_NumberValuesText; //!< 浮動小数点数値または整数値の値の配列の中間ファイル上の文字列です。
    RStringArray m_StringValues; //!< 文字列値（Shift-JIS）の配列です。
    RUCharArray m_StreamValues; //!< ストリーム型値の配列です。

  public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] userDataElem <user_data> 要素です。
    //! @param[in] streamsElem <stream_array> 要素です（バイナリ形式なら NULL を指定します）。
    //! @param[in] binTop 中間ファイルのバイナリ部分の先頭アドレスです（アスキー形式なら NULL を指定します）。
    //!
    RUserData(
        const RXMLElement* userDataElem,
        const RXMLElement* streamsElem,
        const void* binTop
    );

    //! @brief コンストラクタです。
    //!
    //! @param[in] name 名前です。
    //! @param[in] type 値の型です。
    //!
    RUserData(const std::string& name, const Type type)
    : m_Name(name),
      m_Type(type),
      m_NumberValueCount(0)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <user_data> 要素のインデントに必要なタブの数です。
    //! @param[in] index <user_data_array> 内でのインデックスです。
    //!
    void Out(
        std::ostream& os,
        RDataStreamArray& dataStreams,
        const int tabCount,
        const int index
    ) const;
};

//! @brief ユーザーデータ配列の定義です。
typedef std::vector<RUserData> RUserDataArray;

//-----------------------------------------------------------------------------
//! @brief 配列形式の要素を出力します。テンプレート関数です。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 要素のインデントに必要なタブの数です。
//! @param[in] array 配列です。
//! @param[in] name 配列の要素名です。
//-----------------------------------------------------------------------------
template <typename T>
void ROutArrayElement(
    std::ostream& os,
    const int tabCount,
    const std::vector<T>& array,
    const char* name
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount > 0)
    {
        const int& tc = tabCount;
        os << RTab(tc) << "<" << name << " length=\"" << valCount << "\">" << R_ENDL;
        for (int ival = 0; ival < valCount; ++ival)
        {
            array[ival].Out(os, tc + 1, ival);
        }
        os << RTab(tc) << "</" << name << ">" << R_ENDL;
    }
}

//=============================================================================
//! @brief ファイルバッファのクラスです。
//=============================================================================
class RFileBuf
{
  protected:
    std::string m_FilePath; //!< ファイルのパスです。
    u8* m_FileBuf; //!< ファイルバッファです。
    int m_FileSize; //!< ファイルサイズです。

  public:
    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    RFileBuf(const std::string& path)
    : m_FileBuf(NULL)
    {
        Read(path);
    }

    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    RFileBuf(const std::wstring& path)
    : m_FileBuf(NULL)
    {
        Read(RGetShiftJisFromUnicode(path));
    }

    //! デストラクタです。
    virtual ~RFileBuf()
    {
        //std::cerr << "~RFileBuf(): " << m_FilePath << std::endl;
        RFreeAndClearArray(m_FileBuf);
    }

    //! @biref ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    //! @return リード成功なら true を返します。
    //!
    bool Read(const std::string& path);

    //! リード成功なら true を返します。
    operator bool() const { return (m_FileBuf != NULL); }

    //! ファイルのパスを返します。
    std::string GetPath(void) const { return m_FilePath; };

    //! ファイルバッファを返します。
    const u8* GetBuf(void) const { return m_FileBuf; };

    //! ファイルバッファを返します（非 const 版）。
    u8* GetBuf(void) { return m_FileBuf; };

    //! ファイルサイズを返します。
    int GetSize(void) const { return m_FileSize; };

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    const u8& operator[](int i) const { return m_FileBuf[i]; }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です（非 const 版）。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    u8& operator[](int i) { return m_FileBuf[i]; }

  private:
    RFileBuf(const RFileBuf& other);
    RFileBuf& operator=(const RFileBuf& other);
};

//=============================================================================
// テクスチャ関連の関数です。
//=============================================================================

//! @brief 中間ファイルの文字列からサーフェスの次元を取得します。
//!
//! @param[out] pIsArray 配列テクスチャなら true を格納します。NULL なら格納しません。
//! @param[in] str サーフェスの次元を表す文字列です。
//!
//! @return サーフェスの次元を返します。
//!
GX2SurfaceDim RGetSurfaceDimensionFromString(bool* pIsArray, const std::string& str);

//! @brief サーフェスの次元から中間ファイルの文字列を取得します。
//!
//! @param[in] dimension サーフェスの次元です。
//! @param[in] isArray 配列テクスチャなら true を指定します。キューブの場合のみ影響します。
//!
//! @return サーフェスの次元を表す文字列を返します。
//!
std::string RGetSurfaceDimensionString(const GX2SurfaceDim dimension, const bool isArray);

//! @brief 中間ファイルの文字列からサーフェスのフォーマットを取得します。
//!
//! @param[in] str サーフェスのフォーマットを表す文字列です。
//!
//! @return サーフェスのフォーマットを返します。
//!
GX2SurfaceFormat RGetSurfaceFormatFromString(const std::string& str);

//! @brief サーフェスのフォーマットから中間ファイルの文字列を取得します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットを表す文字列を返します。
//!
std::string RGetSurfaceFormatString(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
bool RIsBC123SurfaceFormat(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットが BC フォーマットなら true を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットが BC フォーマットなら true を返します。
//!
bool RIsBCSurfaceFormat(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットが浮動小数点数型なら true を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットが浮動小数点数型なら true を返します。
//!
bool RIsFloatSurfaceFormat(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットが符号ありなら true を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットが符号ありなら true を返します。
//!
bool RIsSignedSurfaceFormat(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットが sRGB フェッチなら true を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return サーフェスのフォーマットが sRGB フェッチなら true を返します。
//!
bool RIsSrgbFetchSurfaceFormat(const GX2SurfaceFormat format);

//! @brief 中間ファイルの文字列からタイルモードを取得します。
//!
//! @param[in] str タイルモードを表す文字列です。
//!
//! @return タイルモードを返します。
//!
GX2TileMode RGetTileModeFromString(const std::string& str);

//! @brief タイルモードから中間ファイルの文字列を取得します。
//!
//! @param[in] tileMode タイルモードです。
//!
//! @return タイルモードを表す文字列を返します。
//!
std::string RGetTileModeString(const GX2TileMode tileMode);

//! @brief 中間ファイルの文字列から成分選択を取得します。
//!
//! @param[in] str 成分選択を表す文字列です。
//!
//! @return 成分選択を返します。
//!
GX2CompSel RGetCompSelFromString(const std::string& str);

//! @brief 成分選択から中間ファイルの文字列を取得します。
//!
//! @param[in] compSel 成分選択です。
//!
//! @return 成分選択を表す文字列を返します。
//!
std::string RGetCompSelString(const GX2CompSel compSel);

//! @brief サーフェスのフォーマットに対応したデフォルトの成分選択を取得します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return 成分選択を返します。
//!
GX2CompSel RGetDefaultCompSel(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットから成分数を取得します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return 成分数を返します。
//!
int RGetCompSize(const GX2SurfaceFormat format);

//! @brief サーフェスのフォーマットの 1 ピクセルのビット数を返します。
//!
//! @param[in] format サーフェスのフォーマットです。
//!
//! @return 1 ピクセルのビット数を返します。
//!
int RGetBitPerPixel(const GX2SurfaceFormat format);

//=============================================================================
// nwFtx ネームスペースを終了します。
//=============================================================================
} // namespace nwFtx

#endif // Common_H_
