﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>
#include <msclr/marshal_cppstd.h>

#include <process.h>

#include "ShaderConverterLogHandler.h"

using namespace std;
using namespace msclr::interop;

using namespace System;
using namespace System::Text;

namespace EffectMaker {
namespace SpecGeneric {
namespace Shader {

//---------------------------------------------------------------------------
//  関数
//---------------------------------------------------------------------------

// ハンドルに出力されたデータを吸い出す
static int DrainHandleData(HANDLE handle, stringstream* pStream);

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
ShaderConverterLogHandler::ShaderConverterLogHandler()
    : m_pShaderConverterDll(), m_drainThread(), m_disconnectEvent(), m_stdOutputHandle(), m_stdErrorHandle(), m_defaultPipe(), m_warningPipe(), m_errorPipe()
{
}

//---------------------------------------------------------------------------
//  デストラクタです。
//---------------------------------------------------------------------------
ShaderConverterLogHandler::~ShaderConverterLogHandler()
{
    assert(m_pShaderConverterDll == nullptr);

    assert(m_drainThread == NULL);
    assert(m_disconnectEvent == NULL);

    assert(m_defaultPipe.readHandle == NULL);
    assert(m_defaultPipe.writeHandle == NULL);

    assert(m_warningPipe.readHandle == NULL);
    assert(m_warningPipe.writeHandle == NULL);

    assert(m_errorPipe.readHandle == NULL);
    assert(m_errorPipe.writeHandle == NULL);
}

//---------------------------------------------------------------------------
//  パイプを初期化します。
//---------------------------------------------------------------------------
void ShaderConverterLogHandler::Initialize(nn::gfxTool::ShaderConverterDll* pShaderCovnerterDll)
{
    m_pShaderConverterDll = pShaderCovnerterDll;

    // パイプの切断イベントを作成
    m_disconnectEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
    assert(m_disconnectEvent != NULL);

    // デフォルトの出力ハンドルを取得
    m_stdOutputHandle = GetStdHandle(STD_OUTPUT_HANDLE);
    m_stdErrorHandle = GetStdHandle(STD_OUTPUT_HANDLE);

    BOOL resCreate;

    // 標準ログ用のパイプを作成
    resCreate = CreatePipe(&m_defaultPipe.readHandle, &m_defaultPipe.writeHandle, NULL, 0);
    assert(resCreate != 0);

    // 警告ログ用のパイプを作成
    resCreate = CreatePipe(&m_warningPipe.readHandle, &m_warningPipe.writeHandle, NULL, 0);
    assert(resCreate != 0);

    // エラーログ用のパイプを作成
    resCreate = CreatePipe(&m_errorPipe.readHandle, &m_errorPipe.writeHandle, NULL, 0);
    assert(resCreate != 0);
}

//---------------------------------------------------------------------------
//  パイプを破棄します。
//---------------------------------------------------------------------------
void ShaderConverterLogHandler::Release()
{
    // 標準ログ用のパイプを破棄
    CloseHandle(m_defaultPipe.readHandle);
    CloseHandle(m_defaultPipe.writeHandle);

    m_defaultPipe = PipeInfo();

    // 警告ログ用のパイプを破棄
    CloseHandle(m_warningPipe.readHandle);
    CloseHandle(m_warningPipe.writeHandle);

    m_warningPipe = PipeInfo();

    // エラーログ用のパイプを破棄
    CloseHandle(m_errorPipe.readHandle);
    CloseHandle(m_errorPipe.writeHandle);

    m_errorPipe = PipeInfo();

    m_stdOutputHandle = NULL;
    m_stdErrorHandle = NULL;

    CloseHandle(m_disconnectEvent);
    m_disconnectEvent = NULL;

    m_pShaderConverterDll = nullptr;
}

//---------------------------------------------------------------------------
//  シェーダコンバータのログ出力先を自作の出力ハンドルに設定します。
//---------------------------------------------------------------------------
void ShaderConverterLogHandler::ConnectPipe()
{
    // 標準ログの出力先を自作のパイプに設定
    {
        nngfxToolSetLogStreamArg arg = {};

        arg.logType    = nngfxToolLogType_Default;
        arg.hLogStream = m_defaultPipe.writeHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }

    // 警告ログの出力先を自作のパイプに設定
    {
        nngfxToolSetLogStreamArg arg = {};

        arg.logType    = nngfxToolLogType_Warning;
        arg.hLogStream = m_warningPipe.writeHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }

    // エラーログの出力先を自作のパイプに設定
    {
        nngfxToolSetLogStreamArg arg = {};

        arg.logType    = nngfxToolLogType_Error;
        arg.hLogStream = m_errorPipe.writeHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }

    // パイプが詰まってスレッドをロックするのを防ぐため、別スレッドにてデータ吸い出し処理を回す
    m_drainThread = (HANDLE)_beginthreadex(NULL, 0, ChooChooDrain, this, 0, NULL);
}

//---------------------------------------------------------------------------
//!  シェーダコンバータのログ出力先をデフォルトの出力ハンドルに戻します。
//---------------------------------------------------------------------------
void ShaderConverterLogHandler::DisconnectPipe()
{
    // 吸い出し処理を停止
    SetEvent(m_disconnectEvent);
    WaitForSingleObject(m_drainThread, INFINITE);

    m_drainThread = NULL;

    // 標準ログの出力先をデフォルトの出力ハンドルに戻す
    {
        nngfxToolSetLogStreamArg arg = { };

        arg.logType    = nngfxToolLogType_Default;
        arg.hLogStream = m_stdOutputHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }

    // 警告ログの出力先をデフォルトの出力ハンドルに戻す
    {
        nngfxToolSetLogStreamArg arg = { };

        arg.logType    = nngfxToolLogType_Warning;
        arg.hLogStream = m_stdOutputHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }

    // エラーログの出力先をデフォルトの出力ハンドルに戻す
    {
        nngfxToolSetLogStreamArg arg = { };

        arg.logType    = nngfxToolLogType_Error;
        arg.hLogStream = m_stdErrorHandle;

        m_pShaderConverterDll->SetLogStream(&arg);
    }
}

//---------------------------------------------------------------------------
//  標準ログとして出力されたデータを取得します。
//---------------------------------------------------------------------------
System::String^ ShaderConverterLogHandler::ReadDefaultLogData()
{
    string text = m_defaultPipe.drainStream.str();

    cli::array<Byte>^ byteArray = gcnew cli::array<Byte>(text.length());

    for (size_t i = 0; i < text.length(); ++i)
    {
        byteArray[i] = text[i];
    }

    System::Text::UTF8Encoding^ encoding = gcnew System::Text::UTF8Encoding();
    String^ result = encoding->GetString(byteArray);

    result = result->Replace("\r\n", "\n");
    result = result->Replace("\r", "\n");

    return result;
}

//---------------------------------------------------------------------------
//  警告ログとして出力されたデータを取得します。
//---------------------------------------------------------------------------
System::String^ ShaderConverterLogHandler::ReadWarningLogData()
{
    string text = m_warningPipe.drainStream.str();

    cli::array<Byte>^ byteArray = gcnew cli::array<Byte>(text.length());

    for (size_t i = 0; i < text.length(); ++i)
    {
        byteArray[i] = text[i];
    }

    System::Text::UTF8Encoding^ encoding = gcnew System::Text::UTF8Encoding();
    String^ result = encoding->GetString(byteArray);

    result = result->Replace("\r\n", "\n");
    result = result->Replace("\r", "\n");

    return result;
}

//---------------------------------------------------------------------------
//  エラーログとして出力されたデータを取得します。
//---------------------------------------------------------------------------
System::String^ ShaderConverterLogHandler::ReadErrorLogData()
{
    string text = m_errorPipe.drainStream.str();

    cli::array<Byte>^ byteArray = gcnew cli::array<Byte>(text.length());

    for (size_t i = 0; i < text.length(); ++i)
    {
        byteArray[i] = text[i];
    }

    System::Text::UTF8Encoding^ encoding = gcnew System::Text::UTF8Encoding();
    String^ result = encoding->GetString(byteArray);

    result = result->Replace("\r\n", "\n");
    result = result->Replace("\r", "\n");

    return result;
}

//---------------------------------------------------------------------------
//  パイプに出力されたデータを定期的に吸い出します。
//---------------------------------------------------------------------------
unsigned _stdcall ShaderConverterLogHandler::ChooChooDrain(void* pArg)
{
    ShaderConverterLogHandler* self = static_cast<ShaderConverterLogHandler*>(pArg);

    bool disconnect = false;
    bool noWait = true;

    while (disconnect == false)
    {
        DWORD resWait = WaitForSingleObject(self->m_disconnectEvent, noWait ? 0 : 30);

        if (resWait == WAIT_OBJECT_0)
        {
            ResetEvent(self->m_disconnectEvent);
            disconnect = true;
        }

        noWait = false;

        int drainedSize;

        drainedSize = DrainHandleData(self->m_defaultPipe.readHandle, &self->m_defaultPipe.drainStream);
        noWait |= (drainedSize > 0);

        drainedSize = DrainHandleData(self->m_warningPipe.readHandle, &self->m_warningPipe.drainStream);
        noWait |= (drainedSize > 0);

        drainedSize = DrainHandleData(self->m_errorPipe.readHandle, &self->m_errorPipe.drainStream);
        noWait |= (drainedSize > 0);
    }

    _endthreadex(0);

    return 0;
}

//---------------------------------------------------------------------------
//! @brief      ハンドルに出力されたデータを吸い出します。
//! @param[in]  handle   入力ハンドル
//! @param[in]  pStream  吸い出し先のストリーム
//!
//! @return     吸い出したデータのサイズを返します。
//---------------------------------------------------------------------------
static int DrainHandleData(HANDLE handle, stringstream* pStream)
{
    DWORD availDataSize;

    // パイプに出力されたデータのサイズを取得する
    BOOL resPeek = PeekNamedPipe(handle, NULL, 0, NULL, &availDataSize, NULL);

    if (resPeek == 0 || availDataSize == 0) {
        return 0;
    }

    DWORD readDataSize;
    char* pBuffer = new char[availDataSize + 1];

    // パイプに出力されたデータを取得
    BOOL resRead = ReadFile(handle, pBuffer, availDataSize, &readDataSize, NULL);
    pBuffer[readDataSize] = '\0';

    pStream->write(pBuffer, readDataSize);

    delete[] pBuffer;

    return readDataSize;
}

}
}
}
