﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderConversionInputData_H__
#define __ShaderConversionInputData_H__


using namespace System;
using namespace System::Collections::Generic;

using namespace EffectMaker::ShaderConverterLib;


namespace EffectMaker {
namespace ShaderConverterWrapper {

/// <summary>
/// Data class that holds the input data needed for shader conversion.
/// The class can also convert the input data to the native C++ data structure
/// the shader converter library uses.
/// </summary>
public ref class ShaderConversionInputData
{
public :
    /// <summary>
    /// Constructor.
    /// </summary>
    ShaderConversionInputData()
    {
        this->BinaryHeader           = nullptr;
        this->EmitterBinary          = nullptr;
        this->OverrideShaderCode     = String::Empty;
        this->ShaderIndexOffset      = 0;
        this->VertexShaderIndex      = 0;
        this->PixelShaderIndex       = 0;
        this->UserVertexShaderIndex1 = 0;
        this->UserPixelShaderIndex1  = 0;
        this->UserVertexShaderIndex2 = 0;
        this->UserPixelShaderIndex2  = 0;
        this->ReservedShaderIndex    = -1;
    };

    /// <summary>
    /// Get or set the binary tree node data.
    /// </summary>
    property array<System::Byte>^ BinaryHeader;

    /// <summary>
    /// Get or set the emitter binary data.
    /// </summary>
    property array<System::Byte>^ EmitterBinary;

    /// <summary>
    /// Get or set the shader code that should override the regular ones.
    /// </summary>
    property String^ OverrideShaderCode;

    /// <summary>
    /// Get or set the offset in the emitter binary of the shader index.
    /// </summary>
    property System::Int64 ShaderIndexOffset;

    /// <summary>
    /// Get or set the vertex shader index.
    /// </summary>
    property System::UInt32 VertexShaderIndex;

    /// <summary>
    /// Get or set the pixel shader index.
    /// </summary>
    property System::UInt32 PixelShaderIndex;

    /// <summary>
    /// Get or set the user vertex shader index 1.
    /// </summary>
    property System::UInt32 UserVertexShaderIndex1;

    /// <summary>
    /// Get or set the user pixel shader index 1.
    /// </summary>
    property System::UInt32 UserPixelShaderIndex1;

    /// <summary>
    /// Get or set the user vertex shader index 2.
    /// </summary>
    property System::UInt32 UserVertexShaderIndex2;

    /// <summary>
    /// Get or set the user pixel shader index 2.
    /// </summary>
    property System::UInt32 UserPixelShaderIndex2;

    /// <summary>
    /// Get or set the reserved shader ID.
    /// </summary>
    property System::Int32 ReservedShaderIndex;

    /// <summary>
    /// ユーザー定義文字列の何番目を利用するかを設定
    /// </summary>
    property System::UInt32 UserDefineIndex;

    /// <summary>
    /// Get the size of the input data.
    /// </summary>
    property size_t Size
    {
        size_t get()
        {
            return this->BinaryHeader->Length +
                this->EmitterBinary->Length +
                System::Text::Encoding::GetEncoding("Shift_JIS")->GetByteCount(this->OverrideShaderCode) + 1;
        }
    }

    /// <summary>
    /// Get the offset from the beginning of the emitter binary data
    /// to the override shader code.
    /// </summary>
    property size_t OverrideShaderOffset
    {
        size_t get()
        {
            return this->EmitterBinary->Length;
        }
    }
};

} // namespace ShaderConverterWrapper
} // namespace EffectMaker


#endif // __ShaderConversionInputData_H__
