﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifndef __GFDFILEIO_H__
#define __GFDFILEIO_H__

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

/// @addtogroup GFDFileIOGroup
/// @{

/// \brief Round up value to power-of-2 size
#define GFD_ROUND_UP(value, size) (((value) + ((size) - 1)) & ~((size) - 1))

//
// Write Functions
//

/// \brief Writes a GFD header to the file.
///
/// \param fp                   Dest Output file pointer.
/// \param gpuVer               Gpu version.
/// \retval Returned true or false
///
GFD_DECLSPEC char * GFD_API GFDWriteMemHeader(char *fp, GFDGPUVersion gpuVer);

/// \brief Writes a block header to the file.
///
/// \param fp                   Dest Output file pointer.
/// \param type                 Type of Block
/// \param dataSize             Data size in bytes following this header.
/// \retval Returned true or false
///
GFD_DECLSPEC char * GFD_API GFDWriteMemBlockHeader(char *fp, GFDBlockType type, u32 dataSize);

/// \brief Writes a GPU data to the file.
///
/// \param fp                   Dest Output file pointer.
/// \param numElements          Number of input elements
/// \param elemSize             Element size
/// \param swapMode             Endian Swap mode for data writing
/// \param pData                Input Data pointer.
/// \retval Returned true or false
///
GFD_DECLSPEC char * GFD_API GFDWriteMemGPUData(char *fp, u32 numElements, GFDElementSize elemSize, GFDEndianSwapMode swapMode, void *pData);

/// \brief Writes a PPC data to the file.
///
/// \param fp                   Dest Output file pointer
/// \param numElements          Number of input elements
/// \param elemSize             Element size
/// \param pData                Input Data pointer.
/// \retval Returned true or false
///
GFD_DECLSPEC inline char * GFD_API GFDWriteMemPPCData(char *fp, u32 numElements, GFDElementSize elemSize, void *pData)
{
    return GFDWriteMemGPUData(fp, numElements, elemSize, GFD_ENDIAN_SWAP_MODE_BIG, pData);
}

/// \brief Writes a padding block header and padding data block with 0 to the file.
///
/// \param fp                   Dest Output file pointer.
/// \param padSize              Padding size in bytes for data block
/// \retval Returned true or false
///
GFD_DECLSPEC char * GFD_API GFDWriteMemPadBlock(char *fp, u32 padSize);

/// \brief Reads a GPU data from the file. Need to use GFDOpenFile for the file pointer.
///
/// \param pData                Output buffer pointer.
/// \param numElements          Number of output elements
/// \param elemSize             Element size
/// \param swapMode             Endian Swap mode for data reading
/// \param fp                   Input file pointer.
/// \retval Returned true or false
///
GFD_DECLSPEC bool GFD_API GFDReadMemGPUData(void *pData, u32 numElements, GFDElementSize elemSize, GFDEndianSwapMode swapMode, char *fp);

/// \brief Reads a CPU data from the file.
///
/// \param pData                Output buffer pointer.
/// \param numElements          Number of output elements
/// \param elemSize             Element size
/// \param fp                   Input file pointer.
/// \retval Returned true or false
///
GFD_DECLSPEC inline bool GFD_API GFDReadMemPPCData(void *pData, u32 numElements, GFDElementSize elemSize, char *fp)
{
    return GFDReadMemGPUData(pData, numElements, elemSize, GFD_ENDIAN_SWAP_MODE_BIG, fp);
}

/// \brief Checks File Header magic and version values.
///
/// \param pHeader              Header pointer to check
/// \retval Returned true or false
///
GFD_DECLSPEC inline bool GFD_API GFDCheckHeaderMagicVersions(const GFDHeader *pHeader)
{
    u32 magic = pHeader->magic;

    if (!(GFD_HEADER_MAGIC == GFD_SWAP_8_IN_32(magic)))
    {
        ASSERT(!"Unknown GFD format.");
        return false;
    }

    if (!(GFD_HEADER_MAJOR == pHeader->majorVersion))
    {
        ASSERT(!"Unsupported GFD version.");
        return false;
    }

    return true;
}

/// \brief Checks Block Header magic and version values.
///
/// \param pBlockHeader              Header pointer to check
/// \retval Returned true or false
///
GFD_DECLSPEC inline bool GFD_API GFDCheckBlockHeaderMagicVersions(const GFDBlockHeader *pBlockHeader)
{
    u32 magic = pBlockHeader->magic;

    if (!(GFD_BLOCK_HEADER_MAGIC == GFD_SWAP_8_IN_32(magic)))
    {
        ASSERT(!"Unknown GFD block format.");
        return false;
    }

    if (!(GFD_BLOCK_HEADER_MAJOR == pBlockHeader->majorVersion))
    {
        ASSERT(!"Unsupported GFD block version.");
        return false;
    }

    return true;
}

/// @}

#ifdef __cplusplus
}
#endif // __cplusplus

#endif  // __GFDFILEIO_H__
